# 原生拖拽事件详解与 draggable 库分析

```html
<div id="div1" ondrop="drop(event)" ondragover="allowDrop(event)"></div>
<br />
<img
  id="drag1"
  src="img_logo.gif"
  draggable="true"
  ondragstart="drag(event)"
  width="336"
  height="69"
/>
```

```js
function allowDrop(ev) {
  ev.preventDefault();
}

function drag(ev) {
  ev.dataTransfer.setData("text", ev.target.id);
}

function drop(ev) {
  ev.preventDefault();
  var data = ev.dataTransfer.getData("text");
  ev.target.appendChild(document.getElementById(data));
}
```

```js
let draggable = document.getElementById("draggable1");

draggable.addEventListener("dragstart", dragStart, false);
draggable.addEventListener("dragend", dragEnd, false);

let droptarget = document.getElementById("droptarget1");

droptarget.addEventListener("dragenter", dragEnter, false);
droptarget.addEventListener("dragover", dragOver, false);
droptarget.addEventListener("dragleave", dragLeave, false);
droptarget.addEventListener("drop", drop, false);

/* Draggable event handlers */
function dragStart(event) {
  event.dataTransfer.setData("text/html", "You dragged the image!");

  // 添加拖拽的可视化反馈效果
  event.target.style.border = "1px solid #cccccc";
}

function dragEnd(event) {
  event.target.style.border = "none";
}

/* Drop target event handlers */
function dragEnter(event) {}

function dragOver(event) {
  event.preventDefault();
  return false;
}

function dragLeave(event) {}

function drop(event) {
  var data = event.dataTransfer.getData("text/html");
  event.preventDefault();
  return false;
}
```

## DataTransfer 对象

The event listener methods for all the drag and drop events accept Event object which has a readonly attribute called dataTransfer. The event.dataTransfer returns DataTransfer object associated with the event as follows −

```js
function EnterHandler(event) {
   DataTransfer dt = event.dataTransfer;
   .............
}
```

The DataTransfer object holds data about the drag and drop operation. This data can be retrieved and set in terms of various attributes associated with DataTransfer object as explained below:

| Sr.No. | DataTransfer attributes and their description                                                                                                                                                                                                                             |
| ------ | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| 1      | **dataTransfer.dropEffect [ = value ]**Returns the kind of operation that is currently selected.This attribute can be set, to change the selected operation.The possible values are **none, copy, link,** and **move**.                                                   |
| 2      | **dataTransfer.effectAllowed [ = value ]**Returns the kinds of operations that are to be allowed.This attribute can be set, to change the allowed operations.The possible values are **none, copy, copyLink, copyMove, link, linkMove, move, all** and **uninitialized**. |
| 3      | **dataTransfer.types**Returns a DOMStringList listing the formats that were set in the dragstart event. In addition, if any files are being dragged, then one of the types will be the string "Files".                                                                    |
| 4      | **dataTransfer.clearData( [ format ] )**Removes the data of the specified formats. Removes all data if the argument is omitted.                                                                                                                                           |
| 5      | **dataTransfer.setData(format, data)**Adds the specified data.                                                                                                                                                                                                            |
| 6      | **data = dataTransfer.getData(format)**Returns the specified data. If there is no such data, returns the empty string.                                                                                                                                                    |
| 7      | **dataTransfer.files**Returns a FileList of the files being dragged, if any.                                                                                                                                                                                              |
| 8      | **dataTransfer.setDragImage(element, x, y)**Uses the given element to update the drag feedback, replacing any previously specified feedback.                                                                                                                              |
| 9      | **dataTransfer.addElement(element)**Adds the given element to the list of elements used to render the drag feedback.                                                                                                                                                      |

The effectsAllowed property is used by the browsers to change the mouse cursor to show what kind of action is performed when dragging and dropping an element. Typically the mouse cursor changes when the dragged element is over a drop target.

The dropEffect is supposed to show the user (via the cursor) what happens when the mouse hovers over a drop target, but at the time of writing

The setDragImage(image, x, y) function can be used to set the image shown by the browser when the user drags an element. By default the browser shows a semi-transparent copy of the original element, but if you want a different image, you can set a different image using this function. The x and y properties can be used to set location offsets for the image when displayed. By default the upper left corner of the drag image is located at the tip of the mouse pointer. By setting different x and y properties you can change this. You can use either positive or negative x and y offsets.

```js
function dragStart(event) {
  event.dataTransfer.effectAllowed = "all";
  event.dataTransfer.dropEffect = "copy";

  var dragImage = document.createElement("img");
  dragImage.src = dragImageUrl;
  dragImage.width = 75;
  event.dataTransfer.setDragImage(dragImage, 0, 0);

  event.dataTransfer.setData("text/html", "You dragged the image!");
  event.target.style.border = "1px solid #cccccc";
}
```

## 文件拖拽

It is possible to drag files into the browser from the file system, and read the name and content of the dragged files from JavaScript. You do so via the HTML5 File API. Here is a drop target listener which detects the file name of the dragged file:

```js
var droptarget2 = document.getElementById("droptarget2");
droptarget2.addEventListener("drop", drop, false);

function drop(event) {
  // Files - array of dragged files.
  var files = event.dataTransfer.files;

  for (var i = 0; i < files.length; i++) {
    var file = files[i];
    console.log("file: " + file.name);
  }

  event.preventDefault();
  return false;
}
```

Notice how the drop() function does not call the getData() function, but instead accesses the files property of the DataTransfer. The files property contains a list of the files that were dragged into the browser. To learn how to read these files, consult the HTML5 File API (I will write about it soon).
