# 练习30：自动化测试

> 原文：[Exercise 30: Automated Testing](http://c.learncodethehardway.org/book/ex30.html)

> 译者：[飞龙](https://github.com/wizardforcel)

自动化测试经常用于例如Python和Ruby的其它语言，但是很少用于C。一部分原因是自动化加载和测试C的代码片段具有较高的难度。这一章中，我们会创建一个非常小型的测试“框架”，并且使用你的框架目录构建测试用例的示例。

我接下来打算使用，并且你会包含进框架目录的框架，叫做“minunit”，它以[Jera Design](http://www.jera.com/techinfo/jtns/jtn002.html)所编写的一小段代码作为开始，之后我扩展了它，就像这样：

```c
#undef NDEBUG
#ifndef _minunit_h
#define _minunit_h

#include <stdio.h>
#include <dbg.h>
#include <stdlib.h>

#define mu_suite_start() char *message = NULL

#define mu_assert(test, message) if (!(test)) { log_err(message); return message; }
#define mu_run_test(test) debug("\n-----%s", " " #test); \
    message = test(); tests_run++; if (message) return message;

#define RUN_TESTS(name) int main(int argc, char *argv[]) {\
    argc = 1; \
    debug("----- RUNNING: %s", argv[0]);\
        printf("----\nRUNNING: %s\n", argv[0]);\
        char *result = name();\
        if (result != 0) {\
            printf("FAILED: %s\n", result);\
        }\
        else {\
            printf("ALL TESTS PASSED\n");\
        }\
    printf("Tests run: %d\n", tests_run);\
        exit(result != 0);\
}


int tests_run;

#endif
```

原始的内容所剩不多了，现在我使用`dbg.h`宏，并且在模板测试运行器的末尾创建了大量的宏。在这小段代码中我们创建了整套函数单元测试系统，一旦它结合上shell脚本来运行测试，你可以将其用于你的C代码。

## 完成测试框架

为了基础这个练习，你应该让你的`src/libex29.c`正常工作，并且完成练习29的附加题，是`ex29.c`加载程序并合理运行。练习29中我这事了一个附加题来使它像单元测试一样工作，但是现在我打算重新想你展示如何使用`minunit.h`来做这件事。

首先我们需要创建一个简单的空单元测试，命名为`tests/libex29_tests.c`，在里面输入：

```c
#include "minunit.h"

char *test_dlopen()
{

    return NULL;
}

char *test_functions()
{

    return NULL;
}

char *test_failures()
{

    return NULL;
}

char *test_dlclose()
{

    return NULL;
}

char *all_tests() {
    mu_suite_start();

    mu_run_test(test_dlopen);
    mu_run_test(test_functions);
    mu_run_test(test_failures);
    mu_run_test(test_dlclose);

    return NULL;
}

RUN_TESTS(all_tests);
```

这份代码展示了`tests/minunit.h`中的`RUN_TESTS`宏，以及如何使用其他的测试运行器宏。我没有编写实际的测试函数，所以你只能看到单元测试的结构。我首先会分解这个文件：

libex29_tests.c:1

包含`minunit.h`框架。

libex29_tests.c:3-7

第一个测试。测试函数具有固定的结构，它们不带任何参数并且返回`char *`，成功时为`NULL`。这非常重要，因为其他宏用于向测试运行器返回错误信息。

libex29_tests.c:9-25

与第一个测试相似的更多测试。

libex29_tests.c:27

控制其他测试的运行器函数。它和其它测试用例格式一致，但是使用额外的东西来配置。

libex29_tests.c:28

为`mu_suite_start`测试设置一些通用的东西。

libex29_tests.c:30

这就是使用`mu_run_test`返回结果的地方。

libex29_tests.c:35

在你运行所有测试之后，你应该返回`NULL`，就像普通的测试函数一样。

libex29_tests.c:38

最后需要使用`RUN_TESTS`宏来启动`main`函数，让它运行`all_tests`启动器。

这就是用于运行测试所有代码了，现在你需要尝试使它运行在项目框架中。下面是我的执行结果：

```sh
not printable
```

我首先执行`make clean`，之后我运行了构建，它将模板改造为`libYOUR_LIBRARY.a`和`libYOUR_LIBRARY.so`文件。要记住你需要在练习29的附加题中完成它。但如果你没有完成的话，下面是我所使用的`Makefile`的文件差异：

```diff
diff --git a/code/c-skeleton/Makefile b/code/c-skeleton/Makefile
index 135d538..21b92bf 100644
--- a/code/c-skeleton/Makefile
+++ b/code/c-skeleton/Makefile
@@ -9,9 +9,10 @@ TEST_SRC=$(wildcard tests/*_tests.c)
 TESTS=$(patsubst %.c,%,$(TEST_SRC))

 TARGET=build/libYOUR_LIBRARY.a
+SO_TARGET=$(patsubst %.a,%.so,$(TARGET))

 # The Target Build
-all: $(TARGET) tests
+all: $(TARGET) $(SO_TARGET) tests

 dev: CFLAGS=-g -Wall -Isrc -Wall -Wextra $(OPTFLAGS)
 dev: all
@@ -21,6 +22,9 @@ $(TARGET): build $(OBJECTS)
         ar rcs $@ $(OBJECTS)
         ranlib $@

+$(SO_TARGET): $(TARGET) $(OBJECTS)
+       $(CC) -shared -o $@ $(OBJECTS)
+
 build:
         @mkdir -p build
         @mkdir -p bin
```

完成这些改变后，你现在应该能够构建任何东西，并且你可以最后补完剩余的单元测试函数：

```c
#include "minunit.h"
#include <dlfcn.h>

typedef int (*lib_function)(const char *data);
char *lib_file = "build/libYOUR_LIBRARY.so";
void *lib = NULL;

int check_function(const char *func_to_run, const char *data, int expected)
{
    lib_function func = dlsym(lib, func_to_run);
    check(func != NULL, "Did not find %s function in the library %s: %s", func_to_run, lib_file, dlerror());

    int rc = func(data);
    check(rc == expected, "Function %s return %d for data: %s", func_to_run, rc, data);

    return 1;
error:
    return 0;
}

char *test_dlopen()
{
    lib = dlopen(lib_file, RTLD_NOW);
    mu_assert(lib != NULL, "Failed to open the library to test.");

    return NULL;
}

char *test_functions()
{
    mu_assert(check_function("print_a_message", "Hello", 0), "print_a_message failed.");
    mu_assert(check_function("uppercase", "Hello", 0), "uppercase failed.");
    mu_assert(check_function("lowercase", "Hello", 0), "lowercase failed.");

    return NULL;
}

char *test_failures()
{
    mu_assert(check_function("fail_on_purpose", "Hello", 1), "fail_on_purpose should fail.");

    return NULL;
}

char *test_dlclose()
{
    int rc = dlclose(lib);
    mu_assert(rc == 0, "Failed to close lib.");

    return NULL;
}

char *all_tests() {
    mu_suite_start();

    mu_run_test(test_dlopen);
    mu_run_test(test_functions);
    mu_run_test(test_failures);
    mu_run_test(test_dlclose);

    return NULL;
}

RUN_TESTS(all_tests);
```

我希望你可以弄清楚它都干了什么，因为这里没有什么新的东西，除了`check_function`函数。这是一个通用的模式，其中我需要重复执行一段代码，然后通过为之创建宏或函数来使它自动化。这里我打算运行`.so`中所加载的函数，所以我创建了一个小型函数来完成它。

## 附加题

+ 这段代码能起作用，但是可能有点乱。清理框架目录，是它包含所有这些文件，但是移除任何和练习29有关的代码。你应该能够复制这个目录并且无需很多编辑操作就能开始新的项目。
+ 研究`runtests.sh`，并且查询有关`bash`语法的资料，来弄懂它的作用。你能够编写这个脚本的C版本吗？
