# 练习29：库和链接

> 原文：[Exercise 29: Libraries And Linking](http://c.learncodethehardway.org/book/ex29.html)

> 译者：[飞龙](https://github.com/wizardforcel)

C语言编程的核心能力之一就是链接OS所提供的库。链接是一种为你的程序添加额外特性的方法，这些特性由其它人在系统中创建并打包。你已经使用了一些自动包含的标准库，但是我打算对库的不同类型和它们的作用做个解释。

首先，库在每个语言中都没有良好的设计。我不知道为什么，但是似乎语言的设计者都将链接视为不是特别重要的东西。它们通常令人混乱，难以使用，不能正确进行版本控制，并以不同的方式链接到各种地方。

C没有什么不同，但是C中的库和链接是Unix操作系统的组件，并且可执行的格式在很多年前就设计好了。学习C如何链接库有助于理解OS如何工作，以及它如何运行你的程序。

C中的库有两种基本类型：

静态

你可以使用`ar`和`ranlib`来构建它，就像上个练习中的`libYOUR_LIBRARY.a`那样（Windows下后缀为`.lib`）。这种库可以当做一系列`.o`对象文件和函数的容器，以及当你构建程序时，可以当做是一个大型的`.o`文件。

动态

它们通常以`.so`（Linux）或`.dll`（Windows）结尾。在OSX中，差不多有一百万种后缀，取决于版本和编写它的人。严格来讲，OSX中的`.dylib`，`.bundle`和`framework`这三个之间没什么不同。这些文件都被构建好并且放置到指定的地方。当你运行程序时，OS会动态加载这些文件并且“凭空”链接到你的程序中。

我倾向于对小型或中型项目使用静态的库，因为它们易于使用，并且工作在在更多操作系统上。我也喜欢将所有代码放入静态库中，之后链接它来执行单元测试，或者链接到所需的程序中。

动态库适用于大型系统，它的空间十分有限，或者其中大量程序都使用相同的功能。这种情况下不应该为每个程序的共同特性静态链接所有代码，而是应该将它放到动态库中，这样它仅仅会为所有程序加载一份。

在上一个练习中，我讲解了如何构建静态库（`.a`），我会在本书的剩余部分用到它。这个练习中我打算向你展示如何构建一个简单的`.so`库，并且如何使用Unix系统的`dlopen`动态加载它。我会手动执行它，以便你可以理解每件实际发生的事情。之后，附加题这部分会使用c项目框架来创建它。

## 动态加载动态库

我创建了两个源文件来完成它。一个用于构建`libex29.so`库，另一个是个叫做`ex29`的程序，它可以加载这个库并运行其中的程序：

```c
#include <stdio.h>
#include <ctype.h>
#include "dbg.h"


int print_a_message(const char *msg)
{
    printf("A STRING: %s\n", msg);

    return 0;
}


int uppercase(const char *msg)
{
    int i = 0;

    // BUG: \0 termination problems
    for(i = 0; msg[i] != '\0'; i++) {
        printf("%c", toupper(msg[i]));
    }

    printf("\n");

    return 0;
}

int lowercase(const char *msg)
{
    int i = 0;

    // BUG: \0 termination problems
    for(i = 0; msg[i] != '\0'; i++) {
        printf("%c", tolower(msg[i]));
    }

    printf("\n");

    return 0;
}

int fail_on_purpose(const char *msg)
{
    return 1;
}
```

这里面没什么神奇之处。其中故意留了一些bug，看你是否注意到了。你需要在随后修复它们。

我们将要使用`dlopen`，`dlsym`，和`dlclose`函数来处理上面的函数。

```c
#include <stdio.h>
#include "dbg.h"
#include <dlfcn.h>

typedef int (*lib_function)(const char *data);


int main(int argc, char *argv[])
{
    int rc = 0;
    check(argc == 4, "USAGE: ex29 libex29.so function data");

    char *lib_file = argv[1];
    char *func_to_run = argv[2];
    char *data = argv[3];

    void *lib = dlopen(lib_file, RTLD_NOW);
    check(lib != NULL, "Failed to open the library %s: %s", lib_file, dlerror());

    lib_function func = dlsym(lib, func_to_run);
    check(func != NULL, "Did not find %s function in the library %s: %s", func_to_run, lib_file, dlerror());

    rc = func(data);
    check(rc == 0, "Function %s return %d for data: %s", func_to_run, rc, data);

    rc = dlclose(lib);
    check(rc == 0, "Failed to close %s", lib_file);

    return 0;

error:
    return 1;
}
```

我现在会拆分这个程序，便于你理解这一小段代码其中的原理。

ex29.c:5

我随后会使用这个函数指针定义，来调用库中的函数。这没什么新东西，确保你理解了它的作用。

ex29.c:17

在为一个小型程序做必要的初始化后，我使用了`dlopen`函数来加载由`lib_file`表示的库。这个函数返回一个句柄，我们随后会用到它，就像来打开文件那样。

ex29.c:18

如果出现错误，我执行了通常的检查并退出，但是要注意最后我使用了`dlerror`来查明发生了什么错误。

ex29.c:20

我使用了`dlsym`来获取`lib`中的函数，通过它的字面名称`func_to_run`。这是最强大的部分，因为我动态获取了一个函数指针，基于我从命令行`argv`获得的字符串。

ex29.c:23

接着我调用`func`函数，获得返回值并进行检查。

ex29.c:26

最后，我像关闭文件那样关闭了库。通常你需要在程序的整个运行期间保证它们打开，所以关闭操作并不非常实用，我只是在这里演示它。

> 译者注：由于能够使用系统调用加载，动态库可以被多种语言的程序调用，而静态库只能被C及兼容C的程序调用。

## 你会看到什么

既然你已经知道这些文件做什么了，下面是我的shell会话，用于构建`libex29.so`和`ex29`并随后运行它。下面的代码中你可以学到如何手动构建：

```sh
# compile the lib file and make the .so
# you may need -fPIC here on some platforms. add that if you get an error
$ cc -c libex29.c -o libex29.o
$ cc -shared -o libex29.so libex29.o

# make the loader program
$ cc -Wall -g -DNDEBUG ex29.c -ldl -o ex29

# try it out with some things that work
$ ex29 ./libex29.so print_a_message "hello there"
-bash: ex29: command not found
$ ./ex29 ./libex29.so print_a_message "hello there"
A STRING: hello there
$ ./ex29 ./libex29.so uppercase "hello there"
HELLO THERE
$ ./ex29 ./libex29.so lowercase "HELLO tHeRe"
hello there
$ ./ex29 ./libex29.so fail_on_purpose "i fail"
[ERROR] (ex29.c:23: errno: None) Function fail_on_purpose return 1 for data: i fail

# try to give it bad args
$ ./ex29 ./libex29.so fail_on_purpose
[ERROR] (ex29.c:11: errno: None) USAGE: ex29 libex29.so function data

# try calling a function that is not there
$ ./ex29 ./libex29.so adfasfasdf asdfadff
[ERROR] (ex29.c:20: errno: None) Did not find adfasfasdf
  function in the library libex29.so: dlsym(0x1076009b0, adfasfasdf): symbol not found

# try loading a .so that is not there
$ ./ex29 ./libex.so adfasfasdf asdfadfas
[ERROR] (ex29.c:17: errno: No such file or directory) Failed to open
    the library libex.so: dlopen(libex.so, 2): image not found
$
```

需要注意，你可能需要在不同OS、不同OS的不同版本，以及不同OS的不同版本的不同编译器上执行构建，则需要修改构建共享库的方式。如果我构建`libex29.so`的方式在你的平台上不起作用，请告诉我，我会为其它平台添加一些注解。

> 译者注：到处编写、到处调试、到处编译、到处发布。--vczh

&zwj;

> 注

> 有时候你像往常一样运行`cc -Wall -g -DNDEBUG -ldl ex29.c -o ex29`，并且认为它能够正常工作，但是没有。在一些平台上，参数的顺序会影响到它是否生效，这也没什么理由。在Debian或者Ubuntu中你需要执行`cc -Wall -g -DNDEBUG ex29.c -ldl -o ex29`，这是唯一的方式。所以虽然我在这里使用了OSX，但是以后如果你链接动态库的时候它找不到某个函数，要试着自己解决问题。

> 这里面比较麻烦的事情是，实际平台的不同会影响到命令参数的顺序。将`-ldl`放到某个位置没有理由与其它位置不同。它只是一个选项，还需要了解这些简直是太气人了。

## 如何使它崩溃

打开`lbex29.so`，并且使用能够处理二进制的编辑器编辑它。修改一些字节，然后关闭。看看你是否能使用`dlopen`函数来打开它，即使你修改了它。

## 附加题

+ 你注意到我在`libex29.c`中写的不良代码了吗？我使用了一个`for`循环来检查`'\0'`的结尾，修改它们使这些函数总是接收字符串长度，并在函数内部使用。
+ 使用项目框架目录，来为这个练习创建新的项目。将`libex29.c`放入`src/`目录，修改`Makefile`使它能够构建`build/libex29.so`。
+ 将`ex29.c`改为`tests/ex29_tests.c`，使它做为单元测试执行。使它能够正常工作，意思是你需要修改它让它加载`build/libex29.so`文件，并且运行上面我手写的测试。
+ 阅读`man dlopen`文档，并且查询所有有关函数。尝试`dlopen`的其它选项，比如`RTLD_NOW`。
