# 练习26：编写第一个真正的程序

> 原文：[Exercise 26: Write A First Real Program](http://c.learncodethehardway.org/book/ex26.html)

> 译者：[飞龙](https://github.com/wizardforcel)

这本书你已经完成一半了，所以你需要做一个期中检测。期中检测中你需要重新构建一个我特地为本书编写的软件，叫做`devpkg`。随后你需要以一些方式扩展它，并且通过编写一些单元测试来改进代码。

> 注

> 我在一些你需要完成的练习之前编写了这个练习。如果你现在尝试这个练习，记住软件可能会含有一些bug，你可能由于我的错误会产生一些问题，也可能不知道需要什么来完成它。如果这样的话，通过[help@learncodethehardway.org](mailto:help@learncodethehardway.org)来告诉我，之后等待我写完其它练习。

## 什么是`devpkg`？

`devpkg`是一个简单的C程序，可以用于安装其它软件。我特地为本书编写了它，作为一种方式来教你真正的软件是如何构建的，以及如何复用他人的库。它使用了一个叫做[Apache可移植运行时（APR）](http://apr.apache.org/)的库，其中含有许多工作跨平台的便利的C函数，包括Windows。此外，它只是从互联网（或本地文件）抓取代码，并且执行通常的`./configure ; make ; make install`命令，每个程序员都用到过。

这个练习中，你的目标是从源码构建`devpkg`，完成我提供的每个挑战，并且使用源码来理解`devpkg`做了什么和为什么这样做。

## 我们打算创建什么

我们打算创建一个具有三个命令的工具：

devpkg -S

在电脑上安装新的软件。

devpkg -I

从URL安装软件。

devpkg -L

列出安装的所有软件。

devpkg -F

为手动构建抓取源代码。

devpkg -B

构建所抓取的源码代码并且安装它，即使它已经安装了。

我们想让`devpkg`能够接受几乎任何URL，判断项目的类型，下载，安装，以及注册已经安装的软件。我们也希望它能够处理一个简单的依赖列表，以便它能够安装项目所需的所有软件。

## 设计

为了完成这一目标，`devpkg`具有非常简单的设计：

使用外部命令

大多数工作都是通过类似于`curl`、`git`和`tar`的外部命令完成的。这样减少了`devpkg`所需的代码量。

简单的文件数据库

你可以轻易使它变得很复杂，但是一开始你需要完成一个简单的文件数据库，位于`/usr/local/.devpkg/db`，来跟踪已安装的软件。

`/usr/local`

同样你可以使它更高级，但是对于初学者来说，假设项目始终位于`/usr/local`中，它是大多数Unix软件的标准安装目录。

`configure; make; make install`

假设大多数软件可以通过`configure; make; make install`来安装，也许`configure`是可选的。如果软件不能通过这种方式安装，要么提供某种方式来修改命令，要么`devpkg`就可以无视它。

用户可以root

我们假设用于可以使用`sudo`来提升至root权限，除非他们直到最后才想root。

这会使我们的程序像当初设想的一样简单，并且对于它的功能来说已经足够了。之后你可以进一步修改它。

## Apache 可移植运行时

你需要做的另外一件事情就是使用[Apache可移植运行时（APR）](http://apr.apache.org/)来未完成这个练习获得一个可移植的工具集。APR并不是必要的，你也可以不用它，但是你需要写的代码就会非常多。我现在强制你使用APR，使你能够熟悉链接和使用其他的库。最后，APR也能在Windows上工作，所以你可以把它迁移到许多其它平台上。

你应该获取`apr-1.4.5`和`apr-util-1.3`的库，以及浏览在[apr.apache.org主站](http://apr.apache.org/)上的文档。

下面是一个ShellScript，用于安装所需的所有库。你应该手动将它写到一个文件中，之后运行它直到APR安装好并且没有任何错误。

```sh
set -e

# go somewhere safe
cd /tmp

# get the source to base APR 1.4.6
curl -L -O http://archive.apache.org/dist/apr/apr-1.4.6.tar.gz

# extract it and go into the source
tar -xzvf apr-1.4.6.tar.gz
cd apr-1.4.6

# configure, make, make install
./configure
make
sudo make install

# reset and cleanup
cd /tmp
rm -rf apr-1.4.6 apr-1.4.6.tar.gz

# do the same with apr-util
curl -L -O http://archive.apache.org/dist/apr/apr-util-1.4.1.tar.gz

# extract
tar -xzvf apr-util-1.4.1.tar.gz
cd apr-util-1.4.1

# configure, make, make install
./configure --with-apr=/usr/local/apr
# you need that extra parameter to configure because
# apr-util can't really find it because...who knows.

make
sudo make install

#cleanup
cd /tmp
rm -rf apr-util-1.4.1* apr-1.4.6*
```

我希望你输入这个脚本，因为这就是`devpkg`基本上所做的事情，只是带有了一些选项和检查项。实际上，你可以使用Shell以更少的代码来完成它，但是这对于一本C语言的书不是一个很好的程序。

简单运行这个脚本，修复它直到正常工作，就完成的所有库的安装，之后你需要完成项目的剩下部分。

## 项目布局

你需要创建一些简单的项目文件来起步。下面是我通常创建一个新项目的方法：

```sh
mkdir devpkg
cd devpkg
touch README Makefile
```

## 其它依赖

你应该已经安装了APR和APR-util，所以你需要一些更多的文件作为基本的依赖：

+ 练习20中的`dbg.h`。
+ 从[http://bstring.sourceforge.net/](http://bstring.sourceforge.net/)下载的`bstrlib.h`和`bstrlib.c`。下载`.zip`文件，解压并且将这个两个文件拷贝到项目中。
+ 运行`make bstrlib.o`，如果这不能正常工作，阅读下面的“修复`bstring`”指南。

> 注

> 在一些平台上`bstring.c`文件会出现下列错误：

> ```sh
> bstrlib.c:2762: error: expected declaration specifiers or '...' before numeric constant
> ```

> 这是由于作者使用了一个不好的定义，它在一些平台上不能工作。你需要修改第2759行的`#ifdef __GNUC__`，并把它改成：

> ```c
> #if defined(__GNUC__) && !defined(__APPLE__)
> ```

之后在Mac OSX平台上就应该能够正常工作了。

做完上面这些后，你应该有了`Makefile`，`README`，`dbg.h`，`bstrlib.h`和`bstrlib.c`，并做好了准备。

## Makefile

我们最好从`Makefile`开始，因为它列出了项目如何构建，以及你会创建哪些源文件。

```make
PREFIX?=/usr/local
CFLAGS=-g -Wall -I${PREFIX}/apr/include/apr-1  -I${PREFIX}/apr/include/apr-util-1
LDFLAGS=-L${PREFIX}/apr/lib -lapr-1 -pthread -laprutil-1

all: devpkg

devpkg: bstrlib.o db.o shell.o commands.o

install: all
     install -d $(DESTDIR)/$(PREFIX)/bin/
     install devpkg $(DESTDIR)/$(PREFIX)/bin/

clean:
     rm -f *.o
     rm -f devpkg
     rm -rf *.dSYM
```

比起之前看到过的，这并没有什么新东西，除了可能有些奇怪的`?=`语法，它表示“如果之前没有定义，就将`PREFIX`设置为该值”。

> 注

> 如果你使用了最近版本的Ubuntu，你会得到`apr_off_t` 或 `off64_t`的错误，之后需要向`CFLAGS`添加`-D_LARGEFILE64_SOURCE=1`。

> 所需的另一件事是，你需要向`/etc/ld.conf.so.d/`添加`/usr/local/apr/lib`，之后运行`ldconfig`使它能够选择正常的库。

## 源文件

我们可以从`makefile`中看到，`devpkg`有四个依赖项，它们是：

`bstrlib.o`

由`bstrlib.c`和`bstrlib.o`产生，你已经将它们引入了。

`db.o`

由`db.c`和`db.h`产生，它包含了一个小型“数据库”程序集的代码。

`shell.o`

由`shell.c`和`shell.h`产生，包含一些函数，是类似`curl`的一些命令运行起来更容易。

`commands.o`

由`commands.c`和`commands.h`产生，包含了`devpkg`所需的所有命令并使它更易用。

`devpkg`

它不会显式提到，但是它是`Makefile`在这一部分的目标。它由`devpkg.c`产生，包含用于整个程序的`main`函数。

你的任务就是创建这些文件，并且输入代码并保证正确。

> 注

> 你读完这个描述可能会想，“Zed为什么那么聪明，坐着就能设计出来这些文件？！”我并不是用我强大的代码功力魔术般地把`devpkg`设计成这样。而是我做了这些：

> + 我编写了简单的`README`来获得如何构建项目的灵感。
> + 我创建了一个简单的bash脚本（就像你编写的那样）来理清所需的所有组件。
> + 我创建了一个`.c`文件，并且在它上面花了几天，酝酿并想出点子。
> + 接着我编写并调试程序，之后我将这一个大文件分成四个文件。
> + 做完这些之后，我重命名和优化了函数和数据结构，使它们在逻辑上更“美观”。
> + 最后，使新程序成功并以相同方式工作之后，我添加了一些新的特性，比如`-F`和`-B`选项。

> 你读到的这份列表是我打算教给你的，但不要认为这是我构建软件的通用方法。有时候我会事先知道主题，并且会做更多的规划。也有时我会编写一份规划并将它扔掉，之后再规划更好的版本。它完全取决于我的经验告诉我哪个比较好，或者我的灵感将我带到何处。

> 如果你碰到一个“专家”，它告诉你只有一个方法可以解决编程问题，那么它在骗你。要么它们实际使用了很多策略，要么他们并不足够好。

## DB函数

程序中必须有个方法来记录已经安装的URL，列出这些URL，并且检查一些程序是否已安装以便跳过。我会使用一个简单、扁平化的文件数据库，以及`bstrlib.h`。

首先，创建`db.h`头文件，以便让你知道需要实现什么。

```c
#ifndef _db_h
#define _db_h

#define DB_FILE "/usr/local/.devpkg/db"
#define DB_DIR "/usr/local/.devpkg"


int DB_init();
int DB_list();
int DB_update(const char *url);
int DB_find(const char *url);

#endif
```

之后实现`db.c`中的这些函数，在你编写它的时候，像之前一样使用`make`。

```c
#include <unistd.h>
#include <apr_errno.h>
#include <apr_file_io.h>

#include "db.h"
#include "bstrlib.h"
#include "dbg.h"

static FILE *DB_open(const char *path, const char *mode)
{
    return fopen(path, mode);
}


static void DB_close(FILE *db)
{
    fclose(db);
}


static bstring DB_load()
{
    FILE *db = NULL;
    bstring data = NULL;

    db = DB_open(DB_FILE, "r");
    check(db, "Failed to open database: %s", DB_FILE);

    data = bread((bNread)fread, db);
    check(data, "Failed to read from db file: %s", DB_FILE);

    DB_close(db);
    return data;

error:
    if(db) DB_close(db);
    if(data) bdestroy(data);
    return NULL;
}


int DB_update(const char *url)
{
    if(DB_find(url)) {
        log_info("Already recorded as installed: %s", url);
    }

    FILE *db = DB_open(DB_FILE, "a+");
    check(db, "Failed to open DB file: %s", DB_FILE);

    bstring line = bfromcstr(url);
    bconchar(line, '\n');
    int rc = fwrite(line->data, blength(line), 1, db);
    check(rc == 1, "Failed to append to the db.");

    return 0;
error:
    if(db) DB_close(db);
    return -1;
}


int DB_find(const char *url)
{
    bstring data = NULL;
    bstring line = bfromcstr(url);
    int res = -1;

    data = DB_load();
    check(data, "Failed to load: %s", DB_FILE);

    if(binstr(data, 0, line) == BSTR_ERR) {
        res = 0;
    } else {
        res = 1;
    }

error: // fallthrough
    if(data) bdestroy(data);
    if(line) bdestroy(line);

    return res;
}


int DB_init()
{
    apr_pool_t *p = NULL;
    apr_pool_initialize();
    apr_pool_create(&p, NULL);

    if(access(DB_DIR, W_OK | X_OK) == -1) {
        apr_status_t rc = apr_dir_make_recursive(DB_DIR,
                APR_UREAD | APR_UWRITE | APR_UEXECUTE |
                APR_GREAD | APR_GWRITE | APR_GEXECUTE, p);
        check(rc == APR_SUCCESS, "Failed to make database dir: %s", DB_DIR);
    }

    if(access(DB_FILE, W_OK) == -1) {
        FILE *db = DB_open(DB_FILE, "w");
        check(db, "Cannot open database: %s", DB_FILE);
        DB_close(db);
    }

    apr_pool_destroy(p);
    return 0;

error:
    apr_pool_destroy(p);
    return -1;
}


int DB_list()
{
    bstring data = DB_load();
    check(data, "Failed to read load: %s", DB_FILE);

    printf("%s", bdata(data));
    bdestroy(data);
    return 0;

error:
    return -1;
}
```

### 挑战1：代码复查

在继续之前，仔细阅读这些文件的每一行，并且确保你以准确地输入了它们。通过逐行阅读代码来实践它。同时，跟踪每个函数调用，并且确保你使用了`check`来校验返回值。最后，在APR网站上的文档，或者bstrlib.h 或 bstrlib.c的源码中，查阅每个你不认识的函数。

## Shell 函数

`devkpg`的一个关键设计是，使用类似于`curl`、`tar`和`git`的外部工具来完成大部分的工作。我们可以找到在程序内部完成这些工作的库，但是如果我们只是需要这些程序的基本功能，这样就毫无意义。在Unix运行其它命令并不丢人。

为了完成这些，我打算使用`apr_thread_proc.h`函数来运行程序，但是我也希望创建一个简单的类“模板”系统。我会使用`struct Shell`，它持有所有运行程序所需的信息，但是在参数中有一些“空位”，我可以将它们替换成实际值。

观察`shell.h`文件来了解我会用到的结构和命令。你可以看到我使用`extern`来表明其他的`.c`文件也能访问到`shell.c`中定义的变量。

```c
#ifndef _shell_h
#define _shell_h

#define MAX_COMMAND_ARGS 100

#include <apr_thread_proc.h>

typedef struct Shell {
    const char *dir;
    const char *exe;

    apr_procattr_t *attr;
    apr_proc_t proc;
    apr_exit_why_e exit_why;
    int exit_code;

    const char *args[MAX_COMMAND_ARGS];
} Shell;

int Shell_run(apr_pool_t *p, Shell *cmd);
int Shell_exec(Shell cmd, ...);

extern Shell CLEANUP_SH;
extern Shell GIT_SH;
extern Shell TAR_SH;
extern Shell CURL_SH;
extern Shell CONFIGURE_SH;
extern Shell MAKE_SH;
extern Shell INSTALL_SH;

#endif
```

确保你已经创建了`shell.h`，并且`extern Shell`变量的名字和数量相同。它们被`Shell_run`和`Shell_exec`函数用于运行命令。我定义了这两个函数，并且在`shell.c`中创建实际变量。

```c
#include "shell.h"
#include "dbg.h"
#include <stdarg.h>

int Shell_exec(Shell template, ...)
{
    apr_pool_t *p = NULL;
    int rc = -1;
    apr_status_t rv = APR_SUCCESS;
    va_list argp;
    const char *key = NULL;
    const char *arg = NULL;
    int i = 0;

    rv = apr_pool_create(&p, NULL);
    check(rv == APR_SUCCESS, "Failed to create pool.");

    va_start(argp, template);

    for(key = va_arg(argp, const char *);
        key != NULL;
        key = va_arg(argp, const char *))
    {
        arg = va_arg(argp, const char *);

        for(i = 0; template.args[i] != NULL; i++) {
            if(strcmp(template.args[i], key) == 0) {
                template.args[i] = arg;
                break; // found it
            }
        }
    }

    rc = Shell_run(p, &template);
    apr_pool_destroy(p);
    va_end(argp);
    return rc;

error:
    if(p) {
        apr_pool_destroy(p);
    }
    return rc;
}

int Shell_run(apr_pool_t *p, Shell *cmd)
{
    apr_procattr_t *attr;
    apr_status_t rv;
    apr_proc_t newproc;

    rv = apr_procattr_create(&attr, p);
    check(rv == APR_SUCCESS, "Failed to create proc attr.");

    rv = apr_procattr_io_set(attr, APR_NO_PIPE, APR_NO_PIPE,
            APR_NO_PIPE);
    check(rv == APR_SUCCESS, "Failed to set IO of command.");

    rv = apr_procattr_dir_set(attr, cmd->dir);
    check(rv == APR_SUCCESS, "Failed to set root to %s", cmd->dir);

    rv = apr_procattr_cmdtype_set(attr, APR_PROGRAM_PATH);
    check(rv == APR_SUCCESS, "Failed to set cmd type.");

    rv = apr_proc_create(&newproc, cmd->exe, cmd->args, NULL, attr, p);
    check(rv == APR_SUCCESS, "Failed to run command.");

    rv = apr_proc_wait(&newproc, &cmd->exit_code, &cmd->exit_why, APR_WAIT);
    check(rv == APR_CHILD_DONE, "Failed to wait.");

    check(cmd->exit_code == 0, "%s exited badly.", cmd->exe);
    check(cmd->exit_why == APR_PROC_EXIT, "%s was killed or crashed", cmd->exe);

    return 0;

error:
    return -1;
}

Shell CLEANUP_SH = {
    .exe = "rm",
    .dir = "/tmp",
    .args = {"rm", "-rf", "/tmp/pkg-build", "/tmp/pkg-src.tar.gz",
        "/tmp/pkg-src.tar.bz2", "/tmp/DEPENDS", NULL}
};

Shell GIT_SH = {
    .dir = "/tmp",
    .exe = "git",
    .args = {"git", "clone", "URL", "pkg-build", NULL}
};

Shell TAR_SH = {
    .dir = "/tmp/pkg-build",
    .exe = "tar",
    .args = {"tar", "-xzf", "FILE", "--strip-components", "1", NULL}
};

Shell CURL_SH = {
    .dir = "/tmp",
    .exe = "curl",
    .args = {"curl", "-L", "-o", "TARGET", "URL", NULL}
};

Shell CONFIGURE_SH = {
    .exe = "./configure",
    .dir = "/tmp/pkg-build",
    .args = {"configure", "OPTS", NULL},
};

Shell MAKE_SH = {
    .exe = "make",
    .dir = "/tmp/pkg-build",
    .args = {"make", "OPTS", NULL}
};

Shell INSTALL_SH = {
    .exe = "sudo",
    .dir = "/tmp/pkg-build",
    .args = {"sudo", "make", "TARGET", NULL}
};
```

自底向上阅读`shell.c`的代码（这也是常见的C源码布局），你会看到我创建了实际的`Shell`变量，它在`shell.h`中以`extern`修饰。它们虽然在这里，但是也被程序的其它部分使用。这就是创建全局变量的方式，它们可以存在于一个`.c`文件中，但是可在任何地方使用。你不应该创建很多这类变量，但是它们的确很方便。

继续阅读代码，我们读到了`Shell_run`，它是一个“基”函数，只是基于`Shell`中的东西执行命令。它使用了许多在`apr_thread_proc.h`中定义的函数，你需要查阅它们的每一个来了解工作原理。这就像是一些使用`system`函数调用的代码一样，但是它可以让你控制其他程序的执行。例如，在我们的`Shell`结构中，存在`.dir`属性在运行之前强制程序必须在指定目录中。

最后，我创建了`Shell_exec`函数，它是个变参函数。你在之前已经看到过了，但是确保你理解了`stdarg.h`函数以及如何编写它们。在下个挑战中你需要分析这一函数。

### 挑战2：分析`Shell_exec`

为这些文件（以及向挑战1那样的完整的代码复查）设置的挑战是完整分析`Shell_exec`，并且拆分代码来了解工作原理。你应该能够理解每一行代码，`for`循环如何工作，以及参数如何被替换。

一旦你分析完成，向`struct Shell`添加一个字段，提供需要替代的`args`变量的数量。更新所有命令来接受参数的正确数量，随后增加一个错误检查，来确认参数被正确替换，以及在错误时退出。

## 命令行函数

现在你需要构造正确的命令来完成功能。这些命令会用到APR的函数、`db.h`和`shell.h`来执行下载和构建软件的真正工作。这些文件最为复杂，所以要小心编写它们。你需要首先编写`commands.h`文件，接着在`commands.c`文件中实现它的函数。

```c
#ifndef _commands_h
#define _commands_h

#include <apr_pools.h>

#define DEPENDS_PATH "/tmp/DEPENDS"
#define TAR_GZ_SRC "/tmp/pkg-src.tar.gz"
#define TAR_BZ2_SRC "/tmp/pkg-src.tar.bz2"
#define BUILD_DIR "/tmp/pkg-build"
#define GIT_PAT "*.git"
#define DEPEND_PAT "*DEPENDS"
#define TAR_GZ_PAT "*.tar.gz"
#define TAR_BZ2_PAT "*.tar.bz2"
#define CONFIG_SCRIPT "/tmp/pkg-build/configure"

enum CommandType {
    COMMAND_NONE, COMMAND_INSTALL, COMMAND_LIST, COMMAND_FETCH,
    COMMAND_INIT, COMMAND_BUILD
};


int Command_fetch(apr_pool_t *p, const char *url, int fetch_only);

int Command_install(apr_pool_t *p, const char *url, const char *configure_opts,
        const char *make_opts, const char *install_opts);

int Command_depends(apr_pool_t *p, const char *path);

int Command_build(apr_pool_t *p, const char *url, const char *configure_opts,
        const char *make_opts, const char *install_opts);

#endif
```

`commands.h`中并没有很多之前没见过的东西。你应该看到了一些字符串的定义，它们在任何地方都会用到。真正的代码在`commands.c`中。

```c
#include <apr_uri.h>
#include <apr_fnmatch.h>
#include <unistd.h>

#include "commands.h"
#include "dbg.h"
#include "bstrlib.h"
#include "db.h"
#include "shell.h"


int Command_depends(apr_pool_t *p, const char *path)
{
    FILE *in = NULL;
    bstring line = NULL;

    in = fopen(path, "r");
    check(in != NULL, "Failed to open downloaded depends: %s", path);

    for(line = bgets((bNgetc)fgetc, in, '\n'); line != NULL;
            line = bgets((bNgetc)fgetc, in, '\n'))
    {
        btrimws(line);
        log_info("Processing depends: %s", bdata(line));
        int rc = Command_install(p, bdata(line), NULL, NULL, NULL);
        check(rc == 0, "Failed to install: %s", bdata(line));
        bdestroy(line);
    }

    fclose(in);
    return 0;

error:
    if(line) bdestroy(line);
    if(in) fclose(in);
    return -1;
}

int Command_fetch(apr_pool_t *p, const char *url, int fetch_only)
{
    apr_uri_t info = {.port = 0};
    int rc = 0;
    const char *depends_file = NULL;
    apr_status_t rv = apr_uri_parse(p, url, &info);

    check(rv == APR_SUCCESS, "Failed to parse URL: %s", url);

    if(apr_fnmatch(GIT_PAT, info.path, 0) == APR_SUCCESS) {
        rc = Shell_exec(GIT_SH, "URL", url, NULL);
        check(rc == 0, "git failed.");
    } else if(apr_fnmatch(DEPEND_PAT, info.path, 0) == APR_SUCCESS) {
        check(!fetch_only, "No point in fetching a DEPENDS file.");

        if(info.scheme) {
            depends_file = DEPENDS_PATH;
            rc = Shell_exec(CURL_SH, "URL", url, "TARGET", depends_file, NULL);
            check(rc == 0, "Curl failed.");
        } else {
            depends_file = info.path;
        }

        // recursively process the devpkg list
        log_info("Building according to DEPENDS: %s", url);
        rv = Command_depends(p, depends_file);
        check(rv == 0, "Failed to process the DEPENDS: %s", url);

        // this indicates that nothing needs to be done
        return 0;

    } else if(apr_fnmatch(TAR_GZ_PAT, info.path, 0) == APR_SUCCESS) {
        if(info.scheme) {
            rc = Shell_exec(CURL_SH,
                    "URL", url,
                    "TARGET", TAR_GZ_SRC, NULL);
            check(rc == 0, "Failed to curl source: %s", url);
        }

        rv = apr_dir_make_recursive(BUILD_DIR,
                APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
        check(rv == APR_SUCCESS, "Failed to make directory %s", BUILD_DIR);

        rc = Shell_exec(TAR_SH, "FILE", TAR_GZ_SRC, NULL);
        check(rc == 0, "Failed to untar %s", TAR_GZ_SRC);
    } else if(apr_fnmatch(TAR_BZ2_PAT, info.path, 0) == APR_SUCCESS) {
        if(info.scheme) {
            rc = Shell_exec(CURL_SH, "URL", url, "TARGET", TAR_BZ2_SRC, NULL);
            check(rc == 0, "Curl failed.");
        }

        apr_status_t rc = apr_dir_make_recursive(BUILD_DIR,
                APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);

        check(rc == 0, "Failed to make directory %s", BUILD_DIR);
        rc = Shell_exec(TAR_SH, "FILE", TAR_BZ2_SRC, NULL);
        check(rc == 0, "Failed to untar %s", TAR_BZ2_SRC);
    } else {
        sentinel("Don't now how to handle %s", url);
    }

    // indicates that an install needs to actually run
    return 1;
error:
    return -1;
}

int Command_build(apr_pool_t *p, const char *url, const char *configure_opts,
        const char *make_opts, const char *install_opts)
{
    int rc = 0;

    check(access(BUILD_DIR, X_OK | R_OK | W_OK) == 0,
            "Build directory doesn't exist: %s", BUILD_DIR);

    // actually do an install
    if(access(CONFIG_SCRIPT, X_OK) == 0) {
        log_info("Has a configure script, running it.");
        rc = Shell_exec(CONFIGURE_SH, "OPTS", configure_opts, NULL);
        check(rc == 0, "Failed to configure.");
    }

    rc = Shell_exec(MAKE_SH, "OPTS", make_opts, NULL);
    check(rc == 0, "Failed to build.");

    rc = Shell_exec(INSTALL_SH,
            "TARGET", install_opts ? install_opts : "install",
            NULL);
    check(rc == 0, "Failed to install.");

    rc = Shell_exec(CLEANUP_SH, NULL);
    check(rc == 0, "Failed to cleanup after build.");

    rc = DB_update(url);
    check(rc == 0, "Failed to add this package to the database.");

    return 0;

error:
    return -1;
}

int Command_install(apr_pool_t *p, const char *url, const char *configure_opts,
        const char *make_opts, const char *install_opts)
{
    int rc = 0;
    check(Shell_exec(CLEANUP_SH, NULL) == 0, "Failed to cleanup before building.");

    rc = DB_find(url);
    check(rc != -1, "Error checking the install database.");

    if(rc == 1) {
        log_info("Package %s already installed.", url);
        return 0;
    }

    rc = Command_fetch(p, url, 0);

    if(rc == 1) {
        rc = Command_build(p, url, configure_opts, make_opts, install_opts);
        check(rc == 0, "Failed to build: %s", url);
    } else if(rc == 0) {
        // no install needed
        log_info("Depends successfully installed: %s", url);
    } else {
        // had an error
        sentinel("Install failed: %s", url);
    }

    Shell_exec(CLEANUP_SH, NULL);
    return 0;

error:
    Shell_exec(CLEANUP_SH, NULL);
    return -1;
}
```

在你输入并编译它之后，就可以开始分析了。如果到目前为止你完成了前面的挑战，你会理解如何使用`shell.c`函数来运行shell命令，以及参数如何被替换。如果没有则需要回退到前面的挑战，确保你真正理解了`Shell_exec`的工作原理。

### 挑战3：评判我的设计

像之前一样，完整地复查一遍代码来保证一模一样。接着浏览每个函数并且确保你知道他如何工作。你也应该跟踪这个文件或其它文件中，每个函数对其它函数的调用。最后，确认你理解了这里的所有调用APR的函数。

一旦你正确编写并分析了这个文件，把我当成一个傻瓜一样来评判我的设计，我需要看看你是否可以改进它。不要真正修改代码，只是创建一个`notes.txt`并且写下你的想法和你需要修改的地方。

## `devpkg`的`main`函数

`devpkg.c`是最后且最重要的，但是也可能是最简单的文件，其中创建了`main`函数。没有与之配套的`.h`文件，因为这个文件包含其他所有文件。这个文件用于创建`devpkg`可执行程序，同时组装了来自`Makefile`的其它`.o`文件。在文件中输入代码并保证正确。

```c
#include <stdio.h>
#include <apr_general.h>
#include <apr_getopt.h>
#include <apr_strings.h>
#include <apr_lib.h>

#include "dbg.h"
#include "db.h"
#include "commands.h"

int main(int argc, const char const *argv[])
{
    apr_pool_t *p = NULL;
    apr_pool_initialize();
    apr_pool_create(&p, NULL);

    apr_getopt_t *opt;
    apr_status_t rv;

    char ch = '\0';
    const char *optarg = NULL;
    const char *config_opts = NULL;
    const char *install_opts = NULL;
    const char *make_opts = NULL;
    const char *url = NULL;
    enum CommandType request = COMMAND_NONE;


    rv = apr_getopt_init(&opt, p, argc, argv);

    while(apr_getopt(opt, "I:Lc:m:i:d:SF:B:", &ch, &optarg) == APR_SUCCESS) {
        switch (ch) {
            case 'I':
                request = COMMAND_INSTALL;
                url = optarg;
                break;

            case 'L':
                request = COMMAND_LIST;
                break;

            case 'c':
                config_opts = optarg;
                break;

            case 'm':
                make_opts = optarg;
                break;

            case 'i':
                install_opts = optarg;
                break;

            case 'S':
                request = COMMAND_INIT;
                break;

            case 'F':
                request = COMMAND_FETCH;
                url = optarg;
                break;

            case 'B':
                request = COMMAND_BUILD;
                url = optarg;
                break;
        }
    }

    switch(request) {
        case COMMAND_INSTALL:
            check(url, "You must at least give a URL.");
            Command_install(p, url, config_opts, make_opts, install_opts);
            break;

        case COMMAND_LIST:
            DB_list();
            break;

        case COMMAND_FETCH:
            check(url != NULL, "You must give a URL.");
            Command_fetch(p, url, 1);
            log_info("Downloaded to %s and in /tmp/", BUILD_DIR);
            break;

        case COMMAND_BUILD:
            check(url, "You must at least give a URL.");
            Command_build(p, url, config_opts, make_opts, install_opts);
            break;

        case COMMAND_INIT:
            rv = DB_init();
            check(rv == 0, "Failed to make the database.");
            break;

        default:
            sentinel("Invalid command given.");
    }


    return 0;

error:
    return 1;
}
```

### 挑战4：README 和测试文件

为这个文件设置的挑战是理解参数如何处理，以及参数是什么，之后创建含有使用指南的`README`文件。在编写`README`的同时，也编写一个简单的`simple.sh`，它运行`./devpkg`来检查每个命令都在实际环境下工作。在你的脚本顶端使用`set -e`，使它跳过第一个错误。

最后，在`Valgrind`下运行程序，确保在进行下一步之前，所有东西都能正常运行。

## 期中检测

最后的挑战就是这个期中检测，它包含三件事情：

+ 将你的代码与我的在线代码对比，以100%的分数开始，每错一行减去1%。
+ 在你的`notes.txt`中记录你是如何改进代码和`devpkg`的功能，并且实现你的改进。
+ 编写一个`devpkg`的替代版本，使用其他你喜欢的语言，或者你觉得最适合编写它的语言。对比二者，之后基于你的结果改进你的`devpkg`的C版本。

你可以执行下列命令来将你的代码与我的对比：

```sh
cd ..  # get one directory above your current one
git clone git://gitorious.org/devpkg/devpkg.git devpkgzed
diff -r devpkg devpkgzed
```

这将会克隆我的`devpkg`版本到`devpkgzed`目录中。之后使用工具`diff`来对比你的和我的代码。书中你所使用的这些文件直接来自于这个项目，所以如果出现了不同的行，肯定就有错误。

要记住这个练习没有真正的及格或不及格，它只是一个方式来让你挑战自己，并尽可能变得精确和谨慎。
