# 练习25：变参函数

> 原文：[Exercise 25: Variable Argument Functions](http://c.learncodethehardway.org/book/ex25.html)

> 译者：[飞龙](https://github.com/wizardforcel)

在C语言中，你可以通过创建“变参函数”来创建你自己的`printf`或者`scanf`版本。这些函数使用`stdarg.h`头，它们可以让你为你的库创建更加便利的接口。它们对于创建特定类型的“构建”函数、格式化函数和任何用到可变参数的函数都非常实用。

理解“变参函数”对于C语言编程并不必要，我在编程生涯中也只有大约20次用到它。但是，理解变参函数如何工作有助于你对它的调试，并且让你更加了解计算机。

```c
/** WARNING: This code is fresh and potentially isn't correct yet. */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include "dbg.h"

#define MAX_DATA 100

int read_string(char **out_string, int max_buffer)
{
    *out_string = calloc(1, max_buffer + 1);
    check_mem(*out_string);

    char *result = fgets(*out_string, max_buffer, stdin);
    check(result != NULL, "Input error.");

    return 0;

error:
    if(*out_string) free(*out_string);
    *out_string = NULL;
    return -1;
}

int read_int(int *out_int)
{
    char *input = NULL;
    int rc = read_string(&input, MAX_DATA);
    check(rc == 0, "Failed to read number.");

    *out_int = atoi(input);

    free(input);
    return 0;

error:
    if(input) free(input);
    return -1;
}

int read_scan(const char *fmt, ...)
{
    int i = 0;
    int rc = 0;
    int *out_int = NULL;
    char *out_char = NULL;
    char **out_string = NULL;
    int max_buffer = 0;

    va_list argp;
    va_start(argp, fmt);

    for(i = 0; fmt[i] != '\0'; i++) {
        if(fmt[i] == '%') {
            i++;
            switch(fmt[i]) {
                case '\0':
                    sentinel("Invalid format, you ended with %%.");
                    break;

                case 'd':
                    out_int = va_arg(argp, int *);
                    rc = read_int(out_int);
                    check(rc == 0, "Failed to read int.");
                    break;

                case 'c':
                    out_char = va_arg(argp, char *);
                    *out_char = fgetc(stdin);
                    break;

                case 's':
                    max_buffer = va_arg(argp, int);
                    out_string = va_arg(argp, char **);
                    rc = read_string(out_string, max_buffer);
                    check(rc == 0, "Failed to read string.");
                    break;

                default:
                    sentinel("Invalid format.");
            }
        } else {
            fgetc(stdin);
        }

        check(!feof(stdin) && !ferror(stdin), "Input error.");
    }

    va_end(argp);
    return 0;

error:
    va_end(argp);
    return -1;
}



int main(int argc, char *argv[])
{
    char *first_name = NULL;
    char initial = ' ';
    char *last_name = NULL;
    int age = 0;

    printf("What's your first name? ");
    int rc = read_scan("%s", MAX_DATA, &first_name);
    check(rc == 0, "Failed first name.");

    printf("What's your initial? ");
    rc = read_scan("%c\n", &initial);
    check(rc == 0, "Failed initial.");

    printf("What's your last name? ");
    rc = read_scan("%s", MAX_DATA, &last_name);
    check(rc == 0, "Failed last name.");

    printf("How old are you? ");
    rc = read_scan("%d", &age);

    printf("---- RESULTS ----\n");
    printf("First Name: %s", first_name);
    printf("Initial: '%c'\n", initial);
    printf("Last Name: %s", last_name);
    printf("Age: %d\n", age);

    free(first_name);
    free(last_name);
    return 0;
error:
    return -1;
}
```

这个程序和上一个练习很像，除了我编写了自己的`scanf`风格函数，它以我自己的方式处理字符串。你应该对`main`函数很清楚了，以及`read_string`和`read_int`两个函数，因为它们并没有做什么新的东西。

这里的变参函数叫做`read_scan`，它使用了`va_list`数据结构执行和`scanf`相同的工作，并支持宏和函数。下面是它的工作原理：

+ 我将函数的最后一个参数设置为`...`，它向C表示这个函数在`fmt`参数之后接受任何数量的参数。我可以在它前面设置许多其它的参数，但是在它后面不能放置任何参数。
+ 在设置完一些参数时，我创建了`va_list`类型的变量，并且使用`va_list`来为其初始化。这配置了`stdarg.h`中的这一可以处理可变参数的组件。
+ 接着我使用了`for`循环，遍历`fmt`格式化字符串，并且处理了类似`scanf`的格式，但比它略简单。它里面只带有整数、字符和字符串。
+ 当我碰到占位符时，我使用了`switch`语句来确定需要做什么。
+ 现在，为了从`va_list argp`中获得遍历，我需要使用`va_arg(argp, TYPE)`宏，其中`TYPE`是我将要向参数传递的准确类型。这一设计的后果是你会非常盲目，所以如果你没有足够的变量传入，程序就会崩溃。
+ 和`scanf`的有趣的不同点是，当它碰到`'s'`占位符时，我使用`read_string`来创建字符串。`va_list argp`栈需要接受两个函数：需要读取的最大尺寸，以及用于输出的字符串指针。`read_string`使用这些信息来执行实际工作。
+ 这使`read_scan`比`scan`更加一致，因为你总是使用`&`提供变量的地址，并且合理地设置它们。
+ 最后，如果它碰到了不在格式中的字符，它仅仅会读取并跳过，而并不关心字符是什么，因为它只需要跳过。

## 你会看到什么

当你运行程序时，会得到与下面详细的结果：

```sh
$ make ex25
cc -Wall -g -DNDEBUG    ex25.c   -o ex25
$ ./ex25
What's your first name? Zed
What's your initial? A
What's your last name? Shaw
How old are you? 37
---- RESULTS ----
First Name: Zed
Initial: 'A'
Last Name: Shaw
Age: 37
```

## 如何使它崩溃

这个程序对缓冲区溢出更加健壮，但是和`scanf`一样，它不能够处理输入的格式错误。为了使它崩溃，试着修改代码，把首先传入用于`'%s'`格式的尺寸去掉。同时试着传入多于`MAX_DATA`的数据，之后找到在`read_string`中不使用`calloc`的方法，并且修改它的工作方式。最后还有个问题是`fgets`会吃掉换行符，所以试着使用`fgetc`修复它，要注意字符串结尾应为`'\0'`。

## 附加题

+ 再三检查确保你明白了每个`out_`变量的作用。最重要的是`out_string`，并且它是指针的指针。所以，理清当你设置时获取到的是指针还是内容尤为重要。
+ 使用变参系统编写一个和`printf`相似的函数，重新编写`main`来使用它。
+ 像往常一样，阅读这些函数/宏的手册页，确保知道了它在你的平台做了什么，一些平台会使用宏而其它平台会使用函数，还有一些平台会让它们不起作用。这完全取决于你所用的编译器和平台。
