/*
 * Definitions that describe SMBIOS - https://www.dmtf.org/standards/smbios
 *
 * This file is part of System Informer.
 */

#ifndef _SMBIOS_H
#define _SMBIOS_H

typedef struct RAW_SMBIOS_DATA
{
    UCHAR Used20CallingMethod;
    UCHAR SMBIOSMajorVersion;
    UCHAR SMBIOSMinorVersion;
    UCHAR DmiRevision;
    ULONG Length;
    UCHAR SMBIOSTableData[ANYSIZE_ARRAY];
} RAW_SMBIOS_DATA, *PRAW_SMBIOS_DATA;

#include <pshpack1.h>

#define SMBIOS_INVALID_HANDLE  ((USHORT)0x0FFFF)
#define SMBIOS_INVALID_STRING  ((UCHAR)0)
#define SMBIOS_STRING_TABLE(x) (PVOID)((ULONG_PTR)x + x->Length)

#define SMBIOS_PROBE_STATUS_OTHER                     ((UCHAR)1)
#define SMBIOS_PROBE_STATUS_UNKNOWN                   ((UCHAR)2)
#define SMBIOS_PROBE_STATUS_OK                        ((UCHAR)3)
#define SMBIOS_PROBE_STATUS_NON_CRITICAL              ((UCHAR)4)
#define SMBIOS_PROBE_STATUS_CRITICAL                  ((UCHAR)5)
#define SMBIOS_PROBE_STATUS_NON_RECOVERABLE           ((UCHAR)6)

#define SMBIOS_MEMORY_ERROR_TYPE_OTHER                ((UCHAR)1)
#define SMBIOS_MEMORY_ERROR_TYPE_UNKNOWN              ((UCHAR)2)
#define SMBIOS_MEMORY_ERROR_TYPE_OK                   ((UCHAR)3)
#define SMBIOS_MEMORY_ERROR_TYPE_BAD_READ             ((UCHAR)4)
#define SMBIOS_MEMORY_ERROR_TYPE_PARITY               ((UCHAR)5)
#define SMBIOS_MEMORY_ERROR_TYPE_SINGLE_BIT           ((UCHAR)6)
#define SMBIOS_MEMORY_ERROR_TYPE_DOUBLE_BIT           ((UCHAR)7)
#define SMBIOS_MEMORY_ERROR_TYPE_MULTI_BIT            ((UCHAR)8)
#define SMBIOS_MEMORY_ERROR_TYPE_NIBBLE               ((UCHAR)9)
#define SMBIOS_MEMORY_ERROR_TYPE_CHECKSUM             ((UCHAR)10)
#define SMBIOS_MEMORY_ERROR_TYPE_CRC                  ((UCHAR)11)
#define SMBIOS_MEMORY_ERROR_TYPE_CORRECTED_SINGLE_BIT ((UCHAR)12)
#define SMBIOS_MEMORY_ERROR_TYPE_CORRECTED            ((UCHAR)13)
#define SMBIOS_MEMORY_ERROR_TYPE_UNCORRECTABLE        ((UCHAR)14)

#define SMBIOS_MEMORY_ERROR_GRANULARITY_OTHER         ((UCHAR)1)
#define SMBIOS_MEMORY_ERROR_GRANULARITY_UNKNOWN       ((UCHAR)2)
#define SMBIOS_MEMORY_ERROR_GRANULARITY_DEVICE        ((UCHAR)3)
#define SMBIOS_MEMORY_ERROR_GRANULARITY_PARTITION     ((UCHAR)4)

#define SMBIOS_MEMORY_ERROR_OPERATION_OTHER           ((UCHAR)1)
#define SMBIOS_MEMORY_ERROR_OPERATION_UNKNOWN         ((UCHAR)2)
#define SMBIOS_MEMORY_ERROR_OPERATION_READ            ((UCHAR)3)
#define SMBIOS_MEMORY_ERROR_OPERATION_WRITE           ((UCHAR)4)
#define SMBIOS_MEMORY_ERROR_OPERATION_PARTIAL_WRITE   ((UCHAR)5)

typedef struct _SMBIOS_ENTRY_POINT_2_1
{
    UCHAR Signature[4];             // _SM_ (ASCII)
    UCHAR Checksum;
    UCHAR Length;
    UCHAR MajorVersion;
    UCHAR MinorVersion;
    USHORT MaximumStructureSize;
    UCHAR EntryPointRevision;
    UCHAR Reserved[5];
    UCHAR Signature2[5];           // _DMI_ (ASCII)
    UCHAR IntermediateChecksum;
    USHORT StructureTableLength;
    ULONG StructureTableAddress;
    USHORT NumberStructures;
    UCHAR Revision;
} SMBIOS_ENTRY_POINT_2_1, *PSMBIOS_ENTRY_POINT_2_1;

typedef struct _SMBIOS_ENTRY_POINT_3_0
{
    UCHAR Signature[5];             // _SM3_ (ASCII)
    UCHAR Checksum;
    UCHAR Length;
    UCHAR MajorVersion;
    UCHAR MinorVersion;
    UCHAR DocumentRevision;
    UCHAR EntryPointRevision;
    UCHAR Reserved;
    ULONG MaximumStructureSize;
    ULONGLONG StructureTableAddress;
} SMBIOS_ENTRY_POINT_3_0, *PSMBIOS_ENTRY_POINT_3_0;

typedef struct _SMBIOS_HEADER
{
    UCHAR Type;
    UCHAR Length;
    USHORT Handle;
} SMBIOS_HEADER, *PSMBIOS_HEADER;

//
// Platform Firmware Information (Type 0)
//

#define SMBIOS_FIRMWARE_INFORMATION_TYPE ((UCHAR)0)

typedef struct _SMBIOS_FIRMWARE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR Vendor;                  // string
    UCHAR Version;                 // string
    USHORT StartingAddressSegment;
    UCHAR ReleaseDate;             // string
    UCHAR RomSize;
    ULONGLONG Characteristics;     // SMBIOS_FIRMWARE_FLAG_*
    // 2.4+
    USHORT Characteristics2;       // SMBIOS_FIRMWARE_FLAG_2_*
    UCHAR MajorRelease;
    UCHAR MinorRelease;
    UCHAR ControllerMajorRelease;
    UCHAR ControllerMinorRelease;
    // 3.1+
    union
    {
        struct
        {
            USHORT Unit : 2;       // SMBIOS_FIRMWARE_ROM_UNIT_*
            USHORT Size : 14;
        };

        USHORT Value;
    } RomSize2;
} SMBIOS_FIRMWARE_INFORMATION, *PSMBIOS_FIRMWARE_INFORMATION;

#define SMBIOS_FIRMWARE_FLAG_RESERVED_0                   0x0000000000000001UL
#define SMBIOS_FIRMWARE_FLAG_RESERVED_1                   0x0000000000000002UL
#define SMBIOS_FIRMWARE_FLAG_UNKNOWN                      0x0000000000000004UL
#define SMBIOS_FIRMWARE_FLAG_NOT_SUPPORTED                0x0000000000000008UL
#define SMBIOS_FIRMWARE_FLAG_ISA_SUPPORTED                0x0000000000000010UL
#define SMBIOS_FIRMWARE_FLAG_MCA_SUPPORTED                0x0000000000000020UL
#define SMBIOS_FIRMWARE_FLAG_EISA_SUPPORTED               0x0000000000000040UL
#define SMBIOS_FIRMWARE_FLAG_PCI_SUPPORTED                0x0000000000000080UL
#define SMBIOS_FIRMWARE_FLAG_PCMCIA_SUPPORTED             0x0000000000000100UL
#define SMBIOS_FIRMWARE_FLAG_PNP_SUPPORTED                0x0000000000000200UL
#define SMBIOS_FIRMWARE_FLAG_APM_SUPPORTED                0x0000000000000400UL
#define SMBIOS_FIRMWARE_FLAG_UPGRADE_SUPPORTED            0x0000000000000800UL
#define SMBIOS_FIRMWARE_FLAG_SHADOWING_SUPPORTED          0x0000000000001000UL
#define SMBIOS_FIRMWARE_FLAG_VL_VESA_SUPPORTED            0x0000000000002000UL
#define SMBIOS_FIRMWARE_FLAG_ESCD_SUPPORTED               0x0000000000004000UL
#define SMBIOS_FIRMWARE_FLAG_BOOT_FROM_CD_SUPPORTED       0x0000000000008000UL
#define SMBIOS_FIRMWARE_FLAG_SELECTABLE_BOOT_SUPPORTED    0x0000000000010000UL
#define SMBIOS_FIRMWARE_FLAG_ROM_SOCKETED                 0x0000000000020000UL
#define SMBIOS_FIRMWARE_FLAG_PCMCIA_BOOT_SUPPORTED        0x0000000000040000UL
#define SMBIOS_FIRMWARE_FLAG_EDD_SUPPORTED                0x0000000000080000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_NEC_9800_SUPPORTED    0x0000000000100000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_TOSHIBA_SUPPORTED     0x0000000000200000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_5_25_360KB_SUPPORTED  0x0000000000400000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_5_25_1_2_MB_SUPPORTED 0x0000000000800000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_3_5_720KB_SUPPORTED   0x0000000001000000UL
#define SMBIOS_FIRMWARE_FLAG_FLOPPY_3_5_2_88MB_SUPPORTED  0x0000000002000000UL
#define SMBIOS_FIRMWARE_FLAG_PRINT_SCREEN_SUPPORTED       0x0000000004000000UL
#define SMBIOS_FIRMWARE_FLAG_8042_KEYBOARD_SUPPORTED      0x0000000008000000UL
#define SMBIOS_FIRMWARE_FLAG_SERIAL_SUPPORTED             0x0000000010000000UL
#define SMBIOS_FIRMWARE_FLAG_PRINTER_SUPPORTED            0x0000000020000000UL
#define SMBIOS_FIRMWARE_FLAG_CGA_VIDEO_SUPPORTED          0x0000000040000000UL
#define SMBIOS_FIRMWARE_FLAG_NEC_PC_98                    0x0000000080000000UL
#define SMBIOS_FIRMWARE_FLAG_PLATFORM_RESERVED            0x0000FFFF00000000UL
#define SMBIOS_FIRMWARE_FLAG_SYSTEM_RESERVED              0xFFFF000000000000UL

#define SMBIOS_FIRMWARE_FLAG_2_ACPI_SUPPORTED                 ((USHORT)0x0001)
#define SMBIOS_FIRMWARE_FLAG_2_USB_LEGACY_SUPPORTED           ((USHORT)0x0002)
#define SMBIOS_FIRMWARE_FLAG_2_AGP_SUPPORTED                  ((USHORT)0x0004)
#define SMBIOS_FIRMWARE_FLAG_2_I20_BOOT_SUPPORTED             ((USHORT)0x0008)
#define SMBIOS_FIRMWARE_FLAG_2_LS_120_BOOT_SUPPORTED          ((USHORT)0x0010)
#define SMBIOS_FIRMWARE_FLAG_2_ZIP_BOOT_SUPPORTED             ((USHORT)0x0020)
#define SMBIOS_FIRMWARE_FLAG_2_1394_BOOT_SUPPORTED            ((USHORT)0x0040)
#define SMBIOS_FIRMWARE_FLAG_2_SMART_BATTERY_SUPPORTED        ((USHORT)0x0080)
#define SMBIOS_FIRMWARE_FLAG_2_BIOS_BOOT_SUPPORTED            ((USHORT)0x0100)
#define SMBIOS_FIRMWARE_FLAG_2_FN_KEY_NET_BOOT_SUPPORTED      ((USHORT)0x0200)
#define SMBIOS_FIRMWARE_FLAG_2_CONTENT_DISTRIBUTION_SUPPORTED ((USHORT)0x0400)
#define SMBIOS_FIRMWARE_FLAG_2_UEFI_SUPPORTED                 ((USHORT)0x0800)
#define SMBIOS_FIRMWARE_FLAG_2_MANUFACTURING_MODE_ENABLED     ((USHORT)0x1000)
#define SMBIOS_FIRMWARE_FLAG_2_RESERVED                       ((USHORT)0xE000)

#define SMBIOS_FIRMWARE_ROM_UNIT_MB         0
#define SMBIOS_FIRMWARE_ROM_UNIT_GB         1
#define SMBIOS_FIRMWARE_ROM_UNIT_RESERVED_1 2
#define SMBIOS_FIRMWARE_ROM_UNIT_RESERVED_2 3

//
// System Information (Type 1)
//

#define SMBIOS_SYSTEM_INFORMATION_TYPE ((UCHAR)1)

typedef struct _SMBIOS_SYSTEM_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR ProductName;  // string
    UCHAR Version;      // string
    UCHAR SerialNumber; // string
    // 2.1+
    GUID UniqueId;
    UCHAR WakeUpType;   // SMBIOS_SYSTEM_WAKE_UP_TYPE
    // 2.4+
    UCHAR SkuNumber;    // string
    UCHAR Family;       // string
} SMBIOS_SYSTEM_INFORMATION, *PSMBIOS_SYSTEM_INFORMATION;

#define SMBIOS_SYSTEM_WAKE_UP_TYPE_RESERVED     ((UCHAR)0)
#define SMBIOS_SYSTEM_WAKE_UP_TYPE_OTHER        ((UCHAR)1)
#define SMBIOS_SYSTEM_WAKE_UP_UNKNOWN           ((UCHAR)2)
#define SMBIOS_SYSTEM_WAKE_UP_APM_TIMER         ((UCHAR)3)
#define SMBIOS_SYSTEM_WAKE_UP_MODEM_RING        ((UCHAR)4)
#define SMBIOS_SYSTEM_WAKE_UP_LAN_REMOTE        ((UCHAR)5)
#define SMBIOS_SYSTEM_WAKE_UP_POWER_SWITCH      ((UCHAR)6)
#define SMBIOS_SYSTEM_WAKE_UP_PCI_PME           ((UCHAR)7)
#define SMBIOS_SYSTEM_WAKE_UP_AC_POWER_RESTORED ((UCHAR)8)

//
// Baseboard (or Module) Information (Type 2)
//

#define SMBIOS_BASEBOARD_INFORMATION_TYPE ((UCHAR)2)

typedef struct _SMBIOS_BASEBOARD_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR Manufacturer;            // string
    UCHAR Product;                 // string
    UCHAR Version;                 // string
    UCHAR SerialNumber;            // string
    UCHAR AssetTag;                // string
    UCHAR Features;                // SMBIOS_BASEBOARD_FEATURE_*
    UCHAR Location;                // string
    USHORT ChassisHandle;
    UCHAR BoardType;               // SMBIOS_BASEBOARD_TYPE_*
    UCHAR NumberOfHandles;
    USHORT Handles[ANYSIZE_ARRAY]; // Contained object handles (processors, ports, memory, etc.)
} SMBIOS_BASEBOARD_INFORMATION, *PSMBIOS_BASEBOARD_INFORMATION;

#define SMBIOS_BASEBOARD_FEATURE_HOSTING_BOARD           ((UCHAR)0x01)
#define SMBIOS_BASEBOARD_FEATURE_REQUIRES_DAUGHTER_BOARD ((UCHAR)0x02)
#define SMBIOS_BASEBOARD_FEATURE_REMOVABLE_BOARD         ((UCHAR)0x04)
#define SMBIOS_BASEBOARD_FEATURE_REPLACEABLE_BOARD       ((UCHAR)0x08)
#define SMBIOS_BASEBOARD_FEATURE_HOT_SWAP_BOARD          ((UCHAR)0x10)
#define SMBIOS_BASEBOARD_FEATURE_RESERVED                ((UCHAR)0xE0)

#define SMBIOS_BASEBOARD_TYPE_UNKNOWN                    ((UCHAR)1)
#define SMBIOS_BASEBOARD_TYPE_OTHER                      ((UCHAR)2)
#define SMBIOS_BASEBOARD_TYPE_SERVER_BLADE               ((UCHAR)3)
#define SMBIOS_BASEBOARD_TYPE_CONNECTIVITY_SWITCH        ((UCHAR)4)
#define SMBIOS_BASEBOARD_TYPE_SYSTEM_MANAGEMENT_MODULE   ((UCHAR)5)
#define SMBIOS_BASEBOARD_TYPE_PROCESSOR_MODULE           ((UCHAR)6)
#define SMBIOS_BASEBOARD_TYPE_IO_MODULE                  ((UCHAR)7)
#define SMBIOS_BASEBOARD_TYPE_MEMORY_MODULE              ((UCHAR)8)
#define SMBIOS_BASEBOARD_TYPE_DAUGHTER_BOARD             ((UCHAR)9)
#define SMBIOS_BASEBOARD_TYPE_MOTHERBOARD                ((UCHAR)10)
#define SMBIOS_BASEBOARD_TYPE_PROCESSOR_MEMORY_MODULE    ((UCHAR)11)
#define SMBIOS_BASEBOARD_TYPE_PROCESSOR_IO_MODULE        ((UCHAR)12)
#define SMBIOS_BASEBOARD_TYPE_INTERCONNECT               ((UCHAR)13)

//
// System Enclosure or Chassis (Type 3)
//

#define SMBIOS_CHASSIS_INFORMATION_TYPE ((UCHAR)3)

typedef struct _SMBIOS_CHASSIS_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    union
    {
        struct
        {
            UCHAR Type : 7;   // SMBIOS_CHASSIS_TYPE_*
            UCHAR Locked : 1;
        };

        UCHAR Value;
    } Chassis;

    UCHAR Version;            // string
    UCHAR SerialNumber;       // string
    UCHAR AssestTag;          // string
    // 2.1+
    UCHAR BootUpState;        // SMBIOS_CHASSIS_STATE_*
    UCHAR PowerSupplyState;   // SMBIOS_CHASSIS_STATE_*
    UCHAR ThermalState;       // SMBIOS_CHASSIS_STATE_*
    UCHAR SecurityState;      // SMBIOS_CHASSIS_SECURITY_STATE_*
    // 2.3+
    ULONG OEMDefined;
    UCHAR Height;
    UCHAR NumberOfPowerCords;
    UCHAR ElementCount;
    UCHAR ElementLength;
    UCHAR Elements[ANYSIZE_ARRAY]; // ElementCount * ElementLength, SMBIOS_CHASSIS_CONTAINED_ELEMENT
    // 2.7+
    // SMBIOS_CHASSIS_INFORMATION_EX
} SMBIOS_CHASSIS_INFORMATION, *PSMBIOS_CHASSIS_INFORMATION;

typedef struct _SMBIOS_CHASSIS_INFORMATION_EX
{
    // 2.7+
    UCHAR SkuNumber; // string
} SMBIOS_CHASSIS_INFORMATION_EX, *PSMBIOS_CHASSIS_INFORMATION_EX;

typedef struct _SMBIOS_CHASSIS_CONTAINED_ELEMENT
{
    // 2.3+
    union
    {
        struct
        {
            UCHAR Type : 7;   // SMBIOS_*_INFORMATION_TYPE or SM_BIOS_BASEBOARD_TYPE_*
            UCHAR Select : 1; // 1 = SMBIOS_*_INFORMATION_TYPE, 0 = SM_BIOS_BASEBOARD_TYPE_*
        };

        UCHAR Value;
    } Type;

    UCHAR ElementMinimum;
    UCHAR ElementMaximum;
} SMBIOS_CHASSIS_CONTAINED_ELEMENT, *PSMBIOS_CHASSIS_CONTAINED_ELEMENT;

#define SMBIOS_CHASSIS_TYPE_OTHER               1
#define SMBIOS_CHASSIS_TYPE_UNKNOWN             2
#define SMBIOS_CHASSIS_TYPE_DESKTOP             3
#define SMBIOS_CHASSIS_TYPE_LOW_PROFILE_DESKTOP 4
#define SMBIOS_CHASSIS_TYPE_PIZZA_BOX           5
#define SMBIOS_CHASSIS_TYPE_MINI_TOWER          6
#define SMBIOS_CHASSIS_TYPE_TOWER               7
#define SMBIOS_CHASSIS_TYPE_PORTABLE            8
#define SMBIOS_CHASSIS_TYPE_LAPTOP              9
#define SMBIOS_CHASSIS_TYPE_NOTEBOOK            10
#define SMBIOS_CHASSIS_TYPE_HAND_HELD           11
#define SMBIOS_CHASSIS_TYPE_DOCKING_STATION     12
#define SMBIOS_CHASSIS_TYPE_ALL_IN_ONE          13
#define SMBIOS_CHASSIS_TYPE_SUB_NOTEBOOK        14
#define SMBIOS_CHASSIS_TYPE_SPACE_SAVING        15
#define SMBIOS_CHASSIS_TYPE_LUNCH_BOX           16
#define SMBIOS_CHASSIS_TYPE_MAIN_SERVER         17
#define SMBIOS_CHASSIS_TYPE_EXPANSION           18
#define SMBIOS_CHASSIS_TYPE_SUB                 19
#define SMBIOS_CHASSIS_TYPE_BUS_EXPANSION       20
#define SMBIOS_CHASSIS_TYPE_PERIPHERAL          21
#define SMBIOS_CHASSIS_TYPE_RAID                22
#define SMBIOS_CHASSIS_TYPE_RACK_MOUNT          23
#define SMBIOS_CHASSIS_TYPE_SEALED_CASE_PC      24
#define SMBIOS_CHASSIS_TYPE_MULTI_SYSTEM        25
#define SMBIOS_CHASSIS_TYPE_COMPACT_PCI         26
#define SMBIOS_CHASSIS_TYPE_ADVANCED_TCA        27
#define SMBIOS_CHASSIS_TYPE_BLADE               28
#define SMBIOS_CHASSIS_TYPE_BLADE_ENCLOSURE     29
#define SMBIOS_CHASSIS_TYPE_TABLET              30
#define SMBIOS_CHASSIS_TYPE_CONVERTIBLE         31
#define SMBIOS_CHASSIS_TYPE_DETACHABLE          32
#define SMBIOS_CHASSIS_TYPE_IOT_GATEWAY         33
#define SMBIOS_CHASSIS_TYPE_EMBEDDED_PC         34
#define SMBIOS_CHASSIS_TYPE_MINI_PC             35
#define SMBIOS_CHASSIS_TYPE_STICK_PC            36

#define SMBIOS_CHASSIS_STATE_OTHER           ((UCHAR)1)
#define SMBIOS_CHASSIS_STATE_UNKNOWN         ((UCHAR)2)
#define SMBIOS_CHASSIS_STATE_SAFE            ((UCHAR)3)
#define SMBIOS_CHASSIS_STATE_WARNING         ((UCHAR)4)
#define SMBIOS_CHASSIS_STATE_CRITICAL        ((UCHAR)5)
#define SMBIOS_CHASSIS_STATE_NON_RECOVERABLE ((UCHAR)6)

#define SMBIOS_CHASSIS_SECURITY_STATE_OTHER      ((UCHAR)1)
#define SMBIOS_CHASSIS_SECURITY_STATE_UNKNOWN    ((UCHAR)2)
#define SMBIOS_CHASSIS_SECURITY_STATE_NONE       ((UCHAR)3)
#define SMBIOS_CHASSIS_SECURITY_STATE_LOCKED_OUT ((UCHAR)4)
#define SMBIOS_CHASSIS_SECURITY_STATE_ENABLED    ((UCHAR)5)

//
// Processor Information (Type 4)
//

#define SMBIOS_PROCESSOR_INFORMATION_TYPE ((UCHAR)4)

typedef struct _SMBIOS_PROCESSOR_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR Designation;     // string
    UCHAR Type;            // SMBIOS_PROCESSOR_TYPE_*
    UCHAR Family;          // SMBIOS_PROCESSOR_FAMILY_*
    UCHAR Manufacturer;    // string
    ULONGLONG Id;
    UCHAR Version;         // string

    struct
    {
        union
        {
            UCHAR Capable5000mV : 1; // 5V
            UCHAR Capable3500mV : 1; // 3.3V
            UCHAR Capable3900mV : 1; // 2.9V
            UCHAR Reserved : 4;
            UCHAR Mode : 1;          // 0 = Legacy Mode
        };

        UCHAR Value;
    } Voltage;

    USHORT ExternalClock;
    USHORT MaxSpeed;
    USHORT CurrentSpeed;

    struct
    {
        union
        {
            UCHAR Status : 3;    // SMBIOS_PROCESSOR_STATUS_*
            UCHAR Reserved : 3;
            UCHAR Populated : 1;
            UCHAR Reserved2 : 1;
        };

        UCHAR Value;
    } Status;

    UCHAR Upgrade;          // SMBIOS_PROCESSOR_UPGRADE_*
    // 2.1+
    UCHAR L1CacheHandle;
    UCHAR L2CacheHandle;
    UCHAR L3CacheHandle;
    // 2.3+
    UCHAR SerialNumber;     // string
    UCHAR AssetTag;         // string
    UCHAR PartNumber;       // string
    // 2.5+
    UCHAR CoreCount;
    UCHAR CoresEnabled;
    UCHAR ThreadCount;
    USHORT Characteristics; // SMBIOS_PROCESSOR_FLAG_*
    // 2.6+
    USHORT Family2;         // SMBIOS_PROCESSOR_FAMILY_*
    // 3.0+
    USHORT CoreCount2;
    USHORT CoresEnabled2;
    USHORT ThreadCount2;
    // 3.6+
    USHORT ThreadsEnabled;
    // 3.8+
    UCHAR SocketType;       // string
} SMBIOS_PROCESSOR_INFORMATION, *PSMBIOS_PROCESSOR_INFORMATION;

#define SMBIOS_PROCESSOR_TYPE_OTHER   ((UCHAR)1)
#define SMBIOS_PROCESSOR_TYPE_UNKNOWN ((UCHAR)2)
#define SMBIOS_PROCESSOR_TYPE_CENTRAL ((UCHAR)3)
#define SMBIOS_PROCESSOR_TYPE_MATH    ((UCHAR)4)
#define SMBIOS_PROCESSOR_TYPE_DSP     ((UCHAR)5)
#define SMBIOS_PROCESSOR_TYPE_VIDEO   ((UCHAR)6)

#define SMBIOS_PROCESSOR_FAMILY_OTHER 1
#define SMBIOS_PROCESSOR_FAMILY_UNKNOWN 2
#define SMBIOS_PROCESSOR_FAMILY_8086 3
#define SMBIOS_PROCESSOR_FAMILY_80286 4
#define SMBIOS_PROCESSOR_FAMILY_INTEL386_PROCESSOR 5
#define SMBIOS_PROCESSOR_FAMILY_INTEL486_PROCESSOR 6
#define SMBIOS_PROCESSOR_FAMILY_8087 7
#define SMBIOS_PROCESSOR_FAMILY_80287 8
#define SMBIOS_PROCESSOR_FAMILY_80387 9
#define SMBIOS_PROCESSOR_FAMILY_80487 10
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PENTIUM_PROCESSOR 11
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_PRO_PROCESSOR 12
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_II_PROCESSOR 13
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_PROCESSOR_WITH_MMX_TECHNOLOGY 14
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CELERON_PROCESSOR 15
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_II_XEON_PROCESSOR 16
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_III_PROCESSOR 17
#define SMBIOS_PROCESSOR_FAMILY_M1_FAMILY 18
#define SMBIOS_PROCESSOR_FAMILY_M2_FAMILY 19
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CELERON_M_PROCESSOR 20
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PENTIUM_4_HT_PROCESSOR 21
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PROCESSOR 22
// 23 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_AMD_DURON_PROCESSOR_FAMILY 24
#define SMBIOS_PROCESSOR_FAMILY_K5_FAMILY 25
#define SMBIOS_PROCESSOR_FAMILY_K6_FAMILY 26
#define SMBIOS_PROCESSOR_FAMILY_K6_2 27
#define SMBIOS_PROCESSOR_FAMILY_K6_3 28
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_PROCESSOR_FAMILY 29
#define SMBIOS_PROCESSOR_FAMILY_AMD29000_FAMILY 30
#define SMBIOS_PROCESSOR_FAMILY_K6_2_PLUS 31
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_FAMILY 32
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_601 33
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_603 34
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_603_PLUS 35
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_604 36
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_620 37
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_X704 38
#define SMBIOS_PROCESSOR_FAMILY_POWER_PC_750 39
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_DUO_PROCESSOR 40
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_DUO_MOBILE_PROCESSOR 41
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_SOLO_MOBILE_PROCESSOR 42
#define SMBIOS_PROCESSOR_FAMILY_INTEL_ATOM_PROCESSOR 43
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_M_PROCESSOR 44
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_M3_PROCESSOR 45
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_M5_PROCESSOR 46
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_M7_PROCESSOR 47
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_FAMILY 48
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21064 49
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21066 50
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21164 51
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21164PC 52
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21164A 53
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21264 54
#define SMBIOS_PROCESSOR_FAMILY_ALPHA_21364 55
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_II_ULTRA_DUAL_CORE_MOBILE_M_PROCESSOR_FAMILY 56
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_II_DUAL_CORE_MOBILE_M_PROCESSOR_FAMILY 57
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_II_DUAL_CORE_M_PROCESSOR_FAMILY 58
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_6100_SERIES_PROCESSOR 59
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_4100_SERIES_PROCESSOR 60
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_6200_SERIES_PROCESSOR 61
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_4200_SERIES_PROCESSOR 62
#define SMBIOS_PROCESSOR_FAMILY_AMD_FX_SERIES_PROCESSOR 63
#define SMBIOS_PROCESSOR_FAMILY_MIPS_FAMILY 64
#define SMBIOS_PROCESSOR_FAMILY_MIPS_R4000 65
#define SMBIOS_PROCESSOR_FAMILY_MIPS_R4200 66
#define SMBIOS_PROCESSOR_FAMILY_MIPS_R4400 67
#define SMBIOS_PROCESSOR_FAMILY_MIPS_R4600 68
#define SMBIOS_PROCESSOR_FAMILY_MIPS_R10000 69
#define SMBIOS_PROCESSOR_FAMILY_AMD_C_SERIES_PROCESSOR 70
#define SMBIOS_PROCESSOR_FAMILY_AMD_E_SERIES_PROCESSOR 71
#define SMBIOS_PROCESSOR_FAMILY_AMD_A_SERIES_PROCESSOR 72
#define SMBIOS_PROCESSOR_FAMILY_AMD_G_SERIES_PROCESSOR 73
#define SMBIOS_PROCESSOR_FAMILY_AMD_Z_SERIES_PROCESSOR 74
#define SMBIOS_PROCESSOR_FAMILY_AMD_R_SERIES_PROCESSOR 75
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_4300_SERIES_PROCESSOR 76
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_6300_SERIES_PROCESSOR 77
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_3300_SERIES_PROCESSOR 78
#define SMBIOS_PROCESSOR_FAMILY_AMD_FIREPRO_SERIES_PROCESSOR 79
#define SMBIOS_PROCESSOR_FAMILY_SPARC_FAMILY 80
#define SMBIOS_PROCESSOR_FAMILY_SUPERSPARC 81
#define SMBIOS_PROCESSOR_FAMILY_MICROSPARC_II 82
#define SMBIOS_PROCESSOR_FAMILY_MICROSPARC_IIep 83
#define SMBIOS_PROCESSOR_FAMILY_ULTRASPARC 84
#define SMBIOS_PROCESSOR_FAMILY_ULTRASPARC_II 85
#define SMBIOS_PROCESSOR_FAMILY_ULTRASPARC_Iii 86
#define SMBIOS_PROCESSOR_FAMILY_ULTRASPARC_III 87
#define SMBIOS_PROCESSOR_FAMILY_ULTRASPARC_IIIi 88
// 89-95 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_68040_FAMILY 96
#define SMBIOS_PROCESSOR_FAMILY_68XXX 97
#define SMBIOS_PROCESSOR_FAMILY_68000 98
#define SMBIOS_PROCESSOR_FAMILY_68010 99
#define SMBIOS_PROCESSOR_FAMILY_68020 100
#define SMBIOS_PROCESSOR_FAMILY_68030 101
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_X4_QUAD_CORE_PROCESSOR_FAMILY 102
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_X1000_SERIES_PROCESSOR 103
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_X2000_SERIES_APU 104
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_A_SERIES_PROCESSOR 105
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_X3000_SERIES_APU 106
#define SMBIOS_PROCESSOR_FAMILY_AMD_ZEN_PROCESSOR_FAMILY 107
// 108-111 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_HOBBIT_FAMILY 112
// 113-119 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_CRUSOE_TM5000_FAMILY 120
#define SMBIOS_PROCESSOR_FAMILY_CRUSOE_TM3000_FAMILY 121
#define SMBIOS_PROCESSOR_FAMILY_EFFICEON_TM8000_FAMILY 122
// 123-127 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_WEITEK 128
// 129 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_ITANIUM_PROCESSOR 130
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_64_PROCESSOR_FAMILY 131
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_PROCESSOR_FAMILY 132
#define SMBIOS_PROCESSOR_FAMILY_AMD_SEMPRON_PROCESSOR_FAMILY 133
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_64_MOBILE_TECHNOLOGY 134
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_AMD_OPTERON_PROCESSOR_FAMILY 135
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_64_X2_DUAL_CORE_PROCESSOR_FAMILY 136
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_64_X2_MOBILE_TECHNOLOGY 137
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_AMD_OPTERON_PROCESSOR_FAMILY 138
#define SMBIOS_PROCESSOR_FAMILY_THIRD_GENERATION_AMD_OPTERON_PROCESSOR_FAMILY 139
#define SMBIOS_PROCESSOR_FAMILY_AMD_PHENOM_FX_QUAD_CORE_PROCESSOR_FAMILY 140
#define SMBIOS_PROCESSOR_FAMILY_AMD_PHENOM_X4_QUAD_CORE_PROCESSOR_FAMILY 141
#define SMBIOS_PROCESSOR_FAMILY_AMD_PHENOM_X2_DUAL_CORE_PROCESSOR_FAMILY 142
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_X2_DUAL_CORE_PROCESSOR_FAMILY 143
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_FAMILY 144
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_8500 145
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_8000 146
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_7300LC 147
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_7200 148
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_7100LC 149
#define SMBIOS_PROCESSOR_FAMILY_PA_RISC_7100 150
// 151-159 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_V30_FAMILY 160
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_3200_SERIES 161
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_3000_SERIES 162
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_5300_SERIES 163
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_5100_SERIES 164
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_5000_SERIES 165
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_LV 166
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_ULV 167
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_7100_SERIES 168
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_5400_SERIES 169
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR 170
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_5200_SERIES 171
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_7200_SERIES 172
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_7300_SERIES 173
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_7400_SERIES 174
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_INTEL_XEON_PROCESSOR_7400_SERIES 175
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_III_XEON_PROCESSOR 176
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_III_PROCESSOR_WITH_INTEL_SPEEDSTEP_TECHNOLOGY 177
#define SMBIOS_PROCESSOR_FAMILY_PENTIUM_4_PROCESSOR 178
#define SMBIOS_PROCESSOR_FAMILY_INTEL_XEON_PROCESSOR 179
#define SMBIOS_PROCESSOR_FAMILY_AS400_FAMILY 180
#define SMBIOS_PROCESSOR_FAMILY_INTEL_XEON_PROCESSOR_MP 181
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_XP_PROCESSOR_FAMILY 182
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_MP_PROCESSOR_FAMILY 183
#define SMBIOS_PROCESSOR_FAMILY_INTEL_ITANIUM_2_PROCESSOR 184
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PENTIUM_M_PROCESSOR 185
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CELERON_D_PROCESSOR 186
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PENTIUM_D_PROCESSOR 187
#define SMBIOS_PROCESSOR_FAMILY_INTEL_PENTIUM_PROCESSOR_EXTREME_EDITION 188
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_SOLO_PROCESSOR 189
#define SMBIOS_PROCESSOR_FAMILY_RESERVED 190
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_DUO_PROCESSOR 191
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_SOLO_PROCESSOR 192
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_EXTREME_PROCESSOR 193
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_QUAD_PROCESSOR 194
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_EXTREME_MOBILE_PROCESSOR 195
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_DUO_MOBILE_PROCESSOR 196
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_2_SOLO_MOBILE_PROCESSOR 197
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_I7_PROCESSOR 198
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_CELERO_PROCESSOR 199
#define SMBIOS_PROCESSOR_FAMILY_IBM390_FAMILY 200
#define SMBIOS_PROCESSOR_FAMILY_G4 201
#define SMBIOS_PROCESSOR_FAMILY_G5 202
#define SMBIOS_PROCESSOR_FAMILY_ESA_390_G6 203
#define SMBIOS_PROCESSOR_FAMILY_Z_ARCHITECTURE_BASE 204
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_I5_PROCESSOR 205
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_I3_PROCESSOR 206
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_I9_PROCESSOR 207
#define SMBIOS_PROCESSOR_FAMILY_INTEL_XEON_D_PROCESSOR_FAMILY 208
// 209 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_VIA_C7_M_PROCESSOR_FAMILY 210
#define SMBIOS_PROCESSOR_FAMILY_VIA_C7_D_PROCESSOR_FAMILY 211
#define SMBIOS_PROCESSOR_FAMILY_VIA_C7_PROCESSOR_FAMILY 212
#define SMBIOS_PROCESSOR_FAMILY_VIA_EDEN_PROCESSOR_FAMILY 213
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_INTEL_XEON_PROCESSOR 214
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_3XXX_SERIES 215
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_3XXX_SERIES 216
#define SMBIOS_PROCESSOR_FAMILY_VIA_NANO_PROCESSOR_FAMILY 217
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_5XXX_SERIES 218
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_5XXX_SERIES 219
// 220 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_INTEL_XEON_PROCESSOR_7XXX_SERIES 221
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_INTEL_XEON_PROCESSOR_7XXX_SERIES 222
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_INTEL_XEON_PROCESSOR_7XXX_SERIES 223
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_INTEL_XEON_PROCESSOR_3400_SERIES 224
// 225-227 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_AMD_OPTERON_3000_SERIES_PROCESSOR 228
#define SMBIOS_PROCESSOR_FAMILY_AMD_SEMPRON_II_PROCESSOR 229
#define SMBIOS_PROCESSOR_FAMILY_EMBEDDED_AMD_OPTERON_QUAD_CORE_PROCESSOR_FAMILY 230
#define SMBIOS_PROCESSOR_FAMILY_AMD_PHENOM_TRIPLE_CORE_PROCESSOR_FAMILY 231
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_ULTRA_DUAL_CORE_MOBILE_PROCESSOR_FAMILY 232
#define SMBIOS_PROCESSOR_FAMILY_AMD_TURION_DUAL_CORE_MOBILE_PROCESSOR_FAMILY 233
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_DUAL_CORE_PROCESSOR_FAMILY 234
#define SMBIOS_PROCESSOR_FAMILY_AMD_SEMPRON_SI_PROCESSOR_FAMILY 235
#define SMBIOS_PROCESSOR_FAMILY_AMD_PHENOM_II_PROCESSOR_FAMILY 236
#define SMBIOS_PROCESSOR_FAMILY_AMD_ATHLON_II_PROCESSOR_FAMILY 237
#define SMBIOS_PROCESSOR_FAMILY_SIX_CORE_AMD_OPTERON_PROCESSOR_FAMILY 238
#define SMBIOS_PROCESSOR_FAMILY_AMD_SEMPRON_M_PROCESSOR_FAMILY 239
// 240-249 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_I860 250
#define SMBIOS_PROCESSOR_FAMILY_I960 251
// 252-253 - Available for assignment
#define SMBIOS_PROCESSOR_FAMILY_USE_2ND_FIELD 254
// 255 RESERVED
// 256-511 - Available for assignment, except for:
#define SMBIOS_PROCESSOR_FAMILY_ARMV7 256
#define SMBIOS_PROCESSOR_FAMILY_ARMV8 257
#define SMBIOS_PROCESSOR_FAMILY_ARMV9 258
// 259 - Reserved for future use by ARM
#define SMBIOS_PROCESSOR_FAMILY_SH_3 260
#define SMBIOS_PROCESSOR_FAMILY_SH_4 261
#define SMBIOS_PROCESSOR_FAMILY_ARM 280
#define SMBIOS_PROCESSOR_FAMILY_STRONGARM 281
#define SMBIOS_PROCESSOR_FAMILY_6X86 300
#define SMBIOS_PROCESSOR_FAMILY_MEDIAGX 301
#define SMBIOS_PROCESSOR_FAMILY_MII 302
#define SMBIOS_PROCESSOR_FAMILY_WINCHIP 320
#define SMBIOS_PROCESSOR_FAMILY_DSP 350
#define SMBIOS_PROCESSOR_FAMILY_VIDEO_PROCESSOR 500
// 512-767 - Available for assignment, except for:
#define SMBIOS_PROCESSOR_FAMILY_RISC_V_RV32 512
#define SMBIOS_PROCESSOR_FAMILY_RISC_V_RV64 513
#define SMBIOS_PROCESSOR_FAMILY_RISC_V_RV128 514
#define SMBIOS_PROCESSOR_FAMILY_LOONGARCH 600
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_1_PROCESSOR_FAMILY 601
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_2_PROCESSOR_FAMILY 602
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3_PROCESSOR_FAMILY 603
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_2K_PROCESSOR_FAMILY 604
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3A_PROCESSOR_FAMILY 605
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3B_PROCESSOR_FAMILY 606
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3C_PROCESSOR_FAMILY 607
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3D_PROCESSOR_FAMILY 608
#define SMBIOS_PROCESSOR_FAMILY_LOONGSON_3E_PROCESSOR_FAMILY 609
#define SMBIOS_PROCESSOR_FAMILY_DUAL_CORE_LOONGSON_2K_PROCESSOR_2XXX_SERIES 610
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_LOONGSON_3A_PROCESSOR_5XXX_SERIES 620
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_LOONGSON_3A_PROCESSOR_5XXX_SERIES 621
#define SMBIOS_PROCESSOR_FAMILY_QUAD_CORE_LOONGSON_3B_PROCESSOR_5XXX_SERIES 622
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_LOONGSON_3B_PROCESSOR_5XXX_SERIES 623
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_LOONGSON_3C_PROCESSOR_5XXX_SERIES 624
#define SMBIOS_PROCESSOR_FAMILY_MULTI_CORE_LOONGSON_3D_PROCESSOR_5XXX_SERIES 625
// 768-1023 - Available for assignment, except for:
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_3 768
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_5 769
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_7 770
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_9 771
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_ULTRA_3 772
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_ULTRA_5 773
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_ULTRA_7 774
#define SMBIOS_PROCESSOR_FAMILY_INTEL_CORE_ULTRA_9 775
// 1024-65533 - Available for assignment
// 65534–65535 - RESERVED

#define SMBIOS_PROCESSOR_STATUS_UNKNOWN              0
#define SMBIOS_PROCESSOR_STATUS_ENABLED              1
#define SMBIOS_PROCESSOR_STATUS_DISABLED_BY_USER     2
#define SMBIOS_PROCESSOR_STATUS_DISABLED_BY_FIRMWARE 3
#define SMBIOS_PROCESSOR_STATUS_IDLE                 4
#define SMBIOS_PROCESSOR_STATUS_RESERVED_5           5
#define SMBIOS_PROCESSOR_STATUS_RESERVED_6           6
#define SMBIOS_PROCESSOR_STATUS_OTHER                7

#define SMBIOS_PROCESSOR_UPGRADE_OTHER 1
#define SMBIOS_PROCESSOR_UPGRADE_UNKNOWN 2
#define SMBIOS_PROCESSOR_UPGRADE_DAUGHTER_BOARD 3
#define SMBIOS_PROCESSOR_UPGRADE_ZIF_SOCKET 4
#define SMBIOS_PROCESSOR_UPGRADE_REPLACEABLE_PIGGY_BACK 5
#define SMBIOS_PROCESSOR_UPGRADE_NONE 6
#define SMBIOS_PROCESSOR_UPGRADE_LIF_SOCKET 7
#define SMBIOS_PROCESSOR_UPGRADE_SLOT_1 8
#define SMBIOS_PROCESSOR_UPGRADE_SLOT_2 9
#define SMBIOS_PROCESSOR_UPGRADE_370_PIN_SOCKET 10
#define SMBIOS_PROCESSOR_UPGRADE_SLOT_A 11
#define SMBIOS_PROCESSOR_UPGRADE_SLOT_M 12
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_423 13
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_A 14 // (Socket 462)
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_478 15
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_754 16
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_940 17
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_939 18
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_MPGA604 19
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA771 20
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA775 21
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_S1 22
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_AM2 23
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_F 24 // (1207)
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1366 25
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_G34 26
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_AM3 27
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_C32 28
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1156 29
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1567 30
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_PGA988A 31
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1288 32
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_RPGA988B 33
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1023 34
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1224 35
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1155 36
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1356 37
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA2011 38
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_FS1 39
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_FS2 40
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_FM1 41
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_FM2 42
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA2011_3 43
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1356_3 44
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1150 45
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1168 46
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1234 47
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1364 48
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_AM4 49
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1151 50
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1356 51
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1440 52
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1515 53
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA3647_1 54
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_SP3 55
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_SP3R2 56
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA2066 57
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1392 58
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1510 59
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1528 60
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA4189 61
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1200 62
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA4677 63
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1700 64
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1744 65
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1781 66
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1211 67
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA2422 68
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1211 69
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA2422 70
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA5773 71
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA5773 72
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_AM5 73
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_SP5 74
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_SP6 75
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA883 76
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1190 77
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA4129 78
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA4710 79
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA7529 80
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1964 81
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA1792 82
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA2049 83
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA2551 84
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_LGA1851 85
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA2114 86
#define SMBIOS_PROCESSOR_UPGRADE_SOCKET_BGA2833 87
#define SMBIOS_PROCESSOR_UPGRADE_INVALID 255

#define SMBIOS_PROCESSOR_FLAG_RESERVED                  ((USHORT)0x0001)
#define SMBIOS_PROCESSOR_FLAG_UNKNOWN                   ((USHORT)0x0002)
#define SMBIOS_PROCESSOR_FLAG_64_BIT                    ((USHORT)0x0004)
#define SMBIOS_PROCESSOR_FLAG_MILT_CORE                 ((USHORT)0x0008)
#define SMBIOS_PROCESSOR_FLAG_HARDWARE_THREADED         ((USHORT)0x0010)
#define SMBIOS_PROCESSOR_FLAG_EXECUTE_PROTECTION        ((USHORT)0x0020)
#define SMBIOS_PROCESSOR_FLAG_ENHANCED_VIRTUALIZATION   ((USHORT)0x0040)
#define SMBIOS_PROCESSOR_FLAG_POWER_PERFORMANCE_CONTROL ((USHORT)0x0080)
#define SMBIOS_PROCESSOR_FLAG_128_BIT_CAPABLE           ((USHORT)0x0100)
#define SMBIOS_PROCESSOR_FLAG_ARM64_SOC                 ((USHORT)0x0200)
#define SMBIOS_PROCESSOR_FLAG_RESERVED_2                ((USHORT)0xFC00)

//
// Memory Controller Information (Type 5, Obsolete)
//

#define SMBIOS_MEMORY_CONTROLLER_INFORMATION_TYPE ((UCHAR)5)

typedef struct _SMBIOS_MEMORY_CONTROLLER_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR ErrorDetectionMethod;        // SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_*
    UCHAR ErrorCorrectionCapabilities; // SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_*
    UCHAR SupportedInterleave;         // SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_*
    UCHAR CurrentInterleave;           // SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_*
    UCHAR MaximumModuleSize;           // 2^n
    USHORT SupportedSpeeds;            // SMBIOS_MEMORY_CONTROLLER_SPEEDS_*
    USHORT SupportedTypes;             // SMBIOS_MEMORY_MODULE_TYPE_*

    struct
    {
        union
        {
            UCHAR Requires5000mV : 1; // 5V
            UCHAR Requires3500mV : 1; // 3.3V
            UCHAR Requires3900mV : 1; // 2.9V
            UCHAR Reserved : 5;
        };

        UCHAR Value;
    } ModuleVoltage;

    UCHAR NumberOfSlots;
    USHORT SlotHandles[ANYSIZE_ARRAY];
    // 2.1+
    // SMBIOS_MEMORY_CONTROLLER_INFORMATION_EX
} SMBIOS_MEMORY_CONTROLLER_INFORMATION, *PSMBIOS_MEMORY_CONTROLLER_INFORMATION;

typedef struct _SMBIOS_MEMORY_CONTROLLER_INFORMATION_EX
{
    UCHAR EnabledErrorCorrectionCapabilities; // SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_*
} SMBIOS_MEMORY_CONTROLLER_INFORMATION_EX, *PSMBIOS_MEMORY_CONTROLLER_INFORMATION_EX;

#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_OTHER        ((UCHAR)1)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_UNKNOWN      ((UCHAR)2)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_NONE         ((UCHAR)3)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_8_BIT_PARITY ((UCHAR)4)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_32_BIT_ECC   ((UCHAR)5)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_64_BIT_ECC   ((UCHAR)6)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_128_BIT_ECC  ((UCHAR)7)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_DETECTION_CRC          ((UCHAR)8)

#define SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_OTHER       ((UCHAR)0x01)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_UNKNOWN     ((UCHAR)0x02)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_SINGLE_BIT  ((UCHAR)0x04)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_DOUBLE_BIT  ((UCHAR)0x10)
#define SMBIOS_MEMORY_CONTROLLER_ERROR_CORRECTION_SCRUBBING   ((UCHAR)0x20)

#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_OTHER             ((UCHAR)1)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_UNKNOWN           ((UCHAR)2)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_ONE_WAY           ((UCHAR)3)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_TWO_WAY           ((UCHAR)4)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_FOUR_WAY          ((UCHAR)5)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_EIGHT_WAY         ((UCHAR)6)
#define SMBIOS_MEMORY_CONTROLLER_INTERLEAVE_SIXTEEN_WAY       ((UCHAR)7)

#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_OTHER                 ((USHORT)0x0001)
#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_UNKNOWN               ((USHORT)0x0002)
#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_70NS                  ((USHORT)0x0004)
#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_60NS                  ((USHORT)0x0008)
#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_50NS                  ((USHORT)0x0010)
#define SMBIOS_MEMORY_CONTROLLER_SPEEDS_RESERVED              ((USHORT)0xFFE0)

//
// Memory Module Information (Type 6, Obsolete)
//

#define SMBIOS_MEMORY_MODULE_INFORMATION_TYPE ((UCHAR)6)

typedef union _SMBIOS_MEMORY_MDOULE_SIZE
{
    struct
    {
        UCHAR Size : 7;
        UCHAR DoubleBank : 1;
    };

    UCHAR Value; // SMBIOS_MEMORY_MDOULE_SIZE_VALUE_*
} SMBIOS_MEMORY_MDOULE_SIZE, *PSMBIOS_MEMORY_MDOULE_SIZE;

#define SMBIOS_MEMORY_MDOULE_SIZE_VALUE_NOT_DETERMINABLE ((UCHAR)0x7D)
#define SMBIOS_MEMORY_MDOULE_SIZE_VALUE_NOT_ENABLED      ((UCHAR)0x7E)
#define SMBIOS_MEMORY_MDOULE_SIZE_VALUE_NOT_INSTALLED    ((UCHAR)0x7F)

typedef struct _SMBIOS_MEMORY_MODULE_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR SocketDesignation; // string
    UCHAR BankConnections;
    UCHAR CurrentSpeed;
    USHORT MemoryType;       // SMBIOS_MEMORY_MODULE_TYPE_*
    SMBIOS_MEMORY_MDOULE_SIZE InstalledSize;
    SMBIOS_MEMORY_MDOULE_SIZE EnabledSize;

    struct
    {
        union
        {
            UCHAR UncorrectableErrors : 1;
            UCHAR CorrectableErrors : 1;
            UCHAR SeeEventLog : 1;
            UCHAR Reserved : 5;
        };

        UCHAR Value;
    } ErrorStatus;
} SMBIOS_MEMORY_MODULE_INFORMATION, *PSMBIOS_MEMORY_MODULE_INFORMATION;

#define SMBIOS_MEMORY_MODULE_TYPE_OTHER          ((USHORT)0x0001)
#define SMBIOS_MEMORY_MODULE_TYPE_UNKNOWN        ((USHORT)0x0002)
#define SMBIOS_MEMORY_MODULE_TYPE_STANDARD       ((USHORT)0x0004)
#define SMBIOS_MEMORY_MODULE_TYPE_FAST_PAGE_MODE ((USHORT)0x0008)
#define SMBIOS_MEMORY_MODULE_TYPE_EDO            ((USHORT)0x0010)
#define SMBIOS_MEMORY_MODULE_TYPE_PARITY         ((USHORT)0x0020)
#define SMBIOS_MEMORY_MODULE_TYPE_ECC            ((USHORT)0x0040)
#define SMBIOS_MEMORY_MODULE_TYPE_SIMM           ((USHORT)0x0080)
#define SMBIOS_MEMORY_MODULE_TYPE_DIMM           ((USHORT)0x0100)
#define SMBIOS_MEMORY_MODULE_TYPE_BURST_EDO      ((USHORT)0x0200)
#define SMBIOS_MEMORY_MODULE_TYPE_SDRAM          ((USHORT)0x0400)
#define SMBIOS_MEMORY_MODULE_TYPE_RESERVED       ((USHORT)0xF800)

//
// Cache Information (Type 7)
//

#define SMBIOS_CACHE_INFORMATION_TYPE ((UCHAR)7)

typedef struct _SMBIOS_CACHE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR SocketDesign;          // string

    union
    {
        struct
        {
            USHORT Level : 3;     // L1 through L8
            USHORT Socketed : 1;
            USHORT Reserved : 1;
            USHORT Location : 2;  // SMBIO_CACHE_LOCATION_*
            USHORT Enabled : 1;
            USHORT Mode : 2;      // SMBIOS_CACHE_MODE_*
            USHORT Reserved2 : 6;
        };

        USHORT Value;
    } Configuration;

    USHORT MaximumSize;
    USHORT InstalledSize;
    USHORT SupportedSRAM;         // SMBIOS_CACHE_SUPPORTED_SRAM_*
    USHORT CurrentSRAM;           // SMBIOS_CACHE_SUPPORTED_SRAM_*
    // 2.1+
    UCHAR CacheSpeed;
    UCHAR ErrorCorrectionType;    // SMBIOS_CACHE_ERROR_CORRECTION_*
    UCHAR SystemCacheType;        // SMBIOS_CACHE_SYSTEM_CACHE_*
    UCHAR Associativity;          // SMBIOS_CACHE_ASSOCIATIVITY_*
    // 3.1+
    ULONG MaximumSize2;
    ULONG InstalledSize2;
} SMBIOS_CACHE_INFORMATION, *PSMBIOS_CACHE_INFORMATION;

#define SMBIO_CACHE_LOCATION_INTERNAL                ((USHORT)0)
#define SMBIO_CACHE_LOCATION_EXTERNAL                ((USHORT)1)
#define SMBIO_CACHE_LOCATION_RESERVED                ((USHORT)2)
#define SMBIO_CACHE_LOCATION_UNKNOWN                 ((USHORT)3)

#define SMBIOS_CACHE_MODE_WRITE_THROUGH              ((USHORT)0)
#define SMBIOS_CACHE_MODE_WRITE_BACK                 ((USHORT)1)
#define SMBIOS_CACHE_MODE_VARIES_WITH_MEMORY_ADDRESS ((USHORT)2)
#define SMBIOS_CACHE_MODE_UNKNOWN                    ((USHORT)3)

#define SMBIOS_CACHE_SUPPORTED_SRAM_OTHER            ((USHORT)0x0001)
#define SMBIOS_CACHE_SUPPORTED_SRAM_UNKNOWN          ((USHORT)0x0002)
#define SMBIOS_CACHE_SUPPORTED_SRAM_NON_BURST        ((USHORT)0x0004)
#define SMBIOS_CACHE_SUPPORTED_SRAM_BURST            ((USHORT)0x0008)
#define SMBIOS_CACHE_SUPPORTED_SRAM_PIPELINE_BURST   ((USHORT)0x0010)
#define SMBIOS_CACHE_SUPPORTED_SRAM_SYNCHRONOUS      ((USHORT)0x0020)
#define SMBIOS_CACHE_SUPPORTED_SRAM_ASYNCHRONOUS     ((USHORT)0x0040)
#define SMBIOS_CACHE_SUPPORTED_SRAM_RESERVED         ((USHORT)0xFF80)

#define SMBIOS_CACHE_ERROR_CORRECTION_OTHER          ((UCHAR)1)
#define SMBIOS_CACHE_ERROR_CORRECTION_UNKNOWN        ((UCHAR)2)
#define SMBIOS_CACHE_ERROR_CORRECTION_NONE           ((UCHAR)3)
#define SMBIOS_CACHE_ERROR_CORRECTION_PARITY         ((UCHAR)4)
#define SMBIOS_CACHE_ERROR_CORRECTION_SINGLE_BIT_ECC ((UCHAR)5)
#define SMBIOS_CACHE_ERROR_CORRECTION_MULTI_BIT_ECC  ((UCHAR)6)

#define SMBIOS_CACHE_SYSTEM_CACHE_OTHER              ((UCHAR)1)
#define SMBIOS_CACHE_SYSTEM_CACHE_UNKNOWN            ((UCHAR)2)
#define SMBIOS_CACHE_SYSTEM_CACHE_INSTRUCTION        ((UCHAR)3)
#define SMBIOS_CACHE_SYSTEM_CACHE_DATA               ((UCHAR)4)
#define SMBIOS_CACHE_SYSTEM_CACHE_UNIFIED            ((UCHAR)5)

#define SMBIOS_CACHE_ASSOCIATIVITY_OTHER             ((UCHAR)1)
#define SMBIOS_CACHE_ASSOCIATIVITY_UNKNOWN           ((UCHAR)2)
#define SMBIOS_CACHE_ASSOCIATIVITY_DIRECT_MAPPED     ((UCHAR)3)
#define SMBIOS_CACHE_ASSOCIATIVITY_2_WAY             ((UCHAR)4)
#define SMBIOS_CACHE_ASSOCIATIVITY_4_WAY             ((UCHAR)5)
#define SMBIOS_CACHE_ASSOCIATIVITY_FULL              ((UCHAR)6)
#define SMBIOS_CACHE_ASSOCIATIVITY_8_WAY             ((UCHAR)7)
#define SMBIOS_CACHE_ASSOCIATIVITY_16_WAY            ((UCHAR)8)
#define SMBIOS_CACHE_ASSOCIATIVITY_12_WAY            ((UCHAR)9)
#define SMBIOS_CACHE_ASSOCIATIVITY_24_WAY            ((UCHAR)10)
#define SMBIOS_CACHE_ASSOCIATIVITY_32_WAY            ((UCHAR)11)
#define SMBIOS_CACHE_ASSOCIATIVITY_48_WAY            ((UCHAR)12)
#define SMBIOS_CACHE_ASSOCIATIVITY_64_WAY            ((UCHAR)13)
#define SMBIOS_CACHE_ASSOCIATIVITY_20_WAY            ((UCHAR)14)

//
// Port Connector Information (Type 8)
//

#define SMBIOS_PORT_CONNECTOR_INFORMATION_TYPE ((UCHAR)8)

typedef struct _SMBIOS_PORT_CONNECTOR_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR InternalReferenceDesignator; // string
    UCHAR InternalConnectorType;       // SMBIOS_PORT_CONNECTOR_TYPE_*
    UCHAR ExternalReferenceDesignator; // string
    UCHAR ExternalConnectorType;       // SMBIOS_PORT_CONNECTOR_TYPE_*
    UCHAR PortType;                    // SMBIOS_PORT_CONNECTOR_PORT_TYPE_*
} SMBIOS_PORT_CONNECTOR_INFORMATION, *PSMBIOS_PORT_CONNECTOR_INFORMATION;

#define SMBIOS_PORT_CONNECTOR_TYPE_NONE                    ((UCHAR)0)
#define SMBIOS_PORT_CONNECTOR_TYPE_CENTRONICS              ((UCHAR)1)
#define SMBIOS_PORT_CONNECTOR_TYPE_MINI_CENTRONICS         ((UCHAR)2)
#define SMBIOS_PORT_CONNECTOR_TYPE_PROPRIETARY             ((UCHAR)3)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_25_MIN_MALE          ((UCHAR)4)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_25_MIN_FEMALE        ((UCHAR)5)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_15_MIN_MALE          ((UCHAR)6)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_15_MIN_FEMALE        ((UCHAR)7)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_9_MIN_MALE           ((UCHAR)8)
#define SMBIOS_PORT_CONNECTOR_TYPE_DB_9_MIN_FEMALE         ((UCHAR)9)
#define SMBIOS_PORT_CONNECTOR_TYPE_RJ_11                   ((UCHAR)10)
#define SMBIOS_PORT_CONNECTOR_TYPE_RJ_54                   ((UCHAR)11)
#define SMBIOS_PORT_CONNECTOR_TYPE_50_MIN_MINI_SCSI        ((UCHAR)12)
#define SMBIOS_PORT_CONNECTOR_TYPE_MINI_DIN                ((UCHAR)13)
#define SMBIOS_PORT_CONNECTOR_TYPE_MICRO_DIN               ((UCHAR)14)
#define SMBIOS_PORT_CONNECTOR_TYPE_PS2                     ((UCHAR)15)
#define SMBIOS_PORT_CONNECTOR_TYPE_INFRARED                ((UCHAR)16)
#define SMBIOS_PORT_CONNECTOR_TYPE_HP_HIL                  ((UCHAR)17)
#define SMBIOS_PORT_CONNECTOR_TYPE_USB                     ((UCHAR)18)
#define SMBIOS_PORT_CONNECTOR_TYPE_SSA_SCSI                ((UCHAR)19)
#define SMBIOS_PORT_CONNECTOR_TYPE_CIRCULAR_DIN_8_MALE     ((UCHAR)20)
#define SMBIOS_PORT_CONNECTOR_TYPE_CIRCULAR_DIN_8_FEMALE   ((UCHAR)21)
#define SMBIOS_PORT_CONNECTOR_TYPE_ON_BOARD_IDE            ((UCHAR)22)
#define SMBIOS_PORT_CONNECTOR_TYPE_ON_BOARD_FLOPPY         ((UCHAR)23)
#define SMBIOS_PORT_CONNECTOR_TYPE_9_PIN_DIAL_INLINE       ((UCHAR)24)
#define SMBIOS_PORT_CONNECTOR_TYPE_25_PIN_DIAL_INLINE      ((UCHAR)25)
#define SMBIOS_PORT_CONNECTOR_TYPE_50_PIN_DIAL_INLINE      ((UCHAR)26)
#define SMBIOS_PORT_CONNECTOR_TYPE_68_PIN_DIAL_INLINE      ((UCHAR)27)
#define SMBIOS_PORT_CONNECTOR_TYPE_ON_BOARD_INPUT_CD_ROM   ((UCHAR)28)
#define SMBIOS_PORT_CONNECTOR_TYPE_MINI_CENTRONICS_TYPE_14 ((UCHAR)29)
#define SMBIOS_PORT_CONNECTOR_TYPE_MINI_CENTRONICS_TYPE_26 ((UCHAR)30)
#define SMBIOS_PORT_CONNECTOR_TYPE_MINI_JACK               ((UCHAR)31)
#define SMBIOS_PORT_CONNECTOR_TYPE_BNC                     ((UCHAR)32)
#define SMBIOS_PORT_CONNECTOR_TYPE_1394                    ((UCHAR)33)
#define SMBIOS_PORT_CONNECTOR_TYPE_SAS_SATA                ((UCHAR)34)
#define SMBIOS_PORT_CONNECTOR_TYPE_USB_TYPE_C              ((UCHAR)35)
#define SMBIOS_PORT_CONNECTOR_TYPE_PC_98                   ((UCHAR)160)
#define SMBIOS_PORT_CONNECTOR_TYPE_PC_98_HIRESO            ((UCHAR)161)
#define SMBIOS_PORT_CONNECTOR_TYPE_PC_H98                  ((UCHAR)162)
#define SMBIOS_PORT_CONNECTOR_TYPE_PC_98_NOTE              ((UCHAR)163)
#define SMBIOS_PORT_CONNECTOR_TYPE_PC_98_FULL              ((UCHAR)164)
#define SMBIOS_PORT_CONNECTOR_TYPE_OTHER                   ((UCHAR)255)

#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_NONE               ((UCHAR)0)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PARALLEL_XT_AT     ((UCHAR)1)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PARALLEL_PS2       ((UCHAR)2)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PARALLEL_ECP       ((UCHAR)3)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PARALLEL_EPP       ((UCHAR)4)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PARALLEL_ECP_EPP   ((UCHAR)5)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SERIAL_XT_AT       ((UCHAR)6)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SERIAL_16450       ((UCHAR)7)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SERIAL_16550       ((UCHAR)8)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SERIAL_16550A      ((UCHAR)9)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SCSI               ((UCHAR)10)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_MIDI               ((UCHAR)11)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_JOY_STICK          ((UCHAR)12)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_KEYBOARD           ((UCHAR)13)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_MOUSE              ((UCHAR)14)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SSA_SCSI           ((UHCAR)15)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_USB                ((UCHAR)16)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_FIRE_WIRE          ((UCHAR)17)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PCMCIA_TYPE_I      ((UCHAR)18)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PCMCIA_TYPE_II     ((UCHAR)19)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PCMCIA_TYPE_III    ((UCHAR)20)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_CARD_BUS           ((UCHAR)21)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_ACCESS_BUS         ((UCHAR)22)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SCSI_II            ((UCHAR)23)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SCSI_WIDE          ((UCHAR)24)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PC_98              ((UCHAR)25)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PC_98_HIRESO       ((UCHAR)26)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_PC_98H             ((UCHAR)27)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_VIDEO              ((UCHAR)28)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_AUDIO              ((UCHAR)29)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_MODEM              ((UCHAR)30)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_NETWORK            ((UCHAR)31)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SATA               ((UCHAR)32)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_SAS                ((UCHAR)33)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_MFDP               ((UCHAR)34)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_THUNDERBOLT        ((UCHAR)35)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_8251               ((UCHAR)160)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_8251_FIFO          ((UCHAR)161)
#define SMBIOS_PORT_CONNECTOR_PORT_TYPE_8251_OTHER         ((UCHAR)255)

//
// System Slots (Type 9)
//

#define SMBIOS_SYSTEM_SLOT_INFORMATION_TYPE ((UCHAR)9)

typedef struct _SMBIOS_SYSTEM_SLOT_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR Designation;      // string
    UCHAR Type;             // SMBIOS_SYSTEM_SLOT_TYPE_*
    UCHAR BusWidth;         // SMBIOS_SYSTEM_SLOT_BUS_WIDTH_*
    UCHAR CurrentUsage;     // SMBIOS_SYSTEM_SLOT_USAGE_*
    UCHAR Length;           // SMBIOS_SYSTEM_SLOT_LENGTH_*
    USHORT Identifier;
    UCHAR Characteristics;  // SMBIOS_SYSTEM_SLOT_FLAG_*
    // 2.1+
    UCHAR Characteristics2; // SMBIOS_SYSTEM_SLOT_FLAG_2_*
    // 2.6+
    USHORT SegmentGroup;
    UCHAR BusNumber;

    union
    {
        struct
        {
            USHORT FunctionNumber : 3;
            USHORT DeviceNumber : 5;
        };

        USHORT Value;
    } DeviceFunctionNumber;

    // 3.2+
    UCHAR BusWidthBase;
    UCHAR PeerGroupingCount;
    UCHAR PeetGroups[ANYSIZE_ARRAY];
    // 3.4+
    // SMBIOS_SYSTEM_SLOT_INFORMATION_EX
} SMBIOS_SYSTEM_SLOT_INFORMATION, *PSMBIOS_SYSTEM_SLOT_INFORMATION;

typedef struct _SMBIOS_SYSTEM_SLOT_INFORMATION_EX
{
    UCHAR Information;
    UCHAR PhysicalWidth;
    USHORT Pitch;
    UCHAR Height;        // SMBIOS_SYSTEM_SLOT_HEIGHT_*
} SMBIOS_SYSTEM_SLOT_INFORMATION_EX, *PSMBIOS_SYSTEM_SLOT_INFORMATION_EX;

#define SMBIOS_SYSTEM_SLOT_TYPE_OTHER                                             ((UCHAR)1)
#define SMBIOS_SYSTEM_SLOT_TYPE_UNKNOWN                                           ((UCHAR)2)
#define SMBIOS_SYSTEM_SLOT_TYPE_ISA                                               ((UCHAR)3)
#define SMBIOS_SYSTEM_SLOT_TYPE_MCA                                               ((UCHAR)4)
#define SMBIOS_SYSTEM_SLOT_TYPE_EISA                                              ((UCHAR)5)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI                                               ((UCHAR)6)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCMCIA                                            ((UCHAR)7)
#define SMBIOS_SYSTEM_SLOT_TYPE_VL_VESA                                           ((UCHAR)8)
#define SMBIOS_SYSTEM_SLOT_TYPE_PROPRIETARY                                       ((UCHAR)9)
#define SMBIOS_SYSTEM_SLOT_TYPE_PROCESSOR_CARD_SLOT                               ((UCHAR)10)
#define SMBIOS_SYSTEM_SLOT_TYPE_PROPRIETARY_MEMORY_CARD_SLOT                      ((UCHAR)11)
#define SMBIOS_SYSTEM_SLOT_TYPE_IO_RISER_CARD_SLOT                                ((UCHAR)12)
#define SMBIOS_SYSTEM_SLOT_TYPE_NUBUS                                             ((UCHAR)13)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_66MHZ_CAPABLE                                 ((UCHAR)14)
#define SMBIOS_SYSTEM_SLOT_TYPE_AGP                                               ((UCHAR)15)
#define SMBIOS_SYSTEM_SLOT_TYPE_AGP_2X                                            ((UCHAR)16)
#define SMBIOS_SYSTEM_SLOT_TYPE_AGP_4X                                            ((UCHAR)17)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_X                                             ((UCHAR)18)
#define SMBIOS_SYSTEM_SLOT_TYPE_AGP_8X                                            ((UCHAR)19)
#define SMBIOS_SYSTEM_SLOT_TYPE_M_2_SOCKET_1_DP_MECHANICAL_KEY_A                  ((UCHAR)20)
#define SMBIOS_SYSTEM_SLOT_TYPE_M_2_SOCKET_1_SD_MECHANICAL_KEY_E                  ((UCHAR)21)
#define SMBIOS_SYSTEM_SLOT_TYPE_M_2_SOCKET_2_MECHANICAL_KEY_B                     ((UCHAR)22)
#define SMBIOS_SYSTEM_SLOT_TYPE_M_2_SOCKET_3_MECHANICAL_KEY_M                     ((UCHAR)23)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_TYPE_I                                        ((UCHAR)24)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_TYPE_II                                       ((UCHAR)25)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_TYPE_III_STANDARD_CONNECTOR                   ((UCHAR)26)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_TYPE_III_HE_CONNECTOR                         ((UCHAR)27)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_TYPE_IV                                       ((UCHAR)28)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_3_0_TYPE_A                                    ((UCHAR)29)
#define SMBIOS_SYSTEM_SLOT_TYPE_MXM_3_0_TYPE_B                                    ((UCHAR)30)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_SFF_8639_U_2                    ((UCHAR)31)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_SFF_8639_U_2                    ((UCHAR)32)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_MINI_52_PIN_CEM_2_0_WITH_KEEP_OUTS    ((UCHAR)33)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_MINI_52_PIN_CEM_2_0_WITHOUT_KEEP_OUTS ((UCHAR)34)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_MINI_76_PIN_CEM_2_0                   ((UCHAR)35)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_SFF_8639_U_2                    ((UCHAR)36)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_SFF_8639_U_2                    ((UCHAR)37)
#define SMBIOS_SYSTEM_SLOT_TYPE_OCP_NIC_3_0_SMALL_FORM_FACTOR_SFF                 ((UCHAR)38)
#define SMBIOS_SYSTEM_SLOT_TYPE_OCP_NIC_3_0_LARGE_FORM_FACTOR_LFF                 ((UCHAR)39)
#define SMBIOS_SYSTEM_SLOT_TYPE_OCP_NIC_PRIOR_TO_3_0                              ((UCHAR)40)
#define SMBIOS_SYSTEM_SLOT_TYPE_CXL_FLEXBUS_1_0                                   ((UCHAR)48)
#define SMBIOS_SYSTEM_SLOT_TYPE_PC_98_C20                                         ((UCHAR)160)
#define SMBIOS_SYSTEM_SLOT_TYPE_PC_98_C24                                         ((UCHAR)161)
#define SMBIOS_SYSTEM_SLOT_TYPE_PC_98_E                                           ((UCHAR)162)
#define SMBIOS_SYSTEM_SLOT_TYPE_PC_98_LOCAL_BUS                                   ((UCHAR)163)
#define SMBIOS_SYSTEM_SLOT_TYPE_PC_98_CARD                                        ((UCHAR)164)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS                                       ((UCHAR)165)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_X1                                    ((UCHAR)166)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_X2                                    ((UCHAR)167)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_X4                                    ((UCHAR)168)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_X8                                    ((UCHAR)169)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_X16                                   ((UCHAR)170)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2                                 ((UCHAR)171)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_X1                              ((UCHAR)172)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_X2                              ((UCHAR)173)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_X4                              ((UCHAR)174)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_X8                              ((UCHAR)175)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_2_X16                             ((UCHAR)176)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3                                 ((UCHAR)177)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_X1                              ((UCHAR)178)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_X2                              ((UCHAR)179)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_X4                              ((UCHAR)180)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_X8                              ((UCHAR)181)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_3_X16                             ((UCHAR)182)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4                                 ((UCHAR)183)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_X1                              ((UCHAR)184)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_X2                              ((UCHAR)185)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_X4                              ((UCHAR)186)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_X8                              ((UCHAR)187)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_4_X16                             ((UCHAR)188)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5                                 ((UCHAR)189)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_X1                              ((UCHAR)190)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_X2                              ((UCHAR)191)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_X4                              ((UCHAR)192)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_X8                              ((UCHAR)193)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_5_X16                             ((UCHAR)194)
#define SMBIOS_SYSTEM_SLOT_TYPE_PCI_EXPRESS_GEN_6                                 ((UCHAR)195)
#define SMBIOS_SYSTEM_SLOT_TYPE_EDSFF_E1_S_E1_L                                   ((UCHAR)196)
#define SMBIOS_SYSTEM_SLOT_TYPE_EDSFF_E3_S_E3_L                                   ((UCHAR)197)

#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_OTHER                                        ((UCHAR)1)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_UNKNOWN                                      ((UCHAR)2)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_8_BIT                                        ((UCHAR)3)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_16_BIT                                       ((UCHAR)4)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_32_BIT                                       ((UCHAR)5)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_64_BIT                                       ((UCHAR)6)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_128_BIT                                      ((UCHAR)7)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_1X_OR_1X                                     ((UCHAR)8)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_2X_OR_3X                                     ((UCHAR)9)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_4X_OR_4X                                     ((UCHAR)10)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_8X_OR_8X                                     ((UCHAR)11)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_12X_OR_12X                                   ((UCHAR)12)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_16X_OR_16X                                   ((UCHAR)13)
#define SMBIOS_SYSTEM_SLOT_BUS_WIDTH_32X_OR_32X                                   ((UCHAR)14)

#define SMBIOS_SYSTEM_SLOT_USAGE_OTHER                                            ((UCHAR)1)
#define SMBIOS_SYSTEM_SLOT_USAGE_UNKNOWN                                          ((UCHAR)2)
#define SMBIOS_SYSTEM_SLOT_USAGE_AVAILABLE                                        ((UCHAR)3)
#define SMBIOS_SYSTEM_SLOT_USAGE_IN_USE                                           ((UCHAR)4)
#define SMBIOS_SYSTEM_SLOT_USAGE_UNAVAILABLE                                      ((UCHAR)5)

#define SMBIOS_SYSTEM_SLOT_LENGTH_OTHER                                           ((UCHAR)1)
#define SMBIOS_SYSTEM_SLOT_LENGTH_UNKNOWN                                         ((UCHAR)2)
#define SMBIOS_SYSTEM_SLOT_LENGTH_SHORT                                           ((UCHAR)3)
#define SMBIOS_SYSTEM_SLOT_LENGTH_LONG                                            ((UCHAR)4)
#define SMBIOS_SYSTEM_SLOT_LENGTH_2_5_DRIVE                                       ((UCHAR)5)
#define SMBIOS_SYSTEM_SLOT_LENGTH_3_4_DRIVE                                       ((UCHAR)6)

#define SMBIOS_SYSTEM_SLOT_FLAG_UNKNOWN                                           ((UCHAR)0x01)
#define SMBIOS_SYSTEM_SLOT_FLAG_5000MV                                            ((UCHAR)0x02)
#define SMBIOS_SYSTEM_SLOT_FLAG_3300MV                                            ((UCHAR)0x04)
#define SMBIOS_SYSTEM_SLOT_FLAG_SHARED                                            ((UCHAR)0x08)
#define SMBIOS_SYSTEM_SLOT_FLAG_CARD_16                                           ((UCHAR)0x10)
#define SMBIOS_SYSTEM_SLOT_FLAG_CARD_BUS                                          ((UCHAR)0x20)
#define SMBIOS_SYSTEM_SLOT_FLAG_CARD_ZOOM_VIDEO                                   ((UCHAR)0x40)
#define SMBIOS_SYSTEM_SLOT_FLAG_CARD_MODEN_RING_RESUME                            ((UCHAR)0x80)

#define SMBIOS_SYSTEM_SLOT_FLAG_2_PME_SIGNAL                                      ((UCHAR)0x01)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_HOT_PLUG                                        ((UCHAR)0x02)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_SMBUS_SIGNAL                                    ((UCHAR)0x04)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_PCIE_BIFURCATION                                ((UCHAR)0x08)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_SURPRISE_REMOVAL                                ((UCHAR)0x10)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_FLEXBUS_CLX_1                                   ((UCHAR)0x20)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_FLEXBUS_CLX_2                                   ((UCHAR)0x40)
#define SMBIOS_SYSTEM_SLOT_FLAG_2_FLEXBUS_CLX_4                                   ((UCHAR)0x80)

#define SMBIOS_SYSTEM_SLOT_HEIGHT_NOT_APPLICABLE                                  ((UCHAR)0)
#define SMBIOS_SYSTEM_SLOT_HEIGHT_OTHER                                           ((UCHAR)1)
#define SMBIOS_SYSTEM_SLOT_HEIGHT_UNKNOWN                                         ((UCHAR)2)
#define SMBIOS_SYSTEM_SLOT_HEIGHT_FULL_HEIGHT                                     ((UCHAR)3)
#define SMBIOS_SYSTEM_SLOT_HEIGHT_LOW_PROFILE                                     ((UCHAR)4)

//
// On Board Devices Information (Type 10, Obsolete)
//

#define SMBIOS_ON_BOARD_DEVICE_INFORMATION_TYPE ((UCHAR)10)

typedef struct _SMBIOS_ON_BOARD_DEVICE_ENTRY
{
    union
    {
        struct
        {
            UCHAR Type : 7; // SMBIOS_ON_BOARD_DEVICE_TYPE_*
            UCHAR Enabled : 1;
        };

        UCHAR Value;
    } Device;
} SMBIOS_ON_BOARD_DEVICE_ENTRY, *PSMBIOS_ON_BOARD_DEVICE_ENTRY;

typedef struct _SMBIOS_ON_BOARD_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    SMBIOS_ON_BOARD_DEVICE_ENTRY Devices[ANYSIZE_ARRAY]; // (Header.Length - 4) / 2
    // UCHAR Description; // string
} SMBIOS_ON_BOARD_DEVICE_INFORMATION, *PSMBIOS_ON_BOARD_DEVICE_INFORMATION;

#define SMBIOS_ON_BOARD_DEVICE_TYPE_OTHER           ((UCHAR)1)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_UNKNOWN         ((UCHAR)2)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_VIDEO           ((UCHAR)3)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_SCSI_CONTROLLER ((UCHAR)4)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_EATHERNET       ((UCHAR)5)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_TOKEN_RING      ((UCHAR)6)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_SOUND           ((UCHAR)7)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_PATA_CONTROLLER ((UCHAR)8)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_SATA_CONTROLLER ((UCHAR)9)
#define SMBIOS_ON_BOARD_DEVICE_TYPE_SAS_CONTROLLER  ((UCHAR)10)

#define SMBIOS_OEM_STRING_INFORMATION_TYPE ((UCHAR)11)

//
// OEM Strings (Type 11)
//

typedef struct _SMBIOS_OEM_STRING_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR Count;
} SMBIOS_OEM_STRING_INFORMATION, *PSMBIOS_OEM_STRING_INFORMATION;

//
// System Configuration Options (Type 12)
//

#define SMBIOS_SYSTEM_CONFIGURATION_OPTION_INFORMATION_TYPE ((UCHAR)12)

typedef struct _SMBIOS_SYSTEM_CONFIGURATION_OPTION_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR Count;
} SMBIOS_SYSTEM_CONFIGURATION_OPTION_INFORMATION, *PSMBIOS_SYSTEM_CONFIGURATION_OPTION_INFORMATION;

//
// Firmware Language Information (Type 13)
//

#define SMBIOS_FIRMWARE_LANGUAGE_INFORMATION_TYPE ((UCHAR)13)

typedef struct _SMBIOS_FIRMWARE_LANGUAGE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    UCHAR InstallableLanguages;

    union
    {
        struct
        {
            // 2.1+
            UCHAR AbbreviatedFormat : 1;
            UCHAR Reserved : 7;
        };

        UCHAR Value;
    } Flags;

    UCHAR Reserved[15];
    UCHAR CurrentLanguage; // string
} SMBIOS_FIRMWARE_LANGUAGE_INFORMATION, *PSMBIOS_FIRMWARE_LANGUAGE_INFORMATION;

//
// Group Associations (Type 14)
//

#define SMBIOS_GROUP_ASSOCIATION_INFORMATION_TYPE ((UCHAR)14)

typedef struct _SMBIOS_GROUP_ASSOCIATION_INFORMATION
{
    SMBIOS_HEADER Header;
    UCHAR Name;        // string
    UCHAR ItemType;
    USHORT ItemHandle;
} SMBIOS_GROUP_ASSOCIATION_INFORMATION, *PSMBIOS_GROUP_ASSOCIATION_INFORMATION;

//
// System Event Log (Type 15)
//

#define SMBIOS_SYSTEM_EVENT_LOG_INFORMATION_TYPE ((UCHAR)15)

typedef struct _SMBIOS_SYSTEM_EVENT_LOG_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.0+
    USHORT AreaLength;
    USHORT HeaderStartOffset;
    USHORT DataStartOffset;
    UCHAR AccessMethod;

    struct
    {
        union
        {
            UCHAR Valid : 1;
            UCHAR Full : 1;
            UCHAR Reserved : 6;
        };

        UCHAR Value;
    } Status;

    ULONG ChangeToken;
    ULONG AccessMethodAddress;
    // 2.1+
    UCHAR HeaderFormat;
    UCHAR NumberOfDescriptors;
    UCHAR LengthOfDescriptors;
    UCHAR Descriptors[ANYSIZE_ARRAY];
} SMBIOS_SYSTEM_EVENT_LOG_INFORMATION, *PSMBIOS_SYSTEM_EVENT_LOG_INFORMATION;

//
// Physical Memory Array (Type 16)
//

#define SMBIOS_PHYSICAL_MEMORY_ARRAY_INFORMATION_TYPE ((UCHAR)16)

typedef struct _SMBIOS_PHYSICAL_MEMORY_ARRAY_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    UCHAR Location;        // SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_*
    UCHAR Use;             // SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_*
    UCHAR ErrorCorrection; // SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_*
    ULONG MaximumCapacity;
    USHORT ErrorInformationHandle;
    USHORT NumberOfMemoryDevices;
    // 2.7+
    ULONGLONG ExtendedMaximumCapacity;
} SMBIOS_PHYSICAL_MEMORY_ARRAY_INFORMATION, *PSMBIOS_PHYSICAL_MEMORY_ARRAY_INFORMATION;

#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_OTHER                  ((UCHAR)1)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_UNKNOWN                ((UCHAR)2)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_MOTHERBOARD            ((UCHAR)3)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_ISA                    ((UCHAR)4)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_EISA                   ((UCHAR)5)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PCI                    ((UCHAR)6)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_MCA                    ((UCHAR)7)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PCMCIA                 ((UCHAR)8)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PROPRIETARY            ((UCHAR)9)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_NUBUS                  ((UCHAR)10)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PC_98_C20              ((UCHAR)160)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PC_98_C24              ((UCHAR)161)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PC_98_E                ((UCHAR)162)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PC_98_LOCAL_BUS        ((UCHAR)163)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_LOCATION_PC_98_CXL              ((UCHAR)164)

#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_OTHER                       ((UCHAR)1)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_UNKNOWN                     ((UCHAR)2)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_SYSTEM                      ((UCHAR)3)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_VIDEO                       ((UCHAR)4)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_FLASH                       ((UCHAR)5)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_NON_VOLATILE                ((UCHAR)6)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_USE_CACHE                       ((UCHAR)7)

#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_OTHER          ((UCHAR)1)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_UNKNOWN        ((UCHAR)2)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_NONE           ((UCHAR)3)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_PARITY         ((UCHAR)4)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_SINGLE_BIT_ECC ((UCHAR)5)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_MULTI_BIT_ECC  ((UCHAR)6)
#define SMBIOS_PHYSICAL_MEMORY_ARRAY_ERROR_CORRECTION_CRC            ((UCHAR)7)

//
// Memory Device (Type 17)
//

#define SMBIOS_MEMORY_DEVICE_INFORMATION_TYPE ((UHCAR)17)

typedef struct _SMBIOS_MEMORY_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    USHORT PhysicalArrayHandle;
    USHORT MemoryErrorInformationHandle;
    USHORT TotalWidth;
    USHORT DataWidth;

    struct
    {
        union
        {
            USHORT Size : 15;
            USHORT Granularity : 1; // 0 = megabytes, 1 = kilobytes
        };

        USHORT Value;
    } Size;

    UCHAR FormFactor;        // SMBIOS_MEMORY_DEVICE_FORM_FACTOR_*
    UCHAR DeviceSet;
    UCHAR DeviceLocator;     // string
    UCHAR BankLocator;       // string
    UCHAR MemoryType;        // SMBIOS_MEMORY_DEVICE_TYPE_*
    USHORT TypeDetail;       // SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_*
    USHORT Speed;
    // 2.3+
    UCHAR Manufacturer;      // string
    UCHAR SerialNumber;      // string
    UCHAR AssetTag;          // string
    UCHAR PartNumber;        // string
    // 2.6+
    struct
    {
        union
        {
            UCHAR Rank : 4;
            UCHAR Reserved : 4;
        };

        UCHAR Value;
    } Attributes;

    // 2.7+
    ULONG ExtendedSize;
    USHORT ConfiguredClockSpeed;
    // 2.8+
    USHORT MinimumVoltage;
    USHORT MaximumVoltage;
    USHORT ConfiguredVoltage;
    // 3.2+
    UCHAR Technology;        // SMBIOS_MEMORY_DEVICE_TECHNOLOGY_*
    USHORT ModeCapabilities; // SMBIOS_MEMORY_DEVICE_MODE_*
    UCHAR FirmwareVersion;   // string
    USHORT ModuleManufacturerID;
    USHORT ModuleProductID;
    USHORT SubsystemControllerManufacturerID;
    USHORT SubsystemControllerProductID;
    ULONGLONG NonVolatileSize;
    ULONGLONG VolatileSize;
    ULONGLONG CacheSize;
    ULONGLONG LogicalSize;
    // 3.3+
    ULONG ExtendedSpeed;
    ULONG ExtendedConfiguredSpeed;
    // 3.7+
    USHORT PMIC0ManufacturerID;
    USHORT PMIC0Revision;
    USHORT RCDManufacturerID;
    USHORT RCDRevision;
} SMBIOS_MEMORY_DEVICE_INFORMATION, *PSMBIOS_MEMORY_DEVICE_INFORMATION;

#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_OTHER                ((UCHAR)1)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_UNKNOWN              ((UCHAR)2)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_SIMM                 ((UCHAR)3)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_SIP                  ((UCHAR)4)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_CHIP                 ((UCHAR)5)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_DIP                  ((UCHAR)6)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_ZIP                  ((UCHAR)7)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_PROPRIETARY          ((UCHAR)8)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_DIMM                 ((UCHAR)9)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_TSOP                 ((UCHAR)10)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_ROW_OF_CHIPS         ((UCHAR)11)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_RIMM                 ((UCHAR)12)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_SODIMM               ((UCHAR)13)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_SRIMM                ((UCHAR)14)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_FB_DIMM              ((UCHAR)15)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_DIE                  ((UCHAR)16)
#define SMBIOS_MEMORY_DEVICE_FORM_FACTOR_CAMM                 ((UCHAR)17)

#define SMBIOS_MEMORY_DEVICE_TYPE_OTHER                       ((UCHAR)1)
#define SMBIOS_MEMORY_DEVICE_TYPE_UNKNOWN                     ((UCHAR)2)
#define SMBIOS_MEMORY_DEVICE_TYPE_DRAM                        ((UCHAR)3)
#define SMBIOS_MEMORY_DEVICE_TYPE_EDRAM                       ((UCHAR)4)
#define SMBIOS_MEMORY_DEVICE_TYPE_VRAM                        ((UCHAR)5)
#define SMBIOS_MEMORY_DEVICE_TYPE_SRAM                        ((UCHAR)6)
#define SMBIOS_MEMORY_DEVICE_TYPE_RAM                         ((UCHAR)7)
#define SMBIOS_MEMORY_DEVICE_TYPE_ROM                         ((UCHAR)8)
#define SMBIOS_MEMORY_DEVICE_TYPE_FLASH                       ((UCHAR)9)
#define SMBIOS_MEMORY_DEVICE_TYPE_EEPROM                      ((UCHAR)10)
#define SMBIOS_MEMORY_DEVICE_TYPE_FEPROM                      ((UCHAR)11)
#define SMBIOS_MEMORY_DEVICE_TYPE_EPROM                       ((UCHAR)12)
#define SMBIOS_MEMORY_DEVICE_TYPE_CDRAM                       ((UCHAR)13)
#define SMBIOS_MEMORY_DEVICE_TYPE_3DRAM                       ((UCHAR)14)
#define SMBIOS_MEMORY_DEVICE_TYPE_SDRAM                       ((UCHAR)15)
#define SMBIOS_MEMORY_DEVICE_TYPE_SGRAM                       ((UCHAR)16)
#define SMBIOS_MEMORY_DEVICE_TYPE_RDRAM                       ((UCHAR)17)
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR                         ((UCHAR)18)
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR2                        ((UCHAR)19)
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR2_FB_DIMM                ((UCHAR)20)
// 21-23 reserved
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR3                        ((UCHAR)24)
#define SMBIOS_MEMORY_DEVICE_TYPE_FBD2                        ((UCHAR)25)
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR4                        ((UCHAR)26)
#define SMBIOS_MEMORY_DEVICE_TYPE_LPDDR                       ((UCHAR)27)
#define SMBIOS_MEMORY_DEVICE_TYPE_LPDDR2                      ((UCHAR)28)
#define SMBIOS_MEMORY_DEVICE_TYPE_LPDDR3                      ((UCHAR)29)
#define SMBIOS_MEMORY_DEVICE_TYPE_LPDDR4                      ((UCHAR)30)
#define SMBIOS_MEMORY_DEVICE_TYPE_LOCAL_NON_VOLATILE          ((UCHAR)31)
#define SMBIOS_MEMORY_DEVICE_TYPE_HBM                         ((UCHAR)32)
#define SMBIOS_MEMORY_DEVICE_TYPE_HBM2                        ((UCHAR)33)
#define SMBIOS_MEMORY_DEVICE_TYPE_DDR5                        ((UCHAR)34)
#define SMBIOS_MEMORY_DEVICE_TYPE_LPDDR5                      ((UCHAR)35)
#define SMBIOS_MEMORY_DEVICE_TYPE_HBM3                        ((UCHAR)36)

#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_RESERVED             ((USHORT)0x0001)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_OTHER                ((USHORT)0x0002)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_UNKNOWN              ((USHORT)0x0004)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_FAST_PAGED           ((USHORT)0x0008)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_STATIC_COL           ((USHORT)0x0010)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_PSEUDO_STATIC        ((USHORT)0x0020)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_RAMBUS               ((USHORT)0x0040)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_SYNCHRONOUS          ((USHORT)0x0080)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_CMOS                 ((USHORT)0x0100)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_EDO                  ((USHORT)0x0200)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_WINDOW_DRAM          ((USHORT)0x0400)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_CACHE_DRAM           ((USHORT)0x0800)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_NON_VOLATILE         ((USHORT)0x1000)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_BUFFERED             ((USHORT)0x2000)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_UNBUFFERED           ((USHORT)0x4000)
#define SMBIOS_MEMORY_DEVICE_TYPE_DETAIL_LRDIMM               ((USHORT)0x8000)

#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_OTHER                 ((UCHAR)1)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_UNKNOWN               ((UCHAR)2)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_DRAM                  ((UCHAR)3)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_NVDIMM_N              ((UCHAR)4)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_NVDIMM_F              ((UCHAR)5)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_NVDIMM_P              ((UCHAR)6)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_INTEL_OPTANE          ((UCHAR)7)
#define SMBIOS_MEMORY_DEVICE_TECHNOLOGY_MRDIMM                ((UCHAR)8)

#define SMBIOS_MEMORY_DEVICE_MODE_RESERVED                    ((USHORT)0x0001)
#define SMBIOS_MEMORY_DEVICE_MODE_OTHER                       ((USHORT)0x0002)
#define SMBIOS_MEMORY_DEVICE_MODE_UNKNOWN                     ((USHORT)0x0004)
#define SMBIOS_MEMORY_DEVICE_MODE_VOLATILE                    ((USHORT)0x0008)
#define SMBIOS_MEMORY_DEVICE_MODE_BYTE_ACCESSIBLE_PERSISTENT  ((USHORT)0x0010)
#define SMBIOS_MEMORY_DEVICE_MODE_BLOCK_ACCESSIBLE_PERSISTENT ((USHORT)0x0020)
#define SMBIOS_MEMORY_DEVICE_MODE_RESERVED2                   ((USHORT)0xFFC0)

//
// 32-Bit Memory Error Information (Type 18)
//

#define SMBIOS_32_BIT_MEMORY_ERROR_INFORMATION_TYPE ((UCHAR)18)

typedef struct _SMBIOS_32_BIT_MEMORY_ERROR_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    UCHAR Type;        // SMBIOS_MEMORY_ERROR_TYPE_*
    UCHAR Granularity; // SMBIOS_MEMORY_ERROR_GRANULARITY_*
    UCHAR Operation;   // SMBIOS_MEMORY_ERROR_OPERATION_*
    ULONG VendorSyndrome;
    ULONG ArrayErrorAddress;
    ULONG DeviceErrorAddress;
    ULONG ErrorResolution;
} SMBIOS_32_BIT_MEMORY_ERROR_INFORMATION, *PSMBIOS_32_BIT_MEMORY_ERROR_INFORMATION;

//
// Memory Array Mapped Address (Type 19)
//

#define SMBIOS_MEMORY_ARRAY_MAPPED_ADDRESS_INFORMATION_TYPE ((UCHAR)19)

typedef struct _SMBIOS_MEMORY_ARRAY_MAPPED_ADDRESS_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    ULONG StartingAddress;
    ULONG EndingAddress;
    USHORT ArrayHandle;
    UCHAR PartitionWidth;
    // 2.7+
    ULONGLONG ExtendedStartingAddress;
    ULONGLONG ExtendedEndingAddress;
} SMBIOS_MEMORY_ARRAY_MAPPED_ADDRESS_INFORMATION, *PSMBIOS_MEMORY_ARRAY_MAPPED_ADDRESS_INFORMATION;

//
// Memory Device Mapped Address (Type 20)
//

#define SMBIOS_MEMORY_DEVICE_MAPPED_DATA_INFORMATION_TYPE ((UCHAR)20)

typedef struct _SMBIOS_MEMORY_DEVICE_MAPPED_DATA_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    USHORT StartingAddress;
    USHORT EndingAddress;
    USHORT DeviceHandle;
    USHORT ArrayMappedAddressHandle;
    UCHAR PartitionRowPosition;
    UCHAR InterleavePosition;
    UCHAR InterleavedDataDepth;
    // 2.7+
    ULONGLONG ExtendedStartingAddress;
    ULONGLONG ExtendedEndingAddress;
} SMBIOS_MEMORY_DEVICE_MAPPED_DATA_INFORMATION, *PSMBIOS_MEMORY_DEVICE_MAPPED_DATA_INFORMATION;

//
// Built-in Pointing Device (Type 21)
//

#define SMBIOS_BUILT_IN_POINTING_DEVICE_INFORMATION_TYPE ((UCHAR)21)

typedef struct _SMBIOS_BUILT_IN_POINTING_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    UCHAR PointerType;      // SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_*
    UCHAR PointerInterface; // SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_*
    UCHAR PointerButtons;
} SMBIOS_BUILT_IN_POINTING_DEVICE_INFORMATION, *PSMBIOS_BUILT_IN_POINTING_DEVICE_INFORMATION;

#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_OTHER          ((UCHAR)1)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_UNKNOWN        ((UCHAR)2)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_MOUSE          ((UCHAR)3)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_TRACK_BALL     ((UCHAR)4)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_TRACK_POINT    ((UCHAR)5)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_GLIDE_POINT    ((UCHAR)6)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_TOUCH_PAD      ((UCHAR)7)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_TOUCH_SCREEN   ((UCHAR)8)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_TYPE_OPTICAL_SENSOR ((UCHAR)9)

#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_OTHER     ((UCHAR)1)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_UNKNOWN   ((UCHAR)2)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_SERIAL    ((UCHAR)3)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_PS2       ((UCHAR)4)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_INFRARED  ((UCHAR)5)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_HP_HIL    ((UCHAR)6)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_BUS_MOUSE ((UCHAR)7)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_ADB       ((UCHAR)8)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_DB_9      ((UCHAR)160)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_MICRO_DIN ((UCHAR)161)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_USB       ((UCHAR)162)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_I2C       ((UCHAR)163)
#define SMBIOS_BUILT_IN_POINTING_DEVICE_INTERFACE_SPI       ((UCHAR)164)

//
// Portable Battery (Type 22)
//

#define SMBIO_PORTABLE_BATTERY_INFORMATION_TYPE ((UCHAR)22)

typedef struct _SMBIOS_PORTABLE_BATTERY_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.1+
    UCHAR Location;             // string
    UCHAR Manufacturer;         // string
    UCHAR ManufactureDate;      // string
    UCHAR SerialNumber;         // string
    UCHAR DeviceName;           // string
    UCHAR DeviceChemistry;      // SMBIOS_PORTABLE_BATTERY_CHEMISTRY_*
    USHORT DesignCapacity;
    USHORT DesignVoltage;
    UCHAR SBDSVersionNumber;    // string
    UCHAR MaximumError;
    // 2.2+
    USHORT SBDSSerialNumber;
    USHORT SBDSManufactureDate;
    UCHAR SBDSDeviceChemistry;  // string
    UCHAR DesignCapacityMultiplier;
    ULONG OEMSpecific;
} SMBIOS_PORTABLE_BATTERY_INFORMATION, *PSMBIOS_PORTABLE_BATTERY_INFORMATION;

#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_OTHER           ((UCHAR)1)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_UNKNOWN         ((UCHAR)2)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_LEAD_ACID       ((UCHAR)3)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_NICKEL_CADMIUM  ((UCHAR)4)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_NICKEL_METAL    ((UCHAR)5)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_LITHIUM_ION     ((UCHAR)6)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_ZINC_AIR        ((UCHAR)7)
#define SMBIO_PORTABLE_BATTERY_CHEMISTRY_LITHIUM_POLYMER ((UCHAR)8)

//
// System Reset (Type 23)
//

#define SMBIOS_SYSTEM_RESET_INFORMATION_TYPE ((UCHAR)23)

typedef struct _SMBIOS_SYSTEM_RESET_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+

    struct
    {
        union
        {
            UCHAR UserEnabled : 1;
            UCHAR WatchdogReset : 2;   // SMBIOS_SYSTEM_RESET_WATCHDOG_*
            UCHAR WatchdogAction : 2;  // SMBIOS_SYSTEM_RESET_WATCHDOG_*
            UCHAR WatchdogExists : 1;
            UCHAR Reserved : 2;
        };

        UCHAR Value;
    } Capabilities;

    USHORT ResetCount;
    USHORT ResetLimit;
    USHORT TimerInterval;
    USHORT Timeout;
} SMBIOS_SYSTEM_RESET_INFORMATION, *PSMBIOS_SYSTEM_RESET_INFORMATION;

#define SMBIOS_SYSTEM_RESET_WATCHDOG_RESERVED         ((UCHAR)0)
#define SMBIOS_SYSTEM_RESET_WATCHDOG_OPERATING_SYSTEM ((UCHAR)1)
#define SMBIOS_SYSTEM_RESET_WATCHDOG_SYSTEM_UTILITES  ((UCHAR)2)
#define SMBIOS_SYSTEM_RESET_WATCHDOG_DO_NOT_REBOOT    ((UCHAR)3)

//
// Hardware Security (Type 24)
//

#define SMBIOS_HARDWARE_SECURITY_INFORMATION_TYPE ((UCHAR)24)

typedef struct _SMBIOS_HARDWARE_SECURITY_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+

    struct
    {
        union
        {
            UCHAR FrontPanelReset : 2;       // SMBIOS_HARDWARE_SECURITY_SETTING_*
            UCHAR AdministratorPassword : 2; // SMBIOS_HARDWARE_SECURITY_SETTING_*
            UCHAR KeyboardPassword : 2;      // SMBIOS_HARDWARE_SECURITY_SETTING_*
            UCHAR PowerOnPassword : 2;       // SMBIOS_HARDWARE_SECURITY_SETTING_*
        };

        UCHAR Value;
    } HardwareSecuritySettings;
} SMBIOS_HARDWARE_SECURITY_INFORMATION, *PSMBIOS_HARDWARE_SECURITY_INFORMATION;

#define SMBIOS_HARDWARE_SECURITY_SETTING_DISABLED        ((UCHAR)0)
#define SMBIOS_HARDWARE_SECURITY_SETTING_ENABLED         ((UCHAR)1)
#define SMBIOS_HARDWARE_SECURITY_SETTING_NOT_IMPLEMENTED ((UCHAR)2)
#define SMBIOS_HARDWARE_SECURITY_SETTING_UNKNOWN         ((UCHAR)3)

//
// System Power Controls (Type 25)
//

#define SMBIOS_SYSTEM_POWER_CONTROLS_INFORMATION_TYPE ((UCHAR)25)

typedef struct _SMBIOS_SYSTEM_POWER_CONTROLS_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    UCHAR Month;
    UCHAR Day;
    UCHAR Hour;
    UCHAR Minute;
    UCHAR Second;
} SMBIOS_SYSTEM_POWER_CONTROLS_INFORMATION, *PSMBIOS_SYSTEM_POWER_CONTROLS_INFORMATION;

//
// Voltage Probe (Type 26)
//

#define SMBIOS_VOLTAGE_PROBE_INFORMATION_TYPE ((UCHAR)26)

typedef struct _SMBIOS_VOLTAGE_PROBE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    UCHAR Description; // string

    struct
    {
        union
        {
            UCHAR Location : 5; // SMBIOS_VOLTAGE_PROBE_LOCATION_*
            UCHAR Status : 3;   // SMBIOS_PROBE_STATUS_*
        };

        UCHAR Value;
    } LocationAndStatus;

    USHORT MaximumValue;
    USHORT MinimumValue;
    USHORT Resolution;
    USHORT Tolerance;
    USHORT Accuracy;
    ULONG OEMDefined;
    USHORT NominalValue;
} SMBIOS_VOLTAGE_PROBE_INFORMATION, *PSMBIOS_VOLTAGE_PROBE_INFORMATION;

#define SMBIOS_VOLTAGE_PROBE_LOCATION_OTHER                    ((UCHAR)1)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_UNKNOWN                  ((UCHAR)2)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_PROCESSOR                ((UCHAR)3)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_DISK                     ((UCHAR)4)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_PERIPHERAL_BAY           ((UCHAR)5)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_SYSTEM_MANAGEMENT_MODULE ((UCHAR)6)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_MOTHERBOARD              ((UCHAR)7)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_MEMORY_MODULE            ((UCHAR)8)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_PROCESSOR_MODULE         ((UCHAR)9)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_POWER_UNIT               ((UCHAR)10)
#define SMBIOS_VOLTAGE_PROBE_LOCATION_ADD_IN_CARD              ((UCHAR)11)

//
// Cooling Device (Type 27)
//

#define SMBIOS_COOLING_DEVICE_INFORMATION_TYPE ((UCHAR)27)

typedef struct _SMBIOS_COOLING_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    USHORT TemperatureProbeHandle;

    struct
    {
        union
        {
            UCHAR DeviceType : 5; // SMBIOS_COOLING_DEVICE_TYPE_*
            UCHAR Status : 3;     // SMBIOS_COOLING_DEVICE_STATUS_*
        };

        UCHAR Value;
    } DeviceTypeAndStatus;

    UCHAR CoolingUnitGroup;
    ULONG OEMDefined;
    USHORT NominalSpeed;
    // 2.7+
    UCHAR Description; // string
} SMBIOS_COOLING_DEVICE_INFORMATION, *PSMBIOS_COOLING_DEVICE_INFORMATION;

#define SMBIOS_COOLING_DEVICE_TYPE_OTHER                    ((UCHAR)1)
#define SMBIOS_COOLING_DEVICE_TYPE_UNKNOWN                  ((UCHAR)2)
#define SMBIOS_COOLING_DEVICE_TYPE_FAN                      ((UCHAR)3)
#define SMBIOS_COOLING_DEVICE_TYPE_CENTRIFUGAL_BLOWER       ((UCHAR)4)
#define SMBIOS_COOLING_DEVICE_TYPE_CHIP_FAN                 ((UCHAR)5)
#define SMBIOS_COOLING_DEVICE_TYPE_CABINET_FAN              ((UCHAR)6)
#define SMBIOS_COOLING_DEVICE_TYPE_POWER_SUPPLY_FAN         ((UCHAR)7)
#define SMBIOS_COOLING_DEVICE_TYPE_HEAT_PIPE                ((UCHAR)8)
#define SMBIOS_COOLING_DEVICE_TYPE_INTEGRATED_REFRIGERATION ((UCHAR)9)
#define SMBIOS_COOLING_DEVICE_TYPE_ACTIVE_COOLING           ((UCHAR)10)
#define SMBIOS_COOLING_DEVICE_TYPE_PASSIVE_COOLING          ((UCHAR)11)

#define SMBIOS_COOLING_DEVICE_STATUS_OTHER                  ((UCHAR)1)
#define SMBIOS_COOLING_DEVICE_STATUS_UNKNOWN                ((UCHAR)2)
#define SMBIOS_COOLING_DEVICE_STATUS_OK                     ((UCHAR)3)
#define SMBIOS_COOLING_DEVICE_STATUS_NON_CRITICAL           ((UCHAR)4)
#define SMBIOS_COOLING_DEVICE_STATUS_CRITICAL               ((UCHAR)5)
#define SMBIOS_COOLING_DEVICE_STATUS_NON_RECOVERABLE        ((UCHAR)6)

//
// Temperature Probe (Type 28)
//

#define SMBIOS_TEMPERATURE_PROBE_INFORMATION_TYPE ((UCHAR)28)

typedef struct _SMBIOS_TEMPERATURE_PROBE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    UCHAR Description; // string

    struct
    {
        union
        {
            UCHAR Location : 5; // SMBIOS_TEMPERATURE_PROBE_LOCATION_*
            UCHAR Status : 3;   // SMBIOS_PROBE_STATUS_*
        };

        UCHAR Value;
    } LocationAndStatus;

    USHORT MaximumValue;
    USHORT MinimumValue;
    USHORT Resolution;
    USHORT Tolerance;
    USHORT Accuracy;
    ULONG OEMDefined;
    USHORT NominalValue;
} SMBIOS_TEMPERATURE_PROBE_INFORMATION, *PSMBIOS_TEMPERATURE_PROBE_INFORMATION;

#define SMBIOS_TEMPERATURE_PROBE_LOCATION_OTHER                    ((UCHAR)1)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_UNKNOWN                  ((UCHAR)2)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_PROCESSOR                ((UCHAR)3)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_DISK                     ((UCHAR)4)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_PERIPHERAL_BAY           ((UCHAR)5)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_SYSTEM_MANAGEMENT_MODULE ((UCHAR)6)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_MOTHERBOARD              ((UCHAR)7)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_MEMORY_MODULE            ((UCHAR)8)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_PROCESSOR_MODULE         ((UCHAR)9)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_POWER_UNIT               ((UCHAR)10)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_ADD_IN_CARD              ((UCHAR)11)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_FRONT_PANEL_BOARD        ((UCHAR)12)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_BACK_PANEL_BOARD         ((UCHAR)13)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_POWER_SYSTEM_BOARD       ((UCHAR)14)
#define SMBIOS_TEMPERATURE_PROBE_LOCATION_DRIVE_BACK_PLANE         ((UCHAR)15)

//
// Electrical Current Probe (Type 29)
//

#define SMBIOS_ELECTRICAL_CURRENT_PROBE_INFORMATION_TYPE ((UCHAR)29)

typedef struct _SMBIOS_ELECTRICAL_CURRENT_PROBE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    UCHAR Description; // string

    struct
    {
        union
        {
            UCHAR Location : 5; // SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_*
            UCHAR Status : 3;   // SMBIOS_PROBE_STATUS_*
        };

        UCHAR Value;
    } LocationAndStatus;

    USHORT MaximumValue;
    USHORT MinimumValue;
    USHORT Resolution;
    USHORT Tolerance;
    USHORT Accuracy;
    ULONG OEMDefined;
    USHORT NominalValue;
} SMBIOS_ELECTRICAL_CURRENT_PROBE_INFORMATION, *PSMBIOS_ELECTRICAL_CURRENT_PROBE_INFORMATION;

#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_OTHER                    ((UCHAR)1)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_UNKNOWN                  ((UCHAR)2)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_PROCESSOR                ((UCHAR)3)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_DISK                     ((UCHAR)4)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_PERIPHERAL_BAY           ((UCHAR)5)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_SYSTEM_MANAGEMENT_MODULE          ((UCHAR)6)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_MOTHERBOARD              ((UCHAR)7)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_MEMORY_MODULE            ((UCHAR)8)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_PROCESSOR_MODULE         ((UCHAR)9)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_POWER_UNIT               ((UCHAR)10)
#define SMBIOS_ELECTRICAL_CURRENT_PROBE_LOCATION_ADD_IN_CARD              ((UCHAR)11)

//
// Out-of-Band Remote Access (Type 30)
//

#define SMBIOS_OUT_OF_BAND_REMOTE_ACCESS_INFORMATION_TYPE ((UCHAR)30)

typedef struct _SMBIOS_OUT_OF_BAND_REMOTE_ACCESS_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.2+
    UCHAR Manufacturer; // string

    struct
    {
        union
        {
            UCHAR InboundEnabled : 1;
            UCHAR OutboundEnabled : 1;
            UCHAR Reserved : 6;
        };

        UCHAR Value;
    } Connections;

} SMBIOS_OUT_OF_BAND_REMOTE_ACCESS_INFORMATION, *PSMBIOS_OUT_OF_BAND_REMOTE_ACCESS_INFORMATION;

//
// Boot Integrity Services (BIS) Entry Point (Type 31)
//

#define SMBIOS_BOOT_INTEGRITY_SERVICES_ENTRY_POINT ((UCHAR)31)

//
// Structure type 31 (decimal) is reserved for use by the Boot Integrity Services (BIS).
// See the Boot Integrity Services API Specification for details.
//

//
// System Boot Information (Type 32)
//

#define SMBIOS_SYSTEM_BOOT_INFORMATION_TYPE ((UCHAR)32)

typedef struct _SMBIOS_SYSTEM_BOOT_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Reserved[6];
    UCHAR BootStatus;  // SMBIOS_SYSTEM_BOOT_STATUS_*
} SMBIOS_SYSTEM_BOOT_INFORMATION, *PSMBIOS_SYSTEM_BOOT_INFORMATION;

#define SMBIOS_SYSTEM_BOOT_STATUS_NO_ERROR                           ((UCHAR)0)
#define SMBIOS_SYSTEM_BOOT_STATUS_NO_BOOTABLE_MEDIA                  ((UCHAR)1)
#define SMBIOS_SYSTEM_BOOT_STATUS_OPERATING_SYSTEM_FAILED_TO_LOAD    ((UCHAR)2)
#define SMBIOS_SYSTEM_BOOT_STATUS_FIRMWARE_DETECTED_HARDWARE_FAILURE ((UCHAR)3)
#define SMBIOS_SYSTEM_BOOT_STATUS_OPERATING_SYSTEM_DETECTED_FAILURE  ((UCHAR)4)
#define SMBIOS_SYSTEM_BOOT_STATUS_USER_REQUESTED_BOOT                ((UCHAR)5)
#define SMBIOS_SYSTEM_BOOT_STATUS_SECURITY_VIOLATION                 ((UCHAR)6)
#define SMBIOS_SYSTEM_BOOT_STATUS_PREVIOUS_REQUESTED_IMAGE           ((UCHAR)7)
#define SMBIOS_SYSTEM_BOOT_STATUS_WATCHDOG_EXPIRED                   ((UCHAR)8)
// 9-127 specification reserved
// 128-191 OEM reserved
// 192-255 product specific

//
// 64-Bit Memory Error Information (Type 33)
//

#define SMBIOS_64_BIT_MEMORY_ERROR_INFORMATION_TYPE ((UCHAR)33)

typedef struct _SMBIOS_64_BIT_MEMORY_ERROR_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Type;        // SMBIOS_MEMORY_ERROR_TYPE_*
    UCHAR Granularity; // SMBIOS_MEMORY_ERROR_GRANULARITY_*
    UCHAR Operation;   // SMBIOS_MEMORY_ERROR_OPERATION_*
    ULONG VendorSyndrome;
    ULONGLONG ArrayErrorAddress;
    ULONGLONG DeviceErrorAddress;
    ULONG ErrorResolution;
} SMBIOS_64_BIT_MEMORY_ERROR_INFORMATION, *PSMBIOS_64_BIT_MEMORY_ERROR_INFORMATION;

//
// Management Device (Type 34)
//

#define SMBIOS_MANAGEMENT_DEVICE_INFORMATION_TYPE ((UCHAR)34)

typedef struct _SMBIOS_MANAGEMENT_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Description; // string
    UCHAR DeviceType;  // SMBIOS_MANAGEMENT_DEVICE_TYPE_*
    ULONG Address;
    UCHAR AddressType; // SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_*
} SMBIOS_MANAGEMENT_DEVICE_INFORMATION, *PSMBIOS_MANAGEMENT_DEVICE_INFORMATION;

#define SMBIOS_MANAGEMENT_DEVICE_TYPE_OTHER           ((UCHAR)1)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_UNKNOWN         ((UCHAR)2)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_LM75            ((UCHAR)3)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_LM78            ((UCHAR)4)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_LM79            ((UCHAR)5)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_LM80            ((UCHAR)6)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_LM81            ((UCHAR)7)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_ADM9240         ((UCHAR)8)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_DS1780          ((UCHAR)9)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_1617            ((UCHAR)10)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_GL518SM         ((UCHAR)11)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_W83781D         ((UCHAR)12)
#define SMBIOS_MANAGEMENT_DEVICE_TYPE_HT82H791        ((UCHAR)13)

#define SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_OTHER   ((UCHAR)1)
#define SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_UNKNOWN ((UCHAR)2)
#define SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_IO_PORT ((UCHAR)3)
#define SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_MEMORY  ((UCHAR)4)
#define SMBIOS_MANAGEMENT_DEVICE_ADDRESS_TYPE_SMBUS   ((UCHAR)5)

//
// Management Device Component (Type 35)
//

#define SMBIOS_MANAGEMENT_DEVICE_COMPONENT_INFORMATION_TYPE ((UCHAR)35)

typedef struct _SMBIOS_MANAGEMENT_DEVICE_COMPONENT_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Description; // string
    USHORT ManagementDeviceHandle;
    USHORT ComponentHandle;
    USHORT ThresholdHandle;
} SMBIOS_MANAGEMENT_DEVICE_COMPONENT_INFORMATION, *PSMBIOS_MANAGEMENT_DEVICE_COMPONENT_INFORMATION;

//
// Management Device Threshold Data (Type 36)
//

#define SMBIOS_MANAGEMENT_DEVICE_THRESHOLD_INFORMATION_TYPE ((UCHAR)36)

typedef struct _SMBIOS_MANAGEMENT_DEVICE_THRESHOLD_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    USHORT LowerThresholdNonCritical;
    USHORT UpperThresholdNonCritical;
    USHORT LowerThresholdCritical;
    USHORT UpperThresholdCritical;
    USHORT LowerThresholdNonRecoverable;
    USHORT UpperThresholdNonRecoverable;
} SMBIOS_MANAGEMENT_DEVICE_THRESHOLD_INFORMATION, *PSMBIOS_MANAGEMENT_DEVICE_THRESHOLD_INFORMATION;

//
// Memory Channel (Type 37)
//

#define SMBIOS_MEMORY_CHANNEL_INFORMATION_TYPE ((UCHAR)37)

typedef struct _SMBIOS_MEMORY_CHANNEL_ENTRY
{
    UCHAR Load;
    USHORT Handle;
} SMBIOS_MEMORY_CHANNEL_ENTRY, *PSMBIOS_MEMORY_CHANNEL_ENTRY;

typedef struct _SMBIOS_MEMORY_CHANNEL_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Type;         // SMBIOS_MEMORY_CHANNEL_TYPE_*
    USHORT MaximumLoad;
    UCHAR Count;
    SMBIOS_MEMORY_CHANNEL_ENTRY Entries[ANYSIZE_ARRAY];
} SMBIOS_MEMORY_CHANNEL_INFORMATION, *PSMBIOS_MEMORY_CHANNEL_INFORMATION;

#define SMBIOS_MEMORY_CHANNEL_TYPE_OTHER     ((UCHAR)1)
#define SMBIOS_MEMORY_CHANNEL_TYPE_UNKNOWN   ((UCHAR)2)
#define SMBIOS_MEMORY_CHANNEL_TYPE_RAMBUS    ((UCHAR)3)
#define SMBIOS_MEMORY_CHANNEL_TYPE_SYNC_LINK ((UCHAR)4)

//
// IPMI Device Information (Type 38)
//

#define SMBIOS_IPMI_DEVICE_INFORMATION_TYPE ((UCHAR)38)

typedef struct _SMBIOS_IPMI_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3+
    UCHAR Type; // SMBIOS_IPMI_INTERFACE_TYPE_*
    UCHAR SpecificationRevision;
    UCHAR I2CTargetAddress;
    UCHAR NVStorageDeviceAddress;
    ULONGLONG BaseAddress;

    struct
    {
        union
        {
            UCHAR InterruptTriggerMode : 1; // 1 = level, 0 = edge
            UCHAR InterruptPolarity : 1;    // 1 = active high, 0 = active low
            UCHAR Reserved : 1;
            UCHAR InterruptInfo : 1;        // 1 = specified, 0 = not specified
            UCHAR LSBAddress : 1;           // address bit 0 value
            UCHAR Reserved2 : 1;
            UCHAR RegisterSpacing : 2;      // SMBIOS_IPMI_REGISTER_SPACING_*
        };

        UCHAR Value;
    } Info;

    UCHAR InterruptNumber;
} SMBIOS_IPMI_DEVICE_INFORMATION, *PSMBIOS_IPMI_DEVICE_INFORMATION;

#define SMBIOS_IPMI_INTERFACE_TYPE_UNKONWN      ((UCHAR)0)
#define SMBIOS_IPMI_INTERFACE_TYPE_KCS          ((UCHAR)1)
#define SMBIOS_IPMI_INTERFACE_TYPE_SMIC         ((UCHAR)2)
#define SMBIOS_IPMI_INTERFACE_TYPE_BT           ((UCHAR)3)
#define SMBIOS_IPMI_INTERFACE_TYPE_SSIF         ((UCHAR)4)

#define SMBIOS_IPMI_REGISTER_SPACING_SUCCESSIVE ((UCHAR)0)
#define SMBIOS_IPMI_REGISTER_SPACING_32_BIT     ((UCHAR)1)
#define SMBIOS_IPMI_REGISTER_SPACING_16_BIT     ((UCHAR)2)
#define SMBIOS_IPMI_REGISTER_SPACING_RESERVED   ((UCHAR)3)

//
// System Power Supply (Type 39)
//

#define SMBIOS_SYSTEM_POWER_SUPPLY_INFORMATION_TYPE ((UCHAR)39)

typedef struct _SMBIOS_SYSTEM_POWER_SUPPLY_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.3.1+
    UCHAR PowerUnitGroup;
    UCHAR Location;          // string
    UCHAR DeviceName;        // string
    UCHAR Manufacturer;      // string
    UCHAR SerialNumber;      // string
    UCHAR AssetTag;          // string
    UCHAR ModelPartNumber;   // string
    UCHAR Revision;          // string
    USHORT MaxPowerCapacity;

    struct
    {
        union
        {
            USHORT HostSwappable : 1;
            USHORT Present : 1;
            USHORT Unplugged : 1;
            USHORT RangeSwitching : 4; // SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_*
            USHORT Status : 3;         // SMBIOS_POWER_SUPPLY_STATUS_*
            USHORT Type : 4;           // SMBIOS_POWER_SUPPLY_TYPE_*
            USHORT Reserved : 2;
        };

        USHORT Value;
    } Characteristics;

    USHORT InputVoltageProbeHandle;
    USHORT CoolingDeviceHandle;
    USHORT InputCurrentProbeHandle;
} SMBIOS_SYSTEM_POWER_SUPPLY_INFORMATION, *PSMBIOS_SYSTEM_POWER_SUPPLY_INFORMATION;

#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_OTHER          ((USHORT)1)
#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_UNKNOWN        ((USHORT)2)
#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_MANUAL         ((USHORT)3)
#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_AUTO_SWITCH    ((USHORT)4)
#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_WIDE_RANGE     ((USHORT)5)
#define SMBIOS_POWER_SUPPLY_RANGE_SWITCHING_NOT_APPLICABLE ((USHORT)6)

#define SMBIOS_POWER_SUPPLE_STATUS_OTHER                   ((USHORT)1)
#define SMBIOS_POWER_SUPPLE_STATUS_UNKNOWN                 ((USHORT)2)
#define SMBIOS_POWER_SUPPLE_STATUS_OK                      ((USHORT)3)
#define SMBIOS_POWER_SUPPLE_STATUS_NON_CRITICAL            ((USHORT)4)
#define SMBIOS_POWER_SUPPLE_STATUS_CRITICAL                ((USHORT)5)

#define SMBIOS_POWER_SUPPLY_TYPE_OTHER                      ((USHORT)1)
#define SMBIOS_POWER_SUPPLY_TYPE_UNKNOWN                    ((USHORT)2)
#define SMBIOS_POWER_SUPPLY_TYPE_LINEAR                     ((USHORT)3)
#define SMBIOS_POWER_SUPPLY_TYPE_SWITCHING                  ((USHORT)4)
#define SMBIOS_POWER_SUPPLY_TYPE_BATTERY                    ((USHORT)5)
#define SMBIOS_POWER_SUPPLY_TYPE_UPS                        ((USHORT)6)
#define SMBIOS_POWER_SUPPLY_TYPE_CONVERTER                  ((USHORT)7)
#define SMBIOS_POWER_SUPPLY_TYPE_REGULATOR                  ((USHORT)8)

//
// Additional Information (Type 40)
//

#define SMBIOS_ADDITIONAL_INFORMATION_TYPE ((UCHAR)40)

typedef struct _SMBIOS_ADDITIONAL_ENTRY
{
    UCHAR Length;
    USHORT Handle;
    UCHAR Offset;
    UCHAR String; // string
    UCHAR Value[ANYSIZE_ARRAY];
} SMBIOS_ADDITIONAL_ENTRY, *PSMBIOS_ADDITIONAL_ENTRY;

typedef struct _SMBIOS_ADDITIONAL_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.6+
    UCHAR Count;
    SMBIOS_ADDITIONAL_ENTRY Entries[ANYSIZE_ARRAY];
} SMBIOS_ADDITIONAL_INFORMATION, *PSMBIOS_ADDITIONAL_INFORMATION;

//
// Onboard Devices Extended Information (Type 41)
//

#define SMBIOS_ONBOARD_DEVICE_INFORMATION_TYPE ((UCHAR)41)

typedef struct _SMBIOS_ONBOARD_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.6+
    UCHAR Designation; // string

    struct
    {
        union
        {
            UCHAR Type : 7; // SMBIOS_ONBOARD_DEVICE_TYPE_*
            UCHAR Enabled : 1;
        };

        UCHAR Value;
    } DeviceType;

    UCHAR DeviceTypeInstance;
    USHORT SegmentGroupNumber;
    UCHAR BusNumber;

    struct
    {
        union
        {
            UCHAR FunctionNumber : 3;
            UCHAR DeviceNumber : 5;
        };

        UCHAR Value;
    } DeviceFunctionNumber;

} SMBIOS_ONBOARD_DEVICE_INFORMATION, *PSMBIOS_ONBOARD_DEVICE_INFORMATION;

#define SMBIOS_ONBOARD_DEVICE_TYPE_OTHER        ((UCHAR)1)
#define SMBIOS_ONBOARD_DEVICE_TYPE_UNKNOWN      ((UCHAR)2)
#define SMBIOS_ONBOARD_DEVICE_TYPE_VIDEO        ((UCHAR)3)
#define SMBIOS_ONBOARD_DEVICE_TYPE_SCSI         ((UCHAR)4)
#define SMBIOS_ONBOARD_DEVICE_TYPE_ETHERNET     ((UCHAR)5)
#define SMBIOS_ONBOARD_DEVICE_TYPE_TOKEN_RING   ((UCHAR)6)
#define SMBIOS_ONBOARD_DEVICE_TYPE_SOUND        ((UCHAR)7)
#define SMBIOS_ONBOARD_DEVICE_TYPE_PATA         ((UCHAR)8)
#define SMBIOS_ONBOARD_DEVICE_TYPE_SATA         ((UCHAR)9)
#define SMBIOS_ONBOARD_DEVICE_TYPE_SAS          ((UCHAR)10)
#define SMBIOS_ONBOARD_DEVICE_TYPE_WIRELESS_LAN ((UCHAR)11)
#define SMBIOS_ONBOARD_DEVICE_TYPE_BLUETOOTH    ((UCHAR)12)
#define SMBIOS_ONBOARD_DEVICE_TYPE_WWAN         ((UCHAR)13)
#define SMBIOS_ONBOARD_DEVICE_TYPE_EMMC         ((UCHAR)14)
#define SMBIOS_ONBOARD_DEIVCE_TYPE_NVME         ((UCHAR)15)
#define SMBIOS_ONBOARD_DEVICE_TYPE_UFS          ((UCHAR)16)

//
// Management Controller Host Interface (Type 42)
//

#define SMBIOS_MCHI_INFORMATION_TYPE ((UCHAR)42)

typedef struct _SMBIOS_MCHI_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.6+
    UCHAR Type;   // SMBIOS_MCHI_TYPE_*
    UCHAR Length;
    UCHAR Data[ANYSIZE_ARRAY];
    // SMBIOS_MCHI_PROTOCOL_RECORDS
} SMBIOS_MCHI_INFORMATION, *PSMBIOS_MCHI_INFORMATION;

typedef struct _SMBIOS_MCHI_PROTOCOL_RECORD
{
    UCHAR Type;   // SMBIOS_MCHI_PROTOCOL_TYPE_*
    UCHAR Length;
    UCHAR Data[ANYSIZE_ARRAY];
} SMBIOS_MCHI_PROTOCOL_RECORD, *PSMBIOS_MCHI_PROTOCOL_RECORD;

typedef struct _SMBIOS_MCHI_PROTOCOL_RECORDS
{
    UCHAR Count;
    SMBIOS_MCHI_PROTOCOL_RECORD Records[ANYSIZE_ARRAY];
} SMBIOS_MCHI_PROTOCOL_RECORDS, *PSMBIOS_MCHI_PROTOCOL_RECORDS;

// 0-63 MCTP host interfaces
#define SMBIOS_MCHI_TYPE_NETWORK_INTERFACE ((UCHAR)64)
#define SMBIOS_MCHI_TYPE_OEM_DEFINED       ((UCHAR)240)

#define SMBIOS_MCHI_PROTOCOL_TYPE_RESERVED_0      ((UCHAR)0)
#define SMBIOS_MCHI_PROTOCOL_TYPE_RESERVED_1      ((UCHAR)1)
#define SMBIOS_MCHI_PROTOCOL_TYPE_IPMI            ((UCHAR)2)
#define SMBIOS_MCHI_PROTOCOL_TYPE_MCTP            ((UCHAR)3)
#define SMBIOS_MCHI_PROTOCOL_TYPE_REFRESH_OVER_IP ((UCHAR)4)
#define SMBIOS_MCHI_PROTOCOL_TYPE_OEM_DEFINED     ((UCHAR)240)

//
// TPM Device (Type 43)
//

#define SMBIOS_TPM_DEVICE_INFORMATION_TYPE ((UCHAR)43)

typedef struct _SMBIOS_TPM_DEVICE_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.6+
    UCHAR VentorID[4];
    UCHAR MajorSpecVersion;
    UCHAR MinorSpecVersion;
    ULONG FirmwareVersion1;
    ULONG FirmwareVersion2;
    UCHAR Description;         // string
    ULONGLONG Characteristics; // SMBIOS_TPM_DEVICE_*
    ULONG OEMDefined;
} SMBIOS_TPM_DEVICE_INFORMATION, *PSMBIOS_TPM_DEVICE_INFORMATION;

#define SMBIOS_TPM_DEVICE_RESERVED_0                          (0x0000000000000001ULL)
#define SMBIOS_TPM_DEVICE_RESERVED_1                          (0x0000000000000002ULL)
#define SMBIOS_TPM_DEVICE_CONFIGURABLE_VIA_FIRMWARE_UPDATE    (0x0000000000000004ULL)
#define SMBIOS_TPM_DEVICE_CONFIGURABLE_VIA_SOFTWARE_UPDATE    (0x0000000000000008ULL)
#define SMBIOS_TPM_DEVICE_CONFIGURABLE_VIA_PROPRIETARY_UPDATE (0x0000000000000010ULL)
#define SMBIOS_TPM_DEIVCE_RESERVED                            (0xFFFFFFFFFFFFFFE0ULL)

//
// Processor Additional Information (Type 44)
//

#define SMBIOS_PROCESSOR_ADDITIONAL_INFORMATION_TYPE ((UCHAR)44)

typedef struct _SMBIOS_PROCESSOR_SPECIFIC_BLOCK
{
    UCHAR Length;
    UCHAR Type;   // SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_*
    UCHAR DAta[ANYSIZE_ARRAY];
} SMBIOS_PROCESSOR_SPECIFIC_BLOCK, *PSMBIOS_PROCESSOR_SPECIFIC_BLOCK;

typedef struct _SMBIOS_PROCESSOR_ADDITIONAL_INFORMATION
{
    SMBIOS_HEADER Header;
    // 2.6+
    USHORT Handle;
    SMBIOS_PROCESSOR_SPECIFIC_BLOCK Blocks[ANYSIZE_ARRAY];
} SMBIOS_PROCESSOR_ADDITIONAL_INFORMATION, *PSMBIOS_PROCESSOR_ADDITIONAL_INFORMATION;

#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_IA32        ((UCHAR)1)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_X64         ((UCHAR)2)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_IA64        ((UCHAR)3)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_ARM32       ((UCHAR)4)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_ARM64       ((UCHAR)5)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_RISCV32     ((UCHAR)6)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_RISCV64     ((UCHAR)7)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_RISCV128    ((UCHAR)8)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_LOONGARCH32 ((UCHAR)9)
#define SMBIOS_PROCESSOR_ARCHITECTURE_TYPE_LOONGARCH64 ((UCHAR)10)

//
// Firmware Inventory Information (Type 45)
//

#define SMBIOS_FIRMWARE_INVENTORY_INFORMATION_TYPE ((UCHAR)45)

typedef struct _SMBIOS_FIRMWARE_INVENTORY_INFORMATION
{
    SMBIOS_HEADER Header;
    // 3.5+
    UCHAR ComponentName;          // string
    UCHAR Version;                // string
    UCHAR Format;
    UCHAR Identifier;             // string
    UCHAR IdentifierFormat;
    UCHAR ReleaseDate;            // string
    UCHAR Manufacturer;           // string
    UCHAR LowestSupportedVersion; // string
    ULONGLONG Characteristics;    // SMBIOS_FIRMWARE_INVENTORY_FLAG_*
    UCHAR State;                  // SMBIOS_FIRMWARE_INVENTORY_STATE_*
    UCHAR AssociatedComponents;
    USHORT AssociatedComponentHandles[ANYSIZE_ARRAY];
} SMBIOS_FIRMWARE_INVENTORY_INFORMATION, *PSMBIOS_FIRMWARE_INVENTORY_INFORMATION;

#define SMBIOS_FIRMWARE_INVENTORY_FLAG_UPDATABLE        (0x0000000000000001ULL)
#define SMBIOS_FIRMWARE_INVENTORY_FLAG_WRITE_PROTECTED  (0x0000000000000002ULL)
#define SMBIOS_FIRMWARE_INVENTORY_FLAG_RESERVED         (0xFFFFFFFFFFFFFFFCULL)

#define SMBIOS_FIRMWARE_INVENTORY_STATE_OTHER           ((UCHAR)1)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_UNKNOWN         ((UCHAR)2)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_DISABLED        ((UCHAR)3)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_ENABLED         ((UCHAR)4)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_ABSENT          ((UCHAR)5)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_STANDBY_OFFLINE ((UCHAR)6)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_STANDBY_SPARE   ((UCHAR)7)
#define SMBIOS_FIRMWARE_INVENTORY_STATE_OFFLINE         ((UCHAR)8)

//
// String Property (Type 46)
//

#define SMBIOS_STRING_PROPERTY_TYPE ((UCHAR)46)

typedef struct _SMBIOS_STRING_PROPERTY
{
    SMBIOS_HEADER Header;
    // 3.5+
    USHORT Identifier; // SMBIOS_STRING_PROPERTY_ID_*
    UCHAR Value;       // string
    USHORT Handle;
} SMBIOS_STRING_PROPERTY, *PSMBIOS_STRING_PROPERTY;

#define SMBIOS_STRING_PROPERTY_ID_RESERVED          ((UCHAR)1)
#define SMBIOS_STRING_PROPERTY_ID_UEIF_DEVICE_PATH  ((UCHAR)2)

//
// Inactive (Type 126)
//

#define SMBIOS_INACTIVE_TYPE     ((UCHAR)126)

typedef struct _SMBIOS_INACTIVE
{
    SMBIOS_HEADER Header;
} SMBIOS_INACTIVE, *PSMBIOS_INACTIVE;

//
// End-of-Table (Type 127)
//

#define SMBIOS_END_OF_TABLE_TYPE ((UCHAR)127)

typedef struct _SMBIOS_END_OF_TABLE
{
    SMBIOS_HEADER Header;
} SMBIOS_END_OF_TABLE, *PSMBIOS_END_OF_TABLE;

#include <poppack.h>

#endif
