﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web;
using System.Xml.Linq;
using Certify.Models;
using Certify.Models.Config;
using Certify.Models.Plugins;
using Certify.Models.Providers;
using Certify.Plugins;

// ReSharper disable once CheckNamespace
namespace Certify.Providers.DNS.NameCheap
{
    public class DnsProviderNameCheapProvider : PluginProviderBase<IDnsProvider, ChallengeProviderDefinition>, IDnsProviderProviderPlugin { }

    public class DnsProviderNameCheap : IDnsProvider
    {
        public DnsProviderNameCheap()
        {
        }

        private string _apiUser;
        private string _apiKey;
        private string _ip;

        private HttpClient _http;

        private ILog _log;

        #region Definition

        private const string PARAM_API_USER = "apiuser";
        private const string PARAM_API_KEY = "apikey";
        private const string PARAM_IP = "ip";

        private const string API_URL = "https://api.namecheap.com/xml.response";
        private const int BATCH_SIZE = 99;

        private static XNamespace _ns;

        static DnsProviderNameCheap()
        {
            Definition = new ChallengeProviderDefinition
            {
                Id = "DNS01.API.NameCheap",
                Title = "NameCheap DNS API (Deprecated - Use Posh-ACME version instead)",
                Description = "Validates via NameCheap APIs. This provider is deprecated and you should switch to the Posh-ACME version.",
                HelpUrl = "https://www.namecheap.com/support/api/intro/",
                PropagationDelaySeconds = 120,

                ProviderParameters = new List<ProviderParameter>
                {
                    new ProviderParameter { Key = PARAM_API_USER, Name = "API User", IsRequired = true, IsPassword = false },
                    new ProviderParameter { Key = PARAM_API_KEY, Name = "API Key", IsRequired = true, IsPassword = true },
                    new ProviderParameter { Key = PARAM_IP, Name = "Your IP", Description = "IP Address of the server that sends requests to NameCheap API", IsRequired = true, IsPassword = false },
                    new ProviderParameter { Key= "propagationdelay", Name="Propagation Delay Seconds", IsRequired=false, IsPassword=false, Value="120", IsCredential=false }
                },
                ChallengeType = SupportedChallengeTypes.CHALLENGE_TYPE_DNS,
                Config = "Provider=Certify.Providers.DNS.NameCheap",
                HandlerType = ChallengeHandlerType.INTERNAL
            };

            _ns = XNamespace.Get("http://api.namecheap.com/xml.response");
        }

        /// <summary>
        /// The definition properties for NameCheap.
        /// </summary>
        public static ChallengeProviderDefinition Definition { get; }

        private int? _customPropagationDelay = null;
        public int PropagationDelaySeconds => (_customPropagationDelay != null ? (int)_customPropagationDelay : Definition.PropagationDelaySeconds);
        public string ProviderId => Definition.Id;
        public string ProviderTitle => Definition.Title;
        public string ProviderDescription => Definition.Description;
        public string ProviderHelpUrl => Definition.HelpUrl;
        public bool IsTestModeSupported => Definition.IsTestModeSupported;
        public List<ProviderParameter> ProviderParameters => Definition.ProviderParameters;

        #endregion

        #region IDnsProvider method implementation

        /// <summary>
        /// Initializes the provider.
        /// </summary>
        public Task<bool> InitProvider(Dictionary<string, string> credentials, Dictionary<string, string> parameters, ILog log = null)
        {
            _log = log;

            _apiUser = credentials[PARAM_API_USER];
            _apiKey = credentials[PARAM_API_KEY];
            _ip = credentials[PARAM_IP];

            _http = new HttpClient();

            if (parameters?.ContainsKey("propagationdelay") == true)
            {
                if (int.TryParse(parameters["propagationdelay"], out var customPropDelay))
                {
                    _customPropagationDelay = customPropDelay;
                }
            }

            return Task.FromResult(true);
        }

        /// <summary>
        /// Tests connection to the service.
        /// </summary>
        public async Task<ActionResult> Test()
        {
            try
            {
                var zones = await GetZonesBatchAsync(1);
                if (zones?.Any() == true)
                {
                    return new ActionResult { IsSuccess = true, Message = "Test completed." };
                }

                return new ActionResult { IsSuccess = true, Message = "Test completed, but no zones returned." };
            }
            catch (Exception ex)
            {
                return new ActionResult { IsSuccess = false, Message = "Test failed: " + ex.Message };
            }
        }

        /// <summary>
        /// Adds a DNS record to the list.
        /// </summary>
        public async Task<ActionResult> CreateRecord(DnsRecord request)
        {
            try
            {
                var domain = await ParseDomainAsync(request.TargetDomainName);
                var hosts = await GetHosts(domain);

                var recordName = request.RecordName.Replace(domain.Mask, "");
                hosts.Add(new NameCheapHostRecord
                {
                    Name = recordName,
                    Address = request.RecordValue,
                    Type = request.RecordType,
                    Ttl = 60
                });

                await SetHosts(domain, hosts);

                return new ActionResult
                {
                    IsSuccess = true,
                    Message = $"DNS record added: {request.RecordName}"
                };
            }
            catch (Exception ex)
            {
                return new ActionResult
                {
                    IsSuccess = false,
                    Message = $"Failed to create DNS record {request.RecordName}: {ex.Message}"
                };
            }
        }

        /// <summary>
        /// Removes a DNS record from the list.
        /// </summary>
        public async Task<ActionResult> DeleteRecord(DnsRecord request)
        {
            try
            {
                var domain = await ParseDomainAsync(request.TargetDomainName);
                var hosts = await GetHosts(domain);
                var recordName = request.RecordName.Replace(domain.Mask, "");
                var toRemove = hosts.FirstOrDefault(x => x.Name == recordName
                                                         && x.Type == request.RecordType
                                                         && x.Address == request.RecordValue);
                if (toRemove == null)
                {
                    return new ActionResult
                    {
                        IsSuccess = true,
                        Message = $"DNS record {request.RecordName} does not exist, nothing to remove."
                    };
                }

                hosts.Remove(toRemove);
                await SetHosts(domain, hosts);

                return new ActionResult
                {
                    IsSuccess = true,
                    Message = $"DNS record removed: {request.RecordName}"
                };
            }
            catch (Exception ex)
            {
                return new ActionResult
                {
                    IsSuccess = false,
                    Message = $"Failed to remove DNS record {request.RecordName}: {ex.Message}"
                };
            }
        }

        /// <summary>
        /// Returns the list of available zones.
        /// </summary>
        public async Task<List<DnsZone>> GetZones()
        {
            var result = new List<DnsZone>();
            var page = 1;

            while (true)
            {
                var zoneBatch = await GetZonesBatchAsync(page);
                result.AddRange(zoneBatch);
                page++;

                if (zoneBatch.Count < BATCH_SIZE)
                {
                    break;
                }

                await Task.Delay(1000); // wait 1 second before querying next batch in case there is a rate limit
            }

            return result;
        }

        #endregion

        #region Private helpers

        /// <summary>
        /// Returns a batch of zones
        /// </summary>
        private async Task<IReadOnlyList<DnsZone>> GetZonesBatchAsync(int page)
        {
            try
            {
                var xmlResponse = await InvokeGetApiAsync("domains.getList", new Dictionary<string, string>
                {
                    ["Page"] = page.ToString(),
                    ["PageSize"] = BATCH_SIZE.ToString(),
                    ["SortBy"] = "NAME"
                });

                return xmlResponse.Element(_ns + "CommandResponse")
                                  .Element(_ns + "DomainGetListResult")
                                  .Elements(_ns + "Domain")
                                  .Where(x => x.Attr<bool>("IsExpired") == false
                                              && x.Attr<bool>("IsLocked") == false
                                              && x.Attr<bool>("IsOurDNS") == true)
                                  .Select(x => x.Attr<string>("Name"))
                                  .Select(x => new DnsZone
                                  {
                                      Name = x,
                                      ZoneId = x
                                  })
                                  .ToList();
            }
            catch (Exception exp)
            {
                _log.Error(exp, "Failed to get a batch of domain zones.");

                return new DnsZone[0];
            }
        }

        /// <summary>
        /// Returns the list of hosts for a domain.
        /// </summary>
        private async Task<List<NameCheapHostRecord>> GetHosts(ParsedDomain domain)
        {
            var xml = await InvokeGetApiAsync("domains.dns.getHosts", new Dictionary<string, string>
            {
                ["SLD"] = domain.SLD,
                ["TLD"] = domain.TLD
            });

            return xml.Element(_ns + "CommandResponse")
                      .Descendants(_ns + "host")
                      .Select(x => new NameCheapHostRecord
                      {
                          Address = x.Attr<string>("Address"),
                          Name = x.Attr<string>("Name"),
                          Type = x.Attr<string>("Type"),
                          HostId = x.Attr<int>("HostId"),
                          MxPref = x.Attr<int>("MXPref"),
                          Ttl = x.Attr<int>("TTL"),
                      })
                      .ToList();
        }

        /// <summary>
        /// Updates the list of hosts in the domain.
        /// </summary>
        private async Task SetHosts(ParsedDomain domain, IReadOnlyList<NameCheapHostRecord> hosts)
        {
            var args = WithDefaultArgs("domains.dns.setHosts", new Dictionary<string, string>
            {
                ["SLD"] = domain.SLD,
                ["TLD"] = domain.TLD
            });

            var idx = 1;
            foreach (var host in hosts)
            {
                args["HostName" + idx] = host.Name;
                args["RecordType" + idx] = host.Type;
                args["Address" + idx] = host.Address;
                args["MXPref" + idx] = host.MxPref.ToString();
                args["TTL" + idx] = host.Ttl.ToString();

                idx++;
            }

            var request = new HttpRequestMessage
            {
                Method = HttpMethod.Post,
                RequestUri = new Uri(API_URL),
                Content = new FormUrlEncodedContent(args)
            };

            await InvokeApiAsync(request);
        }

        #endregion

        #region API invocation

        /// <summary>
        /// Invokes the API method via a GET request, returning the XML results.
        /// </summary>
        private async Task<XElement> InvokeGetApiAsync(string command, Dictionary<string, string> args = null)
        {
            string Encode(string arg) => HttpUtility.UrlEncode(arg);

            args = WithDefaultArgs(command, args);
            var url = API_URL + "?" + string.Join("&", args.Select(kvp => Encode(kvp.Key) + "=" + Encode(kvp.Value)));
            var request = new HttpRequestMessage(HttpMethod.Get, url);

            return await InvokeApiAsync(request);
        }

        /// <summary>
        /// Invokes the API method.
        /// </summary>
        private async Task<XElement> InvokeApiAsync(HttpRequestMessage request)
        {
            var response = await _http.SendAsync(request);
            var content = await response.Content.ReadAsStringAsync();

            if (!response.IsSuccessStatusCode)
            {
                throw new Exception($"NameCheap API method {request.RequestUri} returned HTTP Code {response.StatusCode}");
            }

            XElement xml;
            try
            {
                xml = XElement.Parse(content);
            }
            catch (Exception ex)
            {
                throw new Exception($"NameCheap API method {request.RequestUri} returned invalid XML response:\n{content}", ex);
            }

            if (xml.Attribute("Status")?.Value.ToLower() != "ok")
            {
                throw new Exception($"NameCheap API method {request.RequestUri} returned an error status '{xml.Attribute("Status")?.Value}':\n{content}");
            }

            return xml;
        }

        /// <summary>
        /// Populates the argument list with default args.
        /// </summary>
        private Dictionary<string, string> WithDefaultArgs(string command, Dictionary<string, string> args = null)
        {
            if (args == null)
            {
                args = new Dictionary<string, string>();
            }

            args.Add("ApiUser", _apiUser);
            args.Add("ApiKey", _apiKey);
            args.Add("UserName", _apiUser);
            args.Add("Command", "namecheap." + command);
            args.Add("ClientIp", _ip);

            return args;
        }

        /// <summary>
        /// Returns the parsed domain name from forms like "blabla.com" and "*.blabla.com".
        /// </summary>
        private async Task<ParsedDomain> ParseDomainAsync(string domain)
        {
            if (string.IsNullOrEmpty(domain))
            {
                throw new ArgumentException("Domain was not specified!");
            }

            var knownZones = await GetZones();
            var matchingZone = knownZones.Select(x => x.ZoneId)
                                         .FirstOrDefault(x => domain.EndsWith(x, StringComparison.InvariantCultureIgnoreCase));

            if (matchingZone == null)
            {
                throw new ArgumentException($"Domain {domain} is not managed by this account!");
            }

            // for "example.co.uk": SLD = "example", TLD = "co.uk"
            var dotPos = matchingZone.IndexOf('.');
            return new ParsedDomain
            {
                SLD = matchingZone.Substring(0, dotPos),
                TLD = matchingZone.Substring(dotPos + 1)
            };
        }

        private class ParsedDomain
        {
            public string TLD;
            public string SLD;

            public string Mask => $".{SLD}.{TLD}";
        }

        #endregion
    }
}
