<?php

declare(strict_types=1);

namespace Zolinga\System\Events;

use Zolinga\System\Types\{StatusEnum, SeverityEnum, OriginEnum};
use JsonSerializable;

/**
 * System event class.
 * 
 * If you want to implement stoppable events then extend the class and use StoppableInterface and StoppableTrait.
 * 
 * @property-read StatusEnum $status status code like OK, ERROR, NOT_FOUND, UNAUTHORIZED, FORBIDDEN, BAD_REQUEST, UNDETERMINED
 * @property-read string $message human readable status message
 * @property-read string $statusName status code name like "UNDETERMINED" or "FORBIDDEN" or "OK"
 * @property-read string $statusNiceName human readable status name like "Undetermined" or "Forbidden" or "Not Found" or "OK"
 * @property-read boolean $ok is the status OK? Same value as returned by $event->isOk()
 */
class Event implements JsonSerializable
{
    /**
     * Unique Event identifier. It is generated by the coonstructor
     * but can be overriden if needed. Can be used by gates 
     * between Javascript and PHP and for other purposes.
     *
     * @var string
     */
    public $uuid;

    /**
     * Event type in the format of an URI. Example: example.org:api:myEvent
     *
     * @var string
     */
    public readonly string $type;

    /**
     * Origin of the event. Can be internal, remote or cli.
     * Note it is an enum and not a string so to access the value use $event->origin->value.
     *
     * @var OriginEnum
     */
    public readonly OriginEnum $origin;

    /**
     * Time to process this event.
     * 
     * @var float
     */
    public float $totalTime = 0;

    /**
     * Status of the event.
     * 
     * Note it is an enum and not a string so to access the value use $event->status->value.
     * 
     * This property is publicly readable by the magic method __get() 
     * but it is writeable only by the setStatus() method.   
     *
     * @var StatusEnum 
     */
    private StatusEnum $status = StatusEnum::UNDETERMINED;

    /**
     * Status message. It is a human readable message that describes the status of the event.
     * 
     * This property is publicly readable by the magic method __get() 
     * but it is writeable only by the setStatus() method.   
     *
     * @var ?string
     */
    private ?string $message = null;

    // Just shortcuts to the enums. It may be easier to use $event::STATUS_OK instead of importing enums... 
    final const STATUS_ERROR = StatusEnum::ERROR;
    final const STATUS_NOT_FOUND = StatusEnum::NOT_FOUND;
    final const STATUS_OK = StatusEnum::OK;

    final const STATUS_MULTIPLE_CHOICES = StatusEnum::MULTIPLE_CHOICES;
    final const STATUS_MOVED_PERMANENTLY = StatusEnum::MOVED_PERMANENTLY;
    final const STATUS_FOUND = StatusEnum::FOUND;
    final const STATUS_SEE_OTHER = StatusEnum::SEE_OTHER;
    final const STATUS_NOT_MODIFIED = StatusEnum::NOT_MODIFIED;
    final const STATUS_TEMPORARY_REDIRECT = StatusEnum::TEMPORARY_REDIRECT;
    final const STATUS_PERMANENT_REDIRECT = StatusEnum::PERMANENT_REDIRECT;

    final const STATUS_UNAUTHORIZED = StatusEnum::UNAUTHORIZED;
    final const STATUS_FORBIDDEN = StatusEnum::FORBIDDEN;
    final const STATUS_BAD_REQUEST = StatusEnum::BAD_REQUEST;
    final const STATUS_UNDETERMINED = StatusEnum::UNDETERMINED;
    final const STATUS_CONTINUE = StatusEnum::CONTINUE;
    final const STATUS_PROCESSING = StatusEnum::PROCESSING;
    final const STATUS_TIMEOUT = StatusEnum::TIMEOUT;
    final const STATUS_CONFLICT = StatusEnum::CONFLICT;
    final const STATUS_PRECONDITION_FAILED = StatusEnum::PRECONDITION_FAILED;
    final const STATUS_I_AM_A_TEAPOT = StatusEnum::I_AM_A_TEAPOT;
    final const STATUS_LOCKED = StatusEnum::LOCKED;


    final const ORIGIN_INTERNAL = OriginEnum::INTERNAL;
    final const ORIGIN_REMOTE = OriginEnum::REMOTE;
    final const ORIGIN_CLI = OriginEnum::CLI;
    final const ORIGIN_ANY = OriginEnum::ANY;

    public function __construct(string $type, OriginEnum $origin)
    {
        $this->type = $type;
        $this->origin = $origin;
        $this->uuid = uniqid();
    }

    public function __get(string $name): mixed
    {
        switch ($name) {
            case 'status':
            case 'message':
                return $this->$name;

            case 'statusName':
                return $this->status->name;

            case 'statusNiceName':
                return $this->status->getFriendlyName();

            case 'ok':
                return $this->isOk();
        
            default:
                throw new \Exception("Property $name does not exist or is inaccessible on " . self::class);
        }
    }

    public function __set(string $name, mixed $value)
    {
        switch ($name) {
            case 'status':
            case 'message':
            case 'statusName':
                throw new \Exception("Property $name is read-only on " . self::class . ". Use setStatus() method instead.");

            default:
                throw new \Exception("Property $name does not exist or is inaccessible on " . self::class);
        }
    }

    /**
     * Shortcut to calling $api->dispatchEvent($event)
     *
     * @return self
     */
    public function dispatch(): self
    {
        global $api;
        $api->dispatchEvent($this);
        return $this;
    }

    /**
     * Set the status and status message of the event.
     * 
     * @param StatusEnum $status
     * @param string $message
     * @return StatusEnum current status. May not be the same as $status in case there is already a error status set.
     */
    final public function setStatus(StatusEnum $status, string $message): StatusEnum
    {
        // Priority of status codes is:
        // 1. OK status has higest priority then error statuses
        // 2. Lower status codes have higher priority than higher status codes
        //
        // This can be expressed simply by comparing the status values as OK < 400 <= ERROR
        //
        // Reasoning: Event can have multiple listeners as fail-overs. The event will
        // be considered successful if at least one listener succeeds. If all listeners fail
        // then the event will be considered failed.
        //
        // Example: 
        // If we request resource from 3 different sources and 2 of them return 404
        // and one returns 200 then the event is considered successful.
        //
        // If we check rights if one listener provider authorizes the request 
        // then the request is authorized.
        //
        // To solve complex cases we may need to add third "important" parameter to the setStatus() method.
        if ($this->isUndetermined() || $status->value < $this->status->value) {
            $this->status = $status;
            $this->message = $message;
        } 
        return $this->status;
    }

    /**
     * Does the event originate from the trusted origin "internal" or "cli"?
     *
     * @return boolean
     */
    final public function isTrusted(): bool
    {
        return $this->origin == self::ORIGIN_INTERNAL || $this->origin == self::ORIGIN_CLI;
    }

    /**
     * Does the Event has the status < 400?
     *
     * @return boolean
     */
    final public function isOk(): bool
    {
        return $this->status->isOk();
    }

    /**
     * Does the Event has the status >= 400?
     *
     * @return boolean
     */
    final public function isError(): bool
    {
        return $this->status->isError();
    }

    /**
     * Does the Event has the status UNDETERMINED?
     *
     * @return boolean
     */
    final public function isUndetermined(): bool
    {
        return $this->status->isUndetermined();
    }

    public function __toString(): string
    {
        $tags = [];
        $tags[] = basename(str_replace("\\", "/", $this::class));
        $tags[] = $this->origin->value;
        $tags[] = strtolower($this->status->getFriendlyName())." {$this->status->value}";
        return $this->status->getEmoji() . " Event {$this->type} [" . implode(", ", $tags) . "]";
    }

    /**
     * Specify data which should be serialized to JSON
     *
     * @return array<string, mixed>
     */
    public function jsonSerialize(): mixed
    {
        return [
            "type" => $this->type,
            "origin" => $this->origin->value,
            "status" => $this->status->value,
            "message" => $this->message,
            "totalTime" => round($this->totalTime, 3),
        ];
    }
}
