package org.vitrivr.cottontail.core.queries.predicates

import org.vitrivr.cottontail.core.database.ColumnDef
import org.vitrivr.cottontail.core.queries.binding.Binding
import org.vitrivr.cottontail.core.queries.functions.math.distance.binary.VectorDistance
import org.vitrivr.cottontail.core.queries.planning.cost.Cost
import org.vitrivr.cottontail.core.values.DoubleValue

/**
 * The nearest [ProximityPredicate] is a data structure used to bridge the gap between an expressed,
 * proximity based query (through a function invocation) and an equivalent index lookup
 *
 * @author Ralph Gasser
 * @version 3.0.0
 */
sealed interface ProximityPredicate: Predicate {

    /** The [ColumnDef] this [ProximityPredicate] evaluates. */
    val column: Binding.Column

    /** The [VectorDistance] this [ProximityPredicate] evaluates. */
    val distance: VectorDistance<*>

    /** The [Binding] that represents the query vector. */
    val query: Binding

    /** Columns evaluated by this [ProximityPredicate]. */
    override val columns: Set<Binding.Column>
        get() = setOf(this.column)

    /** The [Cost] required for applying this [ProximityPredicate]. */
    override val cost: Cost
        get() = this.distance.cost

    /** The [ColumnDef] for the distance that is generated by this [ProximityPredicate]. */
    val distanceColumn: Binding.Column

    /**
     * A [ProximityPredicate] that expresses a simple scan over distance values.
     */
    data class Scan(override val column: Binding.Column, override val distanceColumn: Binding.Column, override val distance: VectorDistance<*>, override val query: Binding): ProximityPredicate {
        override fun digest(): Long = 13L * this.hashCode()
    }

    /**
     * A k-limited [ProximityPredicate]
     */
    sealed interface KLimitedSearch: ProximityPredicate {
        val k: Long
    }

    /**
     * A [ProximityPredicate] that expresses a k nearest neighbour search.
     */
    data class NNS(override val column: Binding.Column, override val distanceColumn: Binding.Column, override val k: Long, override val distance: VectorDistance<*>, override val query: Binding): KLimitedSearch {
        override fun digest(): Long = 13L * this.hashCode()
    }

    /**
     * A [ProximityPredicate] that expresses a k farthest neighbour search.
     */
    data class FNS(override val column: Binding.Column, override val distanceColumn: Binding.Column, override val k: Long, override val distance: VectorDistance<*>, override val query: Binding): KLimitedSearch {
        override fun digest(): Long = 7L * this.hashCode()
    }

    /**
     * A [ProximityPredicate] that expresses a range search (εNN).
     */
    data class ENN(override val column: Binding.Column, override val distanceColumn: Binding.Column, val eMin: DoubleValue, val eMax: DoubleValue, override val distance: VectorDistance<*>, override val query: Binding): ProximityPredicate {
        override fun digest(): Long = 7L * this.hashCode()
    }
}