package org.vitrivr.cottontail.core.queries.functions

import org.vitrivr.cottontail.core.queries.functions.exception.FunctionNotFoundException

/**
 * A [FunctionRegistry] manages all the [Function] instances generated and used by Cottontail DB. It can be used  to
 * lookup and obtain instances of those [Function]s.
 *
 * @author Ralph Gasser
 * @version 1.1.0
 */
class FunctionRegistry {
    /**
     * The internal data structure to organize [FunctionGenerator]s.
     *
     * Required for [Function] resolution where different versions of the same [Function] exist
     * and can be generated by a [FunctionGenerator]
     */
    private val generators = mutableMapOf<Signature.Open, FunctionGenerator<*>>()

    /**
     * The internal data structure to organize [FunctionGenerator]s.
     *
     * Required for [Function] resolution where only as single version of a [Function] exists,
     * which can be looked up directly.
     */
    private val functions = mutableMapOf<Signature.SemiClosed, Function<*>>()

    /**
     * Registers a new [Function] with this [FunctionRegistry].
     *
     * @param function The [Function] to register.
     * @throws IllegalStateException If a [Function] or a [FunctionGenerator] with a colliding [Signature] has been registered.
     */
    fun register(function: Function<*>) {
        val semiClosed = function.signature.toSemiClosed()
        val open = semiClosed.toOpen()
        check(!this.functions.containsKey(semiClosed)) { "Function for signature ${function.signature} collides with existing function." }
        check(!this.generators.containsKey(open)) { "Function for signature ${function.signature} collides with existing generator." }
        this.functions[semiClosed] = function
    }

    /**
     * Registers a new [FunctionGenerator] with this [FunctionRegistry].
     *
     * @param generator The [FunctionGenerator] to register.
     * @throws IllegalStateException If a [Function] or a [FunctionGenerator] with a colliding [Signature] has been registered.
     */
    fun register(generator: FunctionGenerator<*>) {
        check(!this.generators.containsKey(generator.signature)) { "Function generator for signature ${generator.signature} collides with existing generator." }
        check(!this.functions.keys.any { generator.signature.collides(it) }) { "Function generator for signature ${generator.signature} collides with existing generator." }
        this.generators[generator.signature] = generator
    }

    /**
     * Tries to resolve a [Signature.Open] and returns a list of [Signature.Closed] for this function.
     *
     * @param signature The [Signature.Open] to resolve a [Signature.Closed] for.
     * @return List of [Signature.Closed] that match the [Signature.Open].
     */
    fun resolve(signature: Signature.Open): List<Signature.Closed<*>> {
        val generator = this.generators[signature] ?: throw FunctionNotFoundException(signature)
        return generator.resolve(signature)
    }

    /**
     * Tries to resolve a [Signature.SemiClosed] and returns the [Signature.Closed] for this function.
     *
     * @param signature The [Signature.SemiClosed] to resolve a [Signature.Closed] for.
     * @return List of [Signature.Closed] that match the [Signature.SemiClosed].
     */
    fun resolve(signature: Signature.SemiClosed): Signature.Closed<*> {
        val function = this.functions[signature]
        if (function != null) return function.signature
        val generator = this.generators[signature.toOpen()] ?: throw FunctionNotFoundException(signature)
        return generator.resolve(signature)
    }

    /**
     * Generates a [Function] instance for the given [Signature.Closed].
     *
     * @param signature The[Signature.Closed] to generate a [Function] for.
     * @return The generated [Function]
     */
    fun obtain(signature: Signature.SemiClosed): Function<*> {
        val function = this.functions[signature]
        if (function != null) return function
        val generator = this.generators[signature.toOpen()] ?: throw FunctionNotFoundException(signature)
        return generator.obtain(signature)
    }

    /**
     * Generates a [Function] instance for the given [Signature.Closed].
     *
     * @param signature The[Signature.Closed] to generate a [Function] for.
     * @return The generated [Function]
     */
    fun obtain(signature: Signature.Closed<*>): Function<*> = obtain(signature.toSemiClosed())
}