"""AIOHTTP startup functions for the task spawner and runner services."""

import asyncio

from aiohttp.web_app import Application
from aiojobs.aiohttp import get_scheduler_from_app
from virtool_core.redis import Redis

from virtool.config import get_config_from_app
from virtool.hmm.tasks import HMMRefreshTask
from virtool.jobs.tasks import TimeoutJobsTask
from virtool.ml.tasks import SyncMLModelsTask
from virtool.pg.utils import connect_pg
from virtool.references.tasks import CleanReferencesTask, RefreshReferenceReleasesTask
from virtool.samples.tasks import (
    UpdateSampleWorkflowsTask,
)
from virtool.tasks.client import TasksClient
from virtool.tasks.data import TasksData
from virtool.tasks.spawner import TaskSpawnerService


async def startup_databases_for_spawner(app: Application):
    """Creates Redis and Postgres connections

    :param app: the app object

    """
    config = get_config_from_app(app)

    redis = Redis(config.redis_connection_string)

    pg, _ = await asyncio.gather(
        connect_pg(config.postgres_connection_string),
        redis.connect(),
    )

    app.update({"pg": pg, "redis": redis})


async def startup_datalayer_for_spawner(app: Application):
    """Creates the tasks datalayer and adds it to the app.

    :param app: the :class:`aiohttp.web.Application` object
    """
    app["tasks_datalayer"] = TasksData(app["pg"], TasksClient(app["redis"]))


async def startup_task_spawner(app: Application):
    """Starts the task spawner."""
    tasks = [
        (CleanReferencesTask, 3600),
        (HMMRefreshTask, 600),
        (RefreshReferenceReleasesTask, 600),
        (SyncMLModelsTask, 600),
        (TimeoutJobsTask, 3600),
        (UpdateSampleWorkflowsTask, 3600),
    ]

    await get_scheduler_from_app(app).spawn(
        TaskSpawnerService(app["pg"], app["tasks_datalayer"]).run(tasks),
    )
