from typing import Any

from pydantic import BaseModel, Field, conlist, constr
from virtool_core.models.analysis import AnalysisMinimal
from virtool_core.models.enums import AnalysisWorkflow, LibraryType
from virtool_core.models.samples import Sample, SampleMinimal
from virtool_core.models.validators import prevent_none


class GetSamplesResponse(SampleMinimal):
    class Config:
        schema_extra = {
            "example": [
                {
                    "created_at": "2022-05-20T23:48:00.901000Z",
                    "host": "Malus domestica",
                    "id": "9zn468u9",
                    "isolate": "",
                    "labels": [],
                    "library_type": "normal",
                    "name": "HX8",
                    "notes": "",
                    "nuvs": False,
                    "pathoscope": True,
                    "ready": True,
                    "subtractions": ["0nhpi36p"],
                    "user": {
                        "administrator": True,
                        "handle": "mrott",
                        "id": "ihvze2u9",
                    },
                    "workflows": {
                        "aodp": "incompatible",
                        "nuvs": "none",
                        "pathoscope": "none",
                    },
                },
            ],
        }


class GetSampleResponse(Sample):
    class Config:
        schema_extra = {
            "example": {
                "all_read": False,
                "all_write": False,
                "artifacts": [],
                "created_at": "2022-05-20T23:48:00.901000Z",
                "format": "fastq",
                "group": {"id": 4, "name": "Sidney"},
                "group_read": True,
                "group_write": True,
                "hold": True,
                "host": "Malus domestica",
                "id": "9zn468u9",
                "is_legacy": False,
                "isolate": "",
                "labels": [],
                "library_type": "normal",
                "locale": "",
                "name": "HX8",
                "notes": "",
                "nuvs": False,
                "paired": True,
                "pathoscope": True,
                "quality": {
                    "bases": [
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                    ],
                    "composition": [
                        [29.0, 18.0, 15.0, 36.5],
                        [25.5, 19.0, 31.5, 22.0],
                    ],
                    "count": 94601674,
                    "encoding": "Sanger / Illumina 1.9\n",
                    "gc": 43.0,
                    "length": [150, 150],
                    "sequences": [
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        18,
                        298,
                    ],
                },
                "reads": [
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_1.fq.gz",
                        "id": 713,
                        "name": "reads_1.fq.gz",
                        "name_on_disk": "reads_1.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3540467819,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:10:27.418000Z",
                    },
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_2.fq.gz",
                        "id": 714,
                        "name": "reads_2.fq.gz",
                        "name_on_disk": "reads_2.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3321721014,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:11:10.743000Z",
                    },
                ],
                "ready": True,
                "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                "user": {"administrator": True, "handle": "mrott", "id": "ihvze2u9"},
                "workflows": {
                    "aodp": "incompatible",
                    "nuvs": "none",
                    "pathoscope": "none",
                },
            },
        }


class CreateSampleRequest(BaseModel):
    name: constr(strip_whitespace=True, min_length=1)
    host: constr(strip_whitespace=True) = ""
    isolate: constr(strip_whitespace=True) = ""
    group: int | None = None
    locale: constr(strip_whitespace=True) = ""
    library_type: LibraryType = LibraryType.normal
    subtractions: list = Field(default_factory=list)
    files: conlist(item_type=Any, min_items=1, max_items=2)
    notes: str = ""
    labels: list = Field(default_factory=list)


class CreateSampleResponse(Sample):
    class Config:
        schema_extra = {
            "example": {
                "all_read": False,
                "all_write": False,
                "artifacts": [],
                "created_at": "2022-05-20T23:48:00.901000Z",
                "format": "fastq",
                "group": {"id": 4, "name": "Sidney"},
                "group_read": True,
                "group_write": True,
                "hold": True,
                "host": "Malus domestica",
                "id": "9zn468u9",
                "is_legacy": False,
                "isolate": "",
                "labels": [],
                "library_type": "normal",
                "locale": "",
                "name": "HX8",
                "notes": "",
                "nuvs": False,
                "paired": True,
                "pathoscope": True,
                "quality": {
                    "bases": [
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                    ],
                    "composition": [
                        [29.0, 18.0, 15.0, 36.5],
                        [25.5, 19.0, 31.5, 22.0],
                    ],
                    "count": 94601674,
                    "encoding": "Sanger / Illumina 1.9\n",
                    "gc": 43.0,
                    "length": [150, 150],
                    "sequences": [
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        18,
                        298,
                    ],
                },
                "reads": [
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_1.fq.gz",
                        "id": 713,
                        "name": "reads_1.fq.gz",
                        "name_on_disk": "reads_1.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3540467819,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:10:27.418000Z",
                    },
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_2.fq.gz",
                        "id": 714,
                        "name": "reads_2.fq.gz",
                        "name_on_disk": "reads_2.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3321721014,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:11:10.743000Z",
                    },
                ],
                "ready": True,
                "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                "user": {"administrator": True, "handle": "mrott", "id": "ihvze2u9"},
                "workflows": {
                    "aodp": "incompatible",
                    "nuvs": "none",
                    "pathoscope": "none",
                },
            },
        }


class UpdateSampleRequest(BaseModel):
    name: constr(strip_whitespace=True, min_length=1) | None
    host: constr(strip_whitespace=True) | None
    isolate: constr(strip_whitespace=True) | None
    locale: constr(strip_whitespace=True) | None
    notes: constr(strip_whitespace=True) | None
    labels: list | None
    subtractions: list | None

    _prevent_none = prevent_none("*")

    class Config:
        schema_extra = {
            "example": {
                "name": "Tobacco mosaic viru",
                "host": "Tobacco",
                "labels": [1, 5, 6],
            },
        }


class UpdateSampleResponse(Sample):
    class Config:
        schema_extra = {
            "example": {
                "all_read": False,
                "all_write": False,
                "artifacts": [],
                "created_at": "2022-05-20T23:48:00.901000Z",
                "format": "fastq",
                "group": {"id": 4, "name": "Sidney"},
                "group_read": True,
                "group_write": True,
                "hold": True,
                "host": "virus",
                "id": "9zn468u9",
                "is_legacy": False,
                "isolate": "",
                "labels": [1, 5, 6],
                "library_type": "normal",
                "locale": "",
                "name": "foo",
                "notes": "",
                "nuvs": False,
                "paired": True,
                "pathoscope": True,
                "quality": {
                    "bases": [
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                    ],
                    "composition": [
                        [29.0, 18.0, 15.0, 36.5],
                        [25.5, 19.0, 31.5, 22.0],
                    ],
                    "count": 94601674,
                    "encoding": "Sanger / Illumina 1.9\n",
                    "gc": 43.0,
                    "length": [150, 150],
                    "sequences": [
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        18,
                        298,
                    ],
                },
                "reads": [
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_1.fq.gz",
                        "id": 713,
                        "name": "reads_1.fq.gz",
                        "name_on_disk": "reads_1.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3540467819,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:10:27.418000Z",
                    },
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_2.fq.gz",
                        "id": 714,
                        "name": "reads_2.fq.gz",
                        "name_on_disk": "reads_2.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3321721014,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:11:10.743000Z",
                    },
                ],
                "ready": True,
                "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                "user": {"administrator": True, "handle": "mrott", "id": "ihvze2u9"},
                "workflows": {
                    "aodp": "incompatible",
                    "nuvs": "none",
                    "pathoscope": "none",
                },
            },
        }


class UpdateRightsRequest(BaseModel):
    group: int | str | None
    all_read: bool | None
    all_write: bool | None
    group_read: bool | None
    group_write: bool | None

    _prevent_none = prevent_none("*")

    class Config:
        schema_extra = {
            "example": {
                "group": "administrator",
                "group_read": True,
                "group_write": True,
            },
        }


class UpdateRightsResponse(Sample):
    class Config:
        schema_extra = {
            "example": {
                "all_read": False,
                "all_write": False,
                "artifacts": [],
                "created_at": "2022-05-20T23:48:00.901000Z",
                "format": "fastq",
                "group": 4,
                "group_read": True,
                "group_write": True,
                "hold": True,
                "host": "virus",
                "id": "9zn468u9",
                "is_legacy": False,
                "isolate": "",
                "labels": [1, 5, 6],
                "library_type": "normal",
                "locale": "",
                "name": "foo",
                "notes": "",
                "nuvs": False,
                "paired": True,
                "pathoscope": True,
                "quality": {
                    "bases": [
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                        [36.0, 37.0, 37.0, 37.0, 37.0, 37.0],
                    ],
                    "composition": [
                        [29.0, 18.0, 15.0, 36.5],
                        [25.5, 19.0, 31.5, 22.0],
                    ],
                    "count": 94601674,
                    "encoding": "Sanger / Illumina 1.9\n",
                    "gc": 43.0,
                    "length": [150, 150],
                    "sequences": [
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        0,
                        18,
                        298,
                    ],
                },
                "reads": [
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_1.fq.gz",
                        "id": 713,
                        "name": "reads_1.fq.gz",
                        "name_on_disk": "reads_1.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3540467819,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:10:27.418000Z",
                    },
                    {
                        "download_url": "/samples/9zn468u9/reads/reads_2.fq.gz",
                        "id": 714,
                        "name": "reads_2.fq.gz",
                        "name_on_disk": "reads_2.fq.gz",
                        "sample": "9zn468u9",
                        "size": 3321721014,
                        "upload": None,
                        "uploaded_at": "2022-05-21T00:11:10.743000Z",
                    },
                ],
                "ready": True,
                "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                "user": {"administrator": True, "handle": "mrott", "id": "ihvze2u9"},
                "workflows": {
                    "aodp": "incompatible",
                    "nuvs": "none",
                    "pathoscope": "none",
                },
            },
        }


class GetSampleAnalysesResponse(AnalysisMinimal):
    class Config:
        schema_extra = {
            "example": [
                {
                    "created_at": "2022-05-21T01:28:55.441000Z",
                    "id": "m9ktiz0i",
                    "index": {"id": "9c5u6wsq", "version": 13},
                    "job": {"id": "bt8nwg9z"},
                    "ready": True,
                    "reference": {"id": "d19exr83", "name": "New Plant Viruses"},
                    "sample": {"id": "9zn468u9"},
                    "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                    "updated_at": "2022-05-21T01:28:55.441000Z",
                    "user": {
                        "administrator": True,
                        "handle": "mrott",
                        "id": "ihvze2u9",
                    },
                    "workflow": "pathoscope_bowtie",
                },
            ],
        }


class CreateAnalysisRequest(BaseModel):
    ml: int | None
    ref_id: str
    subtractions: list[str] = Field(default_factory=list)
    workflow: AnalysisWorkflow

    _prevent_none = prevent_none("subtractions")


class CreateAnalysisResponse(AnalysisMinimal):
    class Config:
        schema_extra = {
            "example": {
                "created_at": "2022-05-21T01:28:55.441000Z",
                "id": "m9ktiz0i",
                "index": {"id": "9c5u6wsq", "version": 13},
                "job": {"id": "bt8nwg9z"},
                "ready": True,
                "reference": {"id": "d19exr83", "name": "New Plant Viruses"},
                "sample": {"id": "9zn468u9"},
                "subtractions": [{"id": "0nhpi36p", "name": "Malus domestica"}],
                "updated_at": "2022-05-21T01:28:55.441000Z",
                "user": {"administrator": True, "handle": "mrott", "id": "ihvze2u9"},
                "workflow": "pathoscope_bowtie",
            },
        }
