#[derive(Clone, Debug)]
pub enum TagDVB {
    NetworkName,
    ServiceList,
    Stuffing,
    SatelliteDeliverySystem,
    CableDeliverySystem,
    VBIData,
    VBITeletext,
    BouquetName,
    Service,
    CountryAvailability,
    Linkage,
    NVODReference,
    TimeShiftedService,
    ShortEvent,
    ExtendedEvent,
    TimeShiftedEvent,
    Component,
    Mosaic,
    StreamIdentifier,
    CAIdentifier,
    Content,
    ParentalRating,
    Teletext,
    Telephone,
    LocalTimeOffset,
    Subtitling,
    TerrestrialDeliverySystem,
    MultilingualNetworkName,
    MultilingualBouquetName,
    MultilingualServiceName,
    MultilingualComponent,
    PrivateDataSpecifier,
    ServiceMove,
    ShortSmoothingBuffer,
    FrequencyList,
    PartialTransportStream,
    DataBroadcast,
    Scrambling,
    DataBroadcastId,
    TransportStream,
    DSNG,
    PDC,
    AC3,
    AncillaryData,
    CellList,
    CellFrequencyLink,
    AnnouncementSupport,
    ApplicationSignalling,
    AdaptationFieldData,
    ServiceIdentifier,
    ServiceAvailability,
    DefaultAuthority,
    RelatedContent,
    TVAId,
    ContentIdentifier,
    TimeSliceFecIdentifier,
    ECMRepetitionRate,
    S2SatelliteDeliverySystem,
    EnhancedAC3,
    DTSDescriptor,
    AAC,
    XAITLocation,
    FTAContentManagement,
    Extension,
}

#[derive(Clone, Debug)]
pub enum Tag {
    VH2621381811172,
    A1381811172,
    Hierarchy,
    RegistrationPrivate,
    DataStreamAlignment,
    TargetBackgroundGrid,
    VideoWindow,
    CASEMMECMPID,
    ISO639,
    SystemClockExternalReference,
    MultiplexBufferUtilizationBounds,
    CopyrightIdentificationSystemAndReference,
    MaximumBitRate,
    PrivateDataIndicator,
    SmoothingBuffer,
    STDVideoBufferLeakControl,
    IBPVideoIFrameIndicator,
    DSMCCCarouselIdentifier,
    DSMCCAssociationTag,
    DSMCCDeferredAssociationTag,
    DSMCCReserved,
    DSMCCNPTReference,
    DSMCCNPTEndpoint,
    DSMCCStreamMode,
    DSMCCStreamEvent,
    VideoStreamHeaderParametersMPEG4H263Based,
    AudioStreamHeaderParametersMPEG4LOASMultiFormatFramed,
    IODParametersForISOIEC144961,
    SLParametersForISOIEC144961,
    FMCParametersForISOIEC144961,
    ExternalESIdentifierForISOIEC144961,
    MuxCodeForISOIEC144961,
    FMXBufferSizeForISOIEC144961,
    MultiplexBufferForISOIEC144961,
    ContentLabelingForISOIEC144961,
    MetadataPointer,
    Metadata,
    MetadataSTD,
    VideoStreamHeaderParametersForITUTRecH264AndISOIEC1449610,
    ISOIEC1381811IPMPDRM,
    TimingAndHRDForITUTRecH264andISOIEC1449610,
    AudioStreamHeaderParametersForISOIEC138187ADTSAAC,
    FlexMuxTimingForISOIEC144961,
    TextStreamHeaderParametersForISOIEC14496,
    AudioExtensionStreamHeaderParametersForISOIEC144963MPEG4LOASMultiFormatFramed,
    VideoAuxiliaryStreamHeaderParameters,
    VideoScalableStreamHeaderParameters,
    VideoMultiStreamHeaderParameters,
    VideoStreamHeaderParametersForITUTRecT800AndISOIEC15444,
    VideoMultiOperationPointStreamHeaderParameters,
    VideoStereoscopic3DStreamHeaderParametersForITUTRecH262ISOIEC138182AndISOIEC111722,
    ProgramStereoscopic3DInformation,
    VideoStereoscopic3DInformation,

    VideoLANFourCCVideoSizeAndCodecInitializationData,

    // 0x40...0x7F
    DVB(TagDVB),

    // 0x00...0x01
    // 0x37...0x3F
    Reserved(u8),

    // 0x80...0x9F | 0xA1...0xCF
    ATSC(u8),

    // 0xD0...0xDF
    ISDB(u8),

    // 0xE0...0xE9
    CableLabs(u8),

    // 0xEA...0xFE
    Other(u8),

    // 0xFF
    Forbidden,
}

impl Tag {
    #[inline(always)]
    pub fn is_dvb_service(&self) -> bool {
        match self {
            Tag::DVB(TagDVB::Service) => true,
            _ => false,
        }
    }

    #[inline(always)]
    pub fn is_dvb_short_event(&self) -> bool {
        match self {
            Tag::DVB(TagDVB::ShortEvent) => true,
            _ => false,
        }
    }
}

impl From<u8> for Tag {
    fn from(d: u8) -> Self {
        match d {
            0x02 => Tag::VH2621381811172,
            0x03 => Tag::A1381811172,
            0x04 => Tag::Hierarchy,
            0x05 => Tag::RegistrationPrivate,
            0x06 => Tag::DataStreamAlignment,
            0x07 => Tag::TargetBackgroundGrid,
            0x08 => Tag::VideoWindow,
            0x09 => Tag::CASEMMECMPID,
            0x0A => Tag::ISO639,
            0x0B => Tag::SystemClockExternalReference,
            0x0C => Tag::MultiplexBufferUtilizationBounds,
            0x0D => Tag::CopyrightIdentificationSystemAndReference,
            0x0E => Tag::MaximumBitRate,
            0x0F => Tag::PrivateDataIndicator,
            0x10 => Tag::SmoothingBuffer,
            0x11 => Tag::STDVideoBufferLeakControl,
            0x12 => Tag::IBPVideoIFrameIndicator,
            0x13 => Tag::DSMCCCarouselIdentifier,
            0x14 => Tag::DSMCCAssociationTag,
            0x15 => Tag::DSMCCDeferredAssociationTag,
            0x16 => Tag::DSMCCReserved,
            0x17 => Tag::DSMCCNPTReference,
            0x18 => Tag::DSMCCNPTEndpoint,
            0x19 => Tag::DSMCCStreamMode,
            0x1A => Tag::DSMCCStreamEvent,
            0x1B => Tag::VideoStreamHeaderParametersMPEG4H263Based,
            0x1C => Tag::AudioStreamHeaderParametersMPEG4LOASMultiFormatFramed,
            0x1D => Tag::IODParametersForISOIEC144961,
            0x1E => Tag::SLParametersForISOIEC144961,
            0x1F => Tag::FMCParametersForISOIEC144961,
            0x20 => Tag::ExternalESIdentifierForISOIEC144961,
            0x21 => Tag::MuxCodeForISOIEC144961,
            0x22 => Tag::FMXBufferSizeForISOIEC144961,
            0x23 => Tag::MultiplexBufferForISOIEC144961,
            0x24 => Tag::ContentLabelingForISOIEC144961,
            0x25 => Tag::MetadataPointer,
            0x26 => Tag::Metadata,
            0x27 => Tag::MetadataSTD,
            0x28 => Tag::VideoStreamHeaderParametersForITUTRecH264AndISOIEC1449610,
            0x29 => Tag::ISOIEC1381811IPMPDRM,
            0x2A => Tag::TimingAndHRDForITUTRecH264andISOIEC1449610,
            0x2B => Tag::AudioStreamHeaderParametersForISOIEC138187ADTSAAC,
            0x2C => Tag::FlexMuxTimingForISOIEC144961,
            0x2D => Tag::TextStreamHeaderParametersForISOIEC14496,
            0x2E => Tag::AudioExtensionStreamHeaderParametersForISOIEC144963MPEG4LOASMultiFormatFramed,
            0x2F => Tag::VideoAuxiliaryStreamHeaderParameters,
            0x30 => Tag::VideoScalableStreamHeaderParameters,
            0x31 => Tag::VideoMultiStreamHeaderParameters,
            0x32 => Tag::VideoStreamHeaderParametersForITUTRecT800AndISOIEC15444,
            0x33 => Tag::VideoMultiOperationPointStreamHeaderParameters,
            0x34 => Tag::VideoStereoscopic3DStreamHeaderParametersForITUTRecH262ISOIEC138182AndISOIEC111722,
            0x35 => Tag::ProgramStereoscopic3DInformation,
            0x36 => Tag::VideoStereoscopic3DInformation,

            0xA0 => Tag::VideoLANFourCCVideoSizeAndCodecInitializationData,

            0x40 => Tag::DVB(TagDVB::NetworkName),
            0x41 => Tag::DVB(TagDVB::ServiceList),
            0x42 => Tag::DVB(TagDVB::Stuffing),
            0x43 => Tag::DVB(TagDVB::SatelliteDeliverySystem),
            0x44 => Tag::DVB(TagDVB::CableDeliverySystem),
            0x45 => Tag::DVB(TagDVB::VBIData),
            0x46 => Tag::DVB(TagDVB::VBITeletext),
            0x47 => Tag::DVB(TagDVB::BouquetName),
            0x48 => Tag::DVB(TagDVB::Service),
            0x49 => Tag::DVB(TagDVB::CountryAvailability),
            0x4A => Tag::DVB(TagDVB::Linkage),
            0x4B => Tag::DVB(TagDVB::NVODReference),
            0x4C => Tag::DVB(TagDVB::TimeShiftedService),
            0x4D => Tag::DVB(TagDVB::ShortEvent),
            0x4E => Tag::DVB(TagDVB::ExtendedEvent),
            0x4F => Tag::DVB(TagDVB::TimeShiftedEvent),
            0x50 => Tag::DVB(TagDVB::Component),
            0x51 => Tag::DVB(TagDVB::Mosaic),
            0x52 => Tag::DVB(TagDVB::StreamIdentifier),
            0x53 => Tag::DVB(TagDVB::CAIdentifier),
            0x54 => Tag::DVB(TagDVB::Content),
            0x55 => Tag::DVB(TagDVB::ParentalRating),
            0x56 => Tag::DVB(TagDVB::Teletext),
            0x57 => Tag::DVB(TagDVB::Telephone),
            0x58 => Tag::DVB(TagDVB::LocalTimeOffset),
            0x59 => Tag::DVB(TagDVB::Subtitling),
            0x5A => Tag::DVB(TagDVB::TerrestrialDeliverySystem),
            0x5B => Tag::DVB(TagDVB::MultilingualNetworkName),
            0x5C => Tag::DVB(TagDVB::MultilingualBouquetName),
            0x5D => Tag::DVB(TagDVB::MultilingualServiceName),
            0x5E => Tag::DVB(TagDVB::MultilingualComponent),
            0x5F => Tag::DVB(TagDVB::PrivateDataSpecifier),
            0x60 => Tag::DVB(TagDVB::ServiceMove),
            0x61 => Tag::DVB(TagDVB::ShortSmoothingBuffer),
            0x62 => Tag::DVB(TagDVB::FrequencyList),
            0x63 => Tag::DVB(TagDVB::PartialTransportStream),
            0x64 => Tag::DVB(TagDVB::DataBroadcast),
            0x65 => Tag::DVB(TagDVB::Scrambling),
            0x66 => Tag::DVB(TagDVB::DataBroadcastId),
            0x67 => Tag::DVB(TagDVB::TransportStream),
            0x68 => Tag::DVB(TagDVB::DSNG),
            0x69 => Tag::DVB(TagDVB::PDC),
            0x6A => Tag::DVB(TagDVB::AC3),
            0x6B => Tag::DVB(TagDVB::AncillaryData),
            0x6C => Tag::DVB(TagDVB::CellList),
            0x6D => Tag::DVB(TagDVB::CellFrequencyLink),
            0x6E => Tag::DVB(TagDVB::AnnouncementSupport),
            0x6F => Tag::DVB(TagDVB::ApplicationSignalling),
            0x70 => Tag::DVB(TagDVB::AdaptationFieldData),
            0x71 => Tag::DVB(TagDVB::ServiceIdentifier),
            0x72 => Tag::DVB(TagDVB::ServiceAvailability),
            0x73 => Tag::DVB(TagDVB::DefaultAuthority),
            0x74 => Tag::DVB(TagDVB::RelatedContent),
            0x75 => Tag::DVB(TagDVB::TVAId),
            0x76 => Tag::DVB(TagDVB::ContentIdentifier),
            0x77 => Tag::DVB(TagDVB::TimeSliceFecIdentifier),
            0x78 => Tag::DVB(TagDVB::ECMRepetitionRate),
            0x79 => Tag::DVB(TagDVB::S2SatelliteDeliverySystem),
            0x7A => Tag::DVB(TagDVB::EnhancedAC3),
            0x7B => Tag::DVB(TagDVB::DTSDescriptor),
            0x7C => Tag::DVB(TagDVB::AAC),
            0x7D => Tag::DVB(TagDVB::XAITLocation),
            0x7E => Tag::DVB(TagDVB::FTAContentManagement),
            0x7F => Tag::DVB(TagDVB::Extension),

            0x00..=0x01 | 0x37..=0x3F => Tag::Reserved(d),
            0x80..=0x9F | 0xA1..=0xCF => Tag::ATSC(d),

            0xD0..=0xDF => Tag::ISDB(d),

            0xE0..=0xE9 => Tag::CableLabs(d),

            0xEA..=0xFE => Tag::Other(d),

            0xFF => Tag::Forbidden,
        }
    }
}

impl From<Tag> for u8 {
    fn from(dt: Tag) -> u8 {
        match dt {
            Tag::VH2621381811172 => 0x02,
            Tag::A1381811172 => 0x03,
            Tag::Hierarchy => 0x04,
            Tag::RegistrationPrivate => 0x05,
            Tag::DataStreamAlignment => 0x06,
            Tag::TargetBackgroundGrid => 0x07,
            Tag::VideoWindow => 0x08,
            Tag::CASEMMECMPID => 0x09,
            Tag::ISO639 => 0x0A,
            Tag::SystemClockExternalReference => 0x0B,
            Tag::MultiplexBufferUtilizationBounds => 0x0C,
            Tag::CopyrightIdentificationSystemAndReference => 0x0D,
            Tag::MaximumBitRate => 0x0E,
            Tag::PrivateDataIndicator => 0x0F,
            Tag::SmoothingBuffer => 0x10,
            Tag::STDVideoBufferLeakControl => 0x11,
            Tag::IBPVideoIFrameIndicator => 0x12,
            Tag::DSMCCCarouselIdentifier => 0x13,
            Tag::DSMCCAssociationTag => 0x14,
            Tag::DSMCCDeferredAssociationTag => 0x15,
            Tag::DSMCCReserved => 0x16,
            Tag::DSMCCNPTReference => 0x17,
            Tag::DSMCCNPTEndpoint => 0x18,
            Tag::DSMCCStreamMode => 0x19,
            Tag::DSMCCStreamEvent => 0x1A,
            Tag::VideoStreamHeaderParametersMPEG4H263Based => 0x1B,
            Tag::AudioStreamHeaderParametersMPEG4LOASMultiFormatFramed => 0x1C,
            Tag::IODParametersForISOIEC144961 => 0x1D,
            Tag::SLParametersForISOIEC144961 => 0x1E,
            Tag::FMCParametersForISOIEC144961 => 0x1F,
            Tag::ExternalESIdentifierForISOIEC144961 => 0x20,
            Tag::MuxCodeForISOIEC144961 => 0x21,
            Tag::FMXBufferSizeForISOIEC144961 => 0x22,
            Tag::MultiplexBufferForISOIEC144961 => 0x23,
            Tag::ContentLabelingForISOIEC144961 => 0x24,
            Tag::MetadataPointer => 0x25,
            Tag::Metadata => 0x26,
            Tag::MetadataSTD => 0x27,
            Tag::VideoStreamHeaderParametersForITUTRecH264AndISOIEC1449610 => 0x28,
            Tag::ISOIEC1381811IPMPDRM => 0x29,
            Tag::TimingAndHRDForITUTRecH264andISOIEC1449610 => 0x2A,
            Tag::AudioStreamHeaderParametersForISOIEC138187ADTSAAC => 0x2B,
            Tag::FlexMuxTimingForISOIEC144961 => 0x2C,
            Tag::TextStreamHeaderParametersForISOIEC14496 => 0x2D,
            Tag::AudioExtensionStreamHeaderParametersForISOIEC144963MPEG4LOASMultiFormatFramed => 0x2E,
            Tag::VideoAuxiliaryStreamHeaderParameters => 0x2F,
            Tag::VideoScalableStreamHeaderParameters => 0x30,
            Tag::VideoMultiStreamHeaderParameters => 0x31,
            Tag::VideoStreamHeaderParametersForITUTRecT800AndISOIEC15444 => 0x32,
            Tag::VideoMultiOperationPointStreamHeaderParameters => 0x33,
            Tag::VideoStereoscopic3DStreamHeaderParametersForITUTRecH262ISOIEC138182AndISOIEC111722 => 0x34,
            Tag::ProgramStereoscopic3DInformation => 0x35,
            Tag::VideoStereoscopic3DInformation => 0x36,

            Tag::VideoLANFourCCVideoSizeAndCodecInitializationData => 0xA0,

            Tag::DVB(TagDVB::NetworkName) => 0x40,
            Tag::DVB(TagDVB::ServiceList) => 0x41,
            Tag::DVB(TagDVB::Stuffing) => 0x42,
            Tag::DVB(TagDVB::SatelliteDeliverySystem) => 0x43,
            Tag::DVB(TagDVB::CableDeliverySystem) => 0x44,
            Tag::DVB(TagDVB::VBIData) => 0x45,
            Tag::DVB(TagDVB::VBITeletext) => 0x46,
            Tag::DVB(TagDVB::BouquetName) => 0x47,
            Tag::DVB(TagDVB::Service) => 0x48,
            Tag::DVB(TagDVB::CountryAvailability) => 0x49,
            Tag::DVB(TagDVB::Linkage) => 0x4A,
            Tag::DVB(TagDVB::NVODReference) => 0x4B,
            Tag::DVB(TagDVB::TimeShiftedService) => 0x4C,
            Tag::DVB(TagDVB::ShortEvent) => 0x4D,
            Tag::DVB(TagDVB::ExtendedEvent) => 0x4E,
            Tag::DVB(TagDVB::TimeShiftedEvent) => 0x4F,
            Tag::DVB(TagDVB::Component) => 0x50,
            Tag::DVB(TagDVB::Mosaic) => 0x51,
            Tag::DVB(TagDVB::StreamIdentifier) => 0x52,
            Tag::DVB(TagDVB::CAIdentifier) => 0x53,
            Tag::DVB(TagDVB::Content) => 0x54,
            Tag::DVB(TagDVB::ParentalRating) => 0x55,
            Tag::DVB(TagDVB::Teletext) => 0x56,
            Tag::DVB(TagDVB::Telephone) => 0x57,
            Tag::DVB(TagDVB::LocalTimeOffset) => 0x58,
            Tag::DVB(TagDVB::Subtitling) => 0x59,
            Tag::DVB(TagDVB::TerrestrialDeliverySystem) => 0x5A,
            Tag::DVB(TagDVB::MultilingualNetworkName) => 0x5B,
            Tag::DVB(TagDVB::MultilingualBouquetName) => 0x5C,
            Tag::DVB(TagDVB::MultilingualServiceName) => 0x5D,
            Tag::DVB(TagDVB::MultilingualComponent) => 0x5E,
            Tag::DVB(TagDVB::PrivateDataSpecifier) => 0x5F,
            Tag::DVB(TagDVB::ServiceMove) => 0x60,
            Tag::DVB(TagDVB::ShortSmoothingBuffer) => 0x61,
            Tag::DVB(TagDVB::FrequencyList) => 0x62,
            Tag::DVB(TagDVB::PartialTransportStream) => 0x63,
            Tag::DVB(TagDVB::DataBroadcast) => 0x64,
            Tag::DVB(TagDVB::Scrambling) => 0x65,
            Tag::DVB(TagDVB::DataBroadcastId) => 0x66,
            Tag::DVB(TagDVB::TransportStream) => 0x67,
            Tag::DVB(TagDVB::DSNG) => 0x68,
            Tag::DVB(TagDVB::PDC) => 0x69,
            Tag::DVB(TagDVB::AC3) => 0x6A,
            Tag::DVB(TagDVB::AncillaryData) => 0x6B,
            Tag::DVB(TagDVB::CellList) => 0x6C,
            Tag::DVB(TagDVB::CellFrequencyLink) => 0x6D,
            Tag::DVB(TagDVB::AnnouncementSupport) => 0x6E,
            Tag::DVB(TagDVB::ApplicationSignalling) => 0x6F,
            Tag::DVB(TagDVB::AdaptationFieldData) => 0x70,
            Tag::DVB(TagDVB::ServiceIdentifier) => 0x71,
            Tag::DVB(TagDVB::ServiceAvailability) => 0x72,
            Tag::DVB(TagDVB::DefaultAuthority) => 0x73,
            Tag::DVB(TagDVB::RelatedContent) => 0x74,
            Tag::DVB(TagDVB::TVAId) => 0x75,
            Tag::DVB(TagDVB::ContentIdentifier) => 0x76,
            Tag::DVB(TagDVB::TimeSliceFecIdentifier) => 0x77,
            Tag::DVB(TagDVB::ECMRepetitionRate) => 0x78,
            Tag::DVB(TagDVB::S2SatelliteDeliverySystem) => 0x79,
            Tag::DVB(TagDVB::EnhancedAC3) => 0x7A,
            Tag::DVB(TagDVB::DTSDescriptor) => 0x7B,
            Tag::DVB(TagDVB::AAC) => 0x7C,
            Tag::DVB(TagDVB::XAITLocation) => 0x7D,
            Tag::DVB(TagDVB::FTAContentManagement) => 0x7E,
            Tag::DVB(TagDVB::Extension) => 0x7F,

            Tag::Reserved(d) => d,
            Tag::ATSC(d) => d,

            Tag::ISDB(d) => d,

            Tag::CableLabs(d) => d,

            Tag::Other(d) => d,

            Tag::Forbidden => 0xFF,
        }
    }
}
