# @vercel/edge

## Table of contents

### Interfaces

- [ExtraResponseInit](interfaces/ExtraResponseInit.md)
- [Geo](interfaces/Geo.md)
- [ModifiedRequest](interfaces/ModifiedRequest.md)
- [RequestContext](interfaces/RequestContext.md)

### Variables

- [CITY_HEADER_NAME](README.md#city_header_name)
- [COUNTRY_HEADER_NAME](README.md#country_header_name)
- [EMOJI_FLAG_UNICODE_STARTING_POSITION](README.md#emoji_flag_unicode_starting_position)
- [IP_HEADER_NAME](README.md#ip_header_name)
- [LATITUDE_HEADER_NAME](README.md#latitude_header_name)
- [LONGITUDE_HEADER_NAME](README.md#longitude_header_name)
- [REGION_HEADER_NAME](README.md#region_header_name)
- [REQUEST_ID_HEADER_NAME](README.md#request_id_header_name)

### Functions

- [geolocation](README.md#geolocation)
- [ipAddress](README.md#ipaddress)
- [json](README.md#json)
- [next](README.md#next)
- [rewrite](README.md#rewrite)

## Variables

### CITY_HEADER_NAME

• `Const` **CITY_HEADER_NAME**: `"x-vercel-ip-city"`

City of the original client IP as calculated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:4](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L4)

---

### COUNTRY_HEADER_NAME

• `Const` **COUNTRY_HEADER_NAME**: `"x-vercel-ip-country"`

Country of the original client IP as calculated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:8](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L8)

---

### EMOJI_FLAG_UNICODE_STARTING_POSITION

• `Const` **EMOJI_FLAG_UNICODE_STARTING_POSITION**: `127397`

Unicode characters for emoji flags start at this number, and run up to 127469.

#### Defined in

[packages/edge/src/edge-headers.ts:34](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L34)

---

### IP_HEADER_NAME

• `Const` **IP_HEADER_NAME**: `"x-real-ip"`

Client IP as calculated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:12](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L12)

---

### LATITUDE_HEADER_NAME

• `Const` **LATITUDE_HEADER_NAME**: `"x-vercel-ip-latitude"`

Latitude of the original client IP as calculated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:16](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L16)

---

### LONGITUDE_HEADER_NAME

• `Const` **LONGITUDE_HEADER_NAME**: `"x-vercel-ip-longitude"`

Longitude of the original client IP as calculated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:20](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L20)

---

### REGION_HEADER_NAME

• `Const` **REGION_HEADER_NAME**: `"x-vercel-ip-country-region"`

Country region of the original client IP calculated by Vercel Proxy.

See [docs](https://vercel.com/docs/concepts/edge-network/headers#x-vercel-ip-country-region).

#### Defined in

[packages/edge/src/edge-headers.ts:26](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L26)

---

### REQUEST_ID_HEADER_NAME

• `Const` **REQUEST_ID_HEADER_NAME**: `"x-vercel-id"`

The request ID for each request generated by Vercel Proxy.

#### Defined in

[packages/edge/src/edge-headers.ts:30](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L30)

## Functions

### geolocation

▸ **geolocation**(`request`): [`Geo`](interfaces/Geo.md)

Returns the location information for the incoming request.

**`See`**

- [CITY_HEADER_NAME](README.md#city_header_name)
- [COUNTRY_HEADER_NAME](README.md#country_header_name)
- [REGION_HEADER_NAME](README.md#region_header_name)
- [LATITUDE_HEADER_NAME](README.md#latitude_header_name)
- [LONGITUDE_HEADER_NAME](README.md#longitude_header_name)

#### Parameters

| Name      | Type      | Description                                                     |
| :-------- | :-------- | :-------------------------------------------------------------- |
| `request` | `Request` | The incoming request object which provides the geolocation data |

#### Returns

[`Geo`](interfaces/Geo.md)

#### Defined in

[packages/edge/src/edge-headers.ts:131](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L131)

---

### ipAddress

▸ **ipAddress**(`request`): `string` \| `undefined`

Returns the IP address of the request from the headers.

**`See`**

[IP_HEADER_NAME](README.md#ip_header_name)

#### Parameters

| Name      | Type      | Description                                       |
| :-------- | :-------- | :------------------------------------------------ |
| `request` | `Request` | The incoming request object which provides the IP |

#### Returns

`string` \| `undefined`

#### Defined in

[packages/edge/src/edge-headers.ts:99](https://github.com/vercel/vercel/blob/main/packages/edge/src/edge-headers.ts#L99)

---

### json

▸ **json**(`data`, `init?`): [`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

Builds a response object from a serializable JavaScript object:

- sets the 'Content-Type' response header to 'application/json'
- sets the response body from provided data

**`See`**

[https://fetch.spec.whatwg.org/#dom-response-json](https://fetch.spec.whatwg.org/#dom-response-json)

**`Example`**

<caption>Building a JSON response</caption>

```ts
import { json } from '@vercel/edge';

const response = json(
  { notification: { success: true, content: 'worked' } },
  { headers: { 'x-custom': '1' } }
);
```

#### Parameters

| Name    | Type           | Description                                             |
| :------ | :------------- | :------------------------------------------------------ |
| `data`  | `any`          | serialized data                                         |
| `init?` | `ResponseInit` | optional custom response status, statusText and headers |

#### Returns

[`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

#### Defined in

[packages/edge/src/response.ts:19](https://github.com/vercel/vercel/blob/main/packages/edge/src/response.ts#L19)

---

### next

▸ **next**(`init?`): [`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

Returns a Response that instructs the system to continue processing the request.

**`Example`**

<caption>No-op middleware</caption>

```ts
import { next } from '@vercel/edge';

export default function middleware(_req: Request) {
  return next();
}
```

**`Example`**

<caption>Add response headers to all requests</caption>

```ts
import { next } from '@vercel/edge';

export default function middleware(_req: Request) {
  return next({
    headers: { 'x-from-middleware': 'true' },
  });
}
```

#### Parameters

| Name    | Type                                                   | Description                         |
| :------ | :----------------------------------------------------- | :---------------------------------- |
| `init?` | [`ExtraResponseInit`](interfaces/ExtraResponseInit.md) | Additional options for the response |

#### Returns

[`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

#### Defined in

[packages/edge/src/middleware-helpers.ts:145](https://github.com/vercel/vercel/blob/main/packages/edge/src/middleware-helpers.ts#L145)

---

### rewrite

▸ **rewrite**(`destination`, `init?`): [`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

Returns a response that rewrites the request to a different URL.

**`Example`**

<caption>Rewrite all feature-flagged requests from `/:path*` to `/experimental/:path*`</caption>

```ts
import { rewrite, next } from '@vercel/edge';

export default async function middleware(req: Request) {
  const flagged = await getFlag(req, 'isExperimental');
  if (flagged) {
    const url = new URL(req.url);
    url.pathname = `/experimental{url.pathname}`;
    return rewrite(url);
  }

  return next();
}
```

**`Example`**

<caption>JWT authentication for `/api/:path*` requests</caption>

```ts
import { rewrite, next } from '@vercel/edge';

export default function middleware(req: Request) {
  const auth = checkJwt(req.headers.get('Authorization'));
  if (!checkJwt) {
    return rewrite(new URL('/api/error-unauthorized', req.url));
  }
  const url = new URL(req.url);
  url.searchParams.set('_userId', auth.userId);
  return rewrite(url);
}

export const config = { matcher: '/api/users/:path*' };
```

#### Parameters

| Name          | Type                                                                      | Description                         |
| :------------ | :------------------------------------------------------------------------ | :---------------------------------- |
| `destination` | `string` \| [`URL`](https://developer.mozilla.org/en-US/docs/Web/API/URL) | new URL to rewrite the request to   |
| `init?`       | [`ExtraResponseInit`](interfaces/ExtraResponseInit.md)                    | Additional options for the response |

#### Returns

[`Response`](https://developer.mozilla.org/en-US/docs/Web/API/Response)

#### Defined in

[packages/edge/src/middleware-helpers.ts:101](https://github.com/vercel/vercel/blob/main/packages/edge/src/middleware-helpers.ts#L101)
