#!/usr/bin/env node
const path = require('path');
const fs = require('fs');
const prettier = require('prettier');
const {
  packageDirectoryNames,
  packagesRoot,
  packageNamespace,
} = require('./packages');

const TSCONFIG_COMMENT = `// Don't edit this file directly. It is generated by /scripts/generate-ts-config.js\n\n`;

const tsConfigBase = {
  extends: '../../tsconfig.json',
  compilerOptions: {
    outDir: './lib',
    rootDir: './lib',
    composite: true,
  },
  include: ['./**/*.ts'],
  exclude: ['./**/*.test.ts', './**/*.story.ts'],
};

const generatePackageTSConfig = () => {
  packageDirectoryNames.forEach(async packageDirectoryName => {
    const packageDirectory = path.join(packagesRoot, packageDirectoryName);
    const packageJSONPath = path.join(packageDirectory, 'package.json');
    const packageJSONExists = fs.existsSync(packageJSONPath);

    if (!packageJSONExists) {
      console.error();
      console.error(`Could not find package.json: ${packageJSONPath}`);
      console.error();
      //process.exit(1);
    }

    if (packageJSONExists) {
      const packageJSONData = JSON.parse(
        fs.readFileSync(packageJSONPath).toString(),
      );
      const { dependencies, devDependencies, peerDependencies } =
        packageJSONData;
      const allDependencies = {
        ...dependencies,
        ...devDependencies,
        ...peerDependencies,
      };
      const dependencyNames = Object.keys(allDependencies);
      const internalDependencies = dependencyNames.filter(name =>
        name.startsWith(packageNamespace),
      );

      const references = internalDependencies.map(name => {
        return { path: `../${name.replace(`${packageNamespace}/`, '')}` };
      });

      const tsConfig = { ...tsConfigBase, references };
      const tsconfigPath = path.join(packageDirectory, 'tsconfig.json');

      const content = TSCONFIG_COMMENT + JSON.stringify(tsConfig, null, '  ');
      const formattedContent = await prettier.format(content, {
        parser: 'json',
      });

      fs.writeFileSync(tsconfigPath, formattedContent);

      console.log(`Generated tsconfig for ${packageJSONData.name}`);
    }
  });
};

generatePackageTSConfig();
