// ReSharper disable once CheckNamespace
namespace LangChain.Providers.OpenAI;

/// <summary>
/// According https://platform.openai.com/docs/guides/text-to-speech.
/// </summary>
public class OpenAiTextToSpeechSettings : TextToSpeechSettings
{
    /// <inheritdoc cref="OpenAiTextToSpeechSettings"/>
    public new static OpenAiTextToSpeechSettings Default { get; } = new()
    {
        Model = CreateSpeechRequestModel.Tts1,
        Voice = CreateSpeechRequestVoice.Alloy,
        ResponseFormat = CreateSpeechRequestResponseFormat.Mp3,
        Speed = 1.0F,
    };

    /// <summary>
    /// 
    /// </summary>
    public AnyOf<string, CreateSpeechRequestModel?>? Model { get; init; }

    /// <summary>
    /// 
    /// </summary>
    public CreateSpeechRequestVoice? Voice { get; init; }

    /// <summary>
    /// 
    /// </summary>
    public CreateSpeechRequestResponseFormat? ResponseFormat { get; init; }

    /// <summary>
    /// 
    /// </summary>
    public float? Speed { get; init; }

    /// <summary>
    /// Calculate the settings to use for the request.
    /// </summary>
    /// <param name="requestSettings"></param>
    /// <param name="modelSettings"></param>
    /// <param name="providerSettings"></param>
    /// <returns></returns>
    /// <exception cref="InvalidOperationException"></exception>
    public static OpenAiTextToSpeechSettings Calculate(
        TextToSpeechSettings? requestSettings,
        TextToSpeechSettings? modelSettings,
        TextToSpeechSettings? providerSettings)
    {
        var requestSettingsCasted = requestSettings as OpenAiTextToSpeechSettings;
        var modelSettingsCasted = modelSettings as OpenAiTextToSpeechSettings;
        var providerSettingsCasted = providerSettings as OpenAiTextToSpeechSettings;

        return new OpenAiTextToSpeechSettings
        {
            Model =
                requestSettingsCasted?.Model ??
                modelSettingsCasted?.Model ??
                providerSettingsCasted?.Model ??
                Default.Model ??
                throw new InvalidOperationException("Default Model is not set."),
            Voice =
                requestSettingsCasted?.Voice ??
                modelSettingsCasted?.Voice ??
                providerSettingsCasted?.Voice ??
                Default.Voice ??
                throw new InvalidOperationException("Default Voice is not set."),
            ResponseFormat =
                requestSettingsCasted?.ResponseFormat ??
                modelSettingsCasted?.ResponseFormat ??
                providerSettingsCasted?.ResponseFormat ??
                Default.ResponseFormat ??
                throw new InvalidOperationException("Default ResponseFormat is not set."),
            Speed =
                requestSettingsCasted?.Speed ??
                modelSettingsCasted?.Speed ??
                providerSettingsCasted?.Speed ??
                Default.Speed ??
                throw new InvalidOperationException("Default Speed is not set."),
        };
    }
}