﻿using System.Diagnostics;
using System.Runtime.CompilerServices;
using System.Text;
using System.Text.Json;

// ReSharper disable once CheckNamespace
namespace LangChain.Providers.Amazon.SageMaker;

public class SageMakerModel(
    SageMakerProvider provider,
    string endpointName)
    : ChatModel(id: endpointName ?? throw new ArgumentNullException(nameof(endpointName), "SageMaker Endpoint Name is not defined"))
{
    public override async IAsyncEnumerable<ChatResponse> GenerateAsync(
        ChatRequest request,
        ChatSettings? settings = null,
        [EnumeratorCancellation] CancellationToken cancellationToken = default)
    {
        request = request ?? throw new ArgumentNullException(nameof(request));

        var messages = request.Messages.ToList();

        var watch = Stopwatch.StartNew();

        var usedSettings = SageMakerSettings.Calculate(
            requestSettings: settings,
            modelSettings: Settings,
            providerSettings: provider.ChatSettings);
        usedSettings.InputParameters?.Add("endpointName", Id);

        using StringContent jsonContent = new(
            JsonSerializer.Serialize(usedSettings.InputParameters),
            Encoding.UTF8,
            usedSettings.ContentType!);

        using var response = await provider.HttpClient.PostAsync(provider.Uri, jsonContent, cancellationToken)
            .ConfigureAwait(false);

        response.EnsureSuccessStatusCode();

        var output = usedSettings.TransformOutput!(response);
        messages.Add(new Message(output!, MessageRole.Ai));

        var usage = Usage.Empty with
        {
            Time = watch.Elapsed,
        };
        AddUsage(usage);
        provider.AddUsage(usage);

        yield return new ChatResponse
        {
            Messages = messages,
            UsedSettings = usedSettings,
            Usage = usage,
        };
    }
}