/* eslint-disable no-console */

import fs from 'fs';
import path from 'path';
import prettier from 'prettier';
import { execSync } from 'child_process';

const CONVERT_SCRIPT_PATH = './scripts/compress-videos.sh';
const COMPRESSED_VIDEOS_PATH = '../assets/compressed';
const VIDEOS_MAP_PATH = './src/videos.ts';

const generateVideoMapFile = () => {
    const videosDir = path.join(__dirname, COMPRESSED_VIDEOS_PATH);
    const videoFileNames = fs.readdirSync(videosDir);

    const mappedVideoNames = videoFileNames.map(
        (fileName: string) =>
            `${fileName.replace('.mp4', '')}: require('${COMPRESSED_VIDEOS_PATH}/${fileName}')`,
    );

    return `
    // !!! IMPORTANT: This file is autogenerated !!!
    // If you want to add of modify videos, please read README.md to find out more.

    export const videos = {
        ${mappedVideoNames.join(',')}
    } as const;
    export type VideoName = keyof typeof videos;
    `;
};

const formatWithPrettier = async (unformatedFileContent: string) => {
    const prettierConfigPath = (await prettier.resolveConfigFile())!;
    const prettierConfig = {
        ...(await prettier.resolveConfig(prettierConfigPath)),
        parser: 'babel-ts',
    };

    return prettier.format(unformatedFileContent, prettierConfig);
};

(async () => {
    const outputBuffer = execSync(CONVERT_SCRIPT_PATH);
    console.log(outputBuffer.toString());
    const videosFileContent = generateVideoMapFile();
    const formattedVideosFileContent = await formatWithPrettier(videosFileContent);
    fs.writeFileSync(path.resolve(VIDEOS_MAP_PATH), formattedVideosFileContent);

    console.log(
        "Video assets processed! See 'assets/compressed' folder and regenerated 'src/videos.ts' config.",
    );
})();
