import { useSelector } from 'react-redux';

import { pipe } from '@mobily/ts-belt';

import { convertCryptoToFiatAmount } from '@suite-common/formatters';
import { Timestamp, WalletAccountTransaction } from '@suite-common/wallet-types';
import { Card, Table, Td, Text, Th, Tr, VStack } from '@suite-native/atoms';
import { NetworkSymbol } from '@suite-common/wallet-config';
import {
    FiatRatesRootState,
    selectFiatRatesByFiatRateKey,
    selectHistoricFiatRatesByTimestamp,
} from '@suite-common/wallet-core';
import { getFiatRateKey } from '@suite-common/wallet-utils';
import { selectFiatCurrencyCode } from '@suite-native/settings';
import {
    CryptoToFiatAmountFormatter,
    PercentageDifferenceFormatter,
} from '@suite-native/formatters';
import { useTranslate } from '@suite-native/intl';

import { TransactionDetailSheet } from './TransactionDetailSheet';

type TransactionDetailValuesSheetProps = {
    isVisible: boolean;
    transaction: WalletAccountTransaction;
    onSheetVisibilityChange: () => void;
};

type TodayHeaderCellProps = {
    cryptoValue: string;
    symbol: NetworkSymbol;
    historicRate?: number;
};

const TodayHeaderCell = ({ cryptoValue, symbol, historicRate }: TodayHeaderCellProps) => {
    const fiatCurrencyCode = useSelector(selectFiatCurrencyCode);
    const fiatRateKey = getFiatRateKey(symbol, fiatCurrencyCode);
    const currentRates = useSelector((state: FiatRatesRootState) =>
        selectFiatRatesByFiatRateKey(state, fiatRateKey),
    );

    if (!currentRates || !historicRate) return null;

    const fiatTotalHistoryNumeric = pipe(
        convertCryptoToFiatAmount({
            amount: cryptoValue,
            symbol,
            rate: historicRate,
        }) ?? 0,
        Number,
    );
    const fiatTotalActualNumeric = pipe(
        convertCryptoToFiatAmount({
            amount: cryptoValue,
            symbol,
            rate: currentRates?.rate,
        }),
        Number,
    );

    return (
        <Text variant="hint" color="textSubdued">
            Today{' '}
            <PercentageDifferenceFormatter
                oldValue={fiatTotalHistoryNumeric}
                newValue={fiatTotalActualNumeric}
                variant="hint"
            />
        </Text>
    );
};

export const TransactionDetailValuesSheet = ({
    isVisible,
    onSheetVisibilityChange,
    transaction,
}: TransactionDetailValuesSheetProps) => {
    const { translate } = useTranslate();

    // Fallback to transaction.amount if totalInput is 0, which is the case for XRP transactions
    const totalInput =
        transaction.details.totalInput === '0'
            ? transaction.amount
            : transaction.details.totalInput;

    const fiatCurrencyCode = useSelector(selectFiatCurrencyCode);
    const fiatRateKey = getFiatRateKey(transaction.symbol, fiatCurrencyCode);
    const historicRate = useSelector((state: FiatRatesRootState) =>
        selectHistoricFiatRatesByTimestamp(state, fiatRateKey, transaction.blockTime as Timestamp),
    );

    return (
        <TransactionDetailSheet
            isVisible={isVisible}
            onVisibilityChange={onSheetVisibilityChange}
            title={translate('transactions.detail.sheet.values')}
            iconName="clockClockwise"
            transactionId={transaction.txid}
        >
            <VStack>
                <Card>
                    <Table>
                        <Tr>
                            <Th />
                            <Th>Transaction</Th>
                            <Th>
                                <TodayHeaderCell
                                    cryptoValue={transaction.amount}
                                    historicRate={historicRate}
                                    symbol={transaction.symbol}
                                />
                            </Th>
                        </Tr>

                        <Tr>
                            <Th>Input</Th>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={totalInput}
                                    symbol={transaction.symbol}
                                    historicRate={historicRate}
                                    useHistoricRate
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={totalInput}
                                    symbol={transaction.symbol}
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                        </Tr>
                        <Tr>
                            <Th>Fee</Th>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={transaction.fee}
                                    symbol={transaction.symbol}
                                    historicRate={historicRate}
                                    useHistoricRate
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={transaction.fee}
                                    symbol={transaction.symbol}
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                        </Tr>
                        <Tr>
                            <Th>Total</Th>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={transaction.amount}
                                    symbol={transaction.symbol}
                                    historicRate={historicRate}
                                    useHistoricRate
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                            <Td>
                                <CryptoToFiatAmountFormatter
                                    variant="hint"
                                    value={transaction.amount}
                                    symbol={transaction.symbol}
                                    numberOfLines={1}
                                    adjustsFontSizeToFit
                                />
                            </Td>
                        </Tr>
                    </Table>
                </Card>
            </VStack>
        </TransactionDetailSheet>
    );
};
