/* eslint-disable import/no-extraneous-dependencies */

import chalk from 'chalk';

const fs = require('fs');
const path = require('path');
const prettier = require('prettier');
const { optimize } = require('svgo');

const iconsFilePath = './src/icons.ts';
const cryptoIconsPath = './src/cryptoIcons.ts';

const assetTypesConfig = [
    {
        name: 'icons',
        dirname: 'assets',
        typeName: 'IconName',
    },
];

const cryptoAssetsTypesConfig = [
    {
        name: 'cryptoIcons',
        dirname: 'cryptoAssets/cryptoIcons',
        typeName: 'CryptoIconName',
    },
];

// https://github.com/svg/svgo#built-in-plugins
const svgoConfig = {
    multipass: true,
    js2svg: {
        indent: 2, // string with spaces or number of spaces. 4 by default
        pretty: true, // boolean, false by default
    },
    plugins: [
        {
            name: 'preset-default',
            params: {
                overrides: {
                    removeViewBox: false,
                },
            },
        },
        'prefixIds',
        // it's necessary to remove all dimension tags to allow resizing
        'removeDimensions',
        'removeRasterImages',
        'removeScriptElement',
    ],
};

const optimizeSvgAssets = assetsDirname => {
    const assetsDir = path.join(__dirname, assetsDirname);
    const assetFileNames = fs.readdirSync(assetsDir);

    return assetFileNames
        .filter(fileName => fileName.endsWith('.svg'))
        .map(fileName => ({
            fileName,
            content: fs.readFileSync(path.resolve(assetsDir, fileName)).toString(),
        }))
        .map(({ fileName, content }) => ({
            fileName,
            content: optimize(content, svgoConfig).data,
        }));
};

const getOptimizedAssetTypes = assetTypesArray =>
    assetTypesArray.map(config => ({
        ...config,
        assets: optimizeSvgAssets(config.dirname),
    }));

const generateIconsFileContent = assetTypesArray => {
    const mappedAssetTypes = assetTypesArray.map(
        ({ name, assets, dirname, typeName }) => `
           export const ${name} = {
            ${assets
                .map(
                    ({ fileName }) =>
                        `${fileName.replace('.svg', '')}: require('../${dirname}/${fileName}')`,
                )
                .join(',')}
        } as const;
        export type ${typeName} = keyof typeof ${name};
       `,
    );

    return `
    // !!! IMPORTANT: This file is autogenerated !!!
    // If you want to add of modify icons please read README.md to find out how to do it

    ${mappedAssetTypes.join('')}
    `;
};

const writeOptimizedAssets = assetTypesArray => {
    assetTypesArray.forEach(({ assets, dirname }) => {
        assets.forEach(({ fileName, content }) =>
            fs.writeFileSync(path.resolve(dirname, fileName), content),
        );
    });
};

const generateFileForAssetTypes = async (assetTypesArray, outputFilePath) => {
    const assetTypes = getOptimizedAssetTypes(assetTypesArray);

    const iconsFileContent = generateIconsFileContent(assetTypes);

    const prettierConfigPath = await prettier.resolveConfigFile();
    const prettierConfig = {
        ...(await prettier.resolveConfig(prettierConfigPath)),
        parser: 'babel-ts',
    };

    const formattedIconTypesFileContent = await prettier.format(iconsFileContent, prettierConfig);

    fs.writeFileSync(path.resolve(outputFilePath), formattedIconTypesFileContent);

    writeOptimizedAssets(assetTypes);
};

(async () => {
    console.log('Generating icons TS file...');
    await generateFileForAssetTypes(assetTypesConfig, iconsFilePath);
    console.log(chalk.green('Icons TS file generated successfully'));
    console.log('Generating crypto icons TS file...');
    await generateFileForAssetTypes(cryptoAssetsTypesConfig, cryptoIconsPath);
    console.log(chalk.green('Crypto icons TS file generated successfully'));
})();
