import { combineReducers, createReducer } from '@reduxjs/toolkit';

import { notificationsActions } from '@suite-common/toast-notifications';
import { DEFAULT_PAYMENT, DEFAULT_VALUES } from '@suite-common/wallet-constants';
import { accountsActions, prepareSendFormReducer } from '@suite-common/wallet-core';
import { PROTO } from '@trezor/connect';
import { testMocks } from '@suite-common/test-utils';

import { extraDependencies } from 'src/support/extraDependencies';

const sendFormReducer = prepareSendFormReducer(extraDependencies);

const UTXO = {
    '00': testMocks.getUtxo({
        amount: '0',
        address: 'should-never-be-used',
        txid: '0000000000000000000000000000000000000000000000000000000000000000',
        vout: 0,
    }),
    AA: testMocks.getUtxo({
        amount: '50000000000',
        address: 'AA',
        txid: 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa',
        vout: 0,
    }),
    BB: testMocks.getUtxo({
        amount: '25000000000',
        address: 'BB',
        txid: 'bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb',
        vout: 0,
    }),
    CC: testMocks.getUtxo({
        amount: '12500000000',
        address: 'CC',
        txid: 'cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc',
        vout: 0,
    }),
    DD: testMocks.getUtxo({
        amount: '6250000000',
        address: 'DD',
        txid: 'dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd',
        vout: 0,
    }),
    EE: testMocks.getUtxo({
        amount: '6250000000',
        address: 'EE',
        txid: 'eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
        vout: 0,
    }),
};

export const BTC_ACCOUNT = {
    status: 'loaded',
    account: testMocks.getWalletAccount({
        symbol: 'btc',
        networkType: 'bitcoin',
        descriptor: 'xpub',
        deviceState: '1stTestnetAddress@device_id:0',
        key: 'xpub-btc-1stTestnetAddress@device_id:0',
        addresses: {
            change: [
                { path: "m/44'/0'/0'/1/0", address: '1-change', transfers: 0 },
                { path: "m/44'/0'/0'/1/1", address: '2-change', transfers: 0 },
            ],
            used: [
                { path: "m/44'/0'/0'/0/0", address: '1-used', transfers: 1 },
                { path: "m/44'/0'/0'/0/1", address: '2-used', transfers: 1 },
            ],
            unused: [
                { path: "m/44'/0'/0'/0/2", address: '1-unused', transfers: 0 },
                { path: "m/44'/0'/0'/0/3", address: '2-unused', transfers: 0 },
            ],
        },
        balance: '100000000000',
        availableBalance: '100000000000',
        formattedBalance: '1000 BTC',
        utxo: Object.values(UTXO),
    }),
    network: { networkType: 'bitcoin', symbol: 'btc', decimals: 8, features: ['rbf'] },
};

export const ETH_ACCOUNT = {
    status: 'loaded',
    account: testMocks.getWalletAccount({
        symbol: 'eth',
        networkType: 'ethereum',
        descriptor: '0xdB09b793984B862C430b64B9ed53AcF867cC041F',
        deviceState: '1stTestnetAddress@device_id:0',
        key: '0xdB09b793984B862C430b64B9ed53AcF867cC041F-eth-1stTestnetAddress@device_id:0',
        balance: '10000000000000000000', // 10 ETH
        availableBalance: '10000000000000000000', // 10 ETH
        misc: { nonce: '0' },
        tokens: [
            { type: 'ERC20', contract: '0xABCD', symbol: '0xABCD', decimals: 3, balance: '1' },
        ],
    }),
    network: { networkType: 'ethereum', symbol: 'eth', decimals: 18, chainId: 1 },
};

export const XRP_ACCOUNT = {
    status: 'loaded',
    account: testMocks.getWalletAccount({
        symbol: 'xrp',
        networkType: 'ripple',
        descriptor: 'rAPERVgXZavGgiGv6xBgtiZurirW2yAmY',
        deviceState: '1stTestnetAddress@device_id:0',
        key: 'rAPERVgXZavGgiGv6xBgtiZurirW2yAmY-xrp-1stTestnetAddress@device_id:0',
        balance: '100000000', // 100 XRP
        availableBalance: '100000000', // 100 XRP
        misc: { reserve: '21', sequence: 0 },
    }),
    network: { networkType: 'ripple', symbol: 'xrp', decimals: 6 },
};

export const SOL_ACCOUNT = {
    status: 'loaded',
    account: {
        symbol: 'sol',
        networkType: 'solana',
        descriptor: 'ETxHeBBcuw9Yu4dGuP3oXrD12V5RECvmi8ogQ9PkjyVF',
        deviceState: '1stTestnetAddress@device_id:0',
        key: 'ETxHeBBcuw9Yu4dGuP3oXrD12V5RECvmi8ogQ9PkjyVF-sol-1stTestnetAddress@device_id:0',
        balance: '10000000000', // 10 SOL
        availableBalance: '10000000000', // 10 SOL
        misc: {
            rent: 10,
        },
        history: {},
        tokens: [],
    },
    network: { networkType: 'solana', symbol: 'sol', decimals: 9, chainId: 1399811149 },
};

const DEVICE = testMocks.getSuiteDevice({
    state: '1stTestnetAddress@device_id:0',
    connected: true,
    available: true,
});

const DEFAULT_FEES = {
    btc: {
        minFee: 1,
        maxFee: 100,
        blockHeight: 1,
        blockTime: 1,
        levels: [{ label: 'normal', feePerUnit: '4', blocks: 1 }],
    },
    eth: {
        minFee: 1,
        maxFee: 100,
        blockHeight: 1,
        blockTime: 1,
        levels: [
            {
                label: 'normal',
                feePerUnit: '3300000000',
                feeLimit: '21000',
                blocks: -1,
            },
        ],
    },
    xrp: {
        minFee: 1,
        maxFee: 100,
        blockHeight: 1,
        blockTime: 1,
        levels: [{ label: 'normal', feePerUnit: '12', blocks: -1 }],
    },
    sol: {
        minFee: -1,
        maxFee: -1,
        blockHeight: -1,
        blockTime: -1,
        levels: [
            {
                label: 'normal',
                feePerUnit: '100000',
                feeLimit: '50000',
                feePerTx: '10000',
                blocks: -1,
            },
        ],
    },
};

// - default selectedAccount needs to be explicitly passed from test. merging default with custom will override custom
// - default fees needs to be explicitly passed from test. merge Arrays will add items, not replace them
export const getRootReducer = (selectedAccount = BTC_ACCOUNT, fees = DEFAULT_FEES) =>
    combineReducers({
        suite: createReducer(
            {
                locks: [],
                online: true,
                settings: { debug: {}, theme: { variant: 'light' } },
                evmSettings: { confirmExplanationModalClosed: {}, explanationBannerClosed: {} },
                prefillFields: { sendForm: '', transactionHistory: '' },
                flags: { stakeEthBannerClosed: false },
            },
            () => ({}),
        ),
        device: createReducer({ selectedDevice: DEVICE, devices: [DEVICE] }, () => {}),
        wallet: combineReducers({
            send: sendFormReducer,
            accounts: createReducer(
                [
                    BTC_ACCOUNT.account,
                    ETH_ACCOUNT.account,
                    XRP_ACCOUNT.account,
                    SOL_ACCOUNT.account,
                ],
                () => ({}),
            ),
            selectedAccount: createReducer(selectedAccount, () => ({})),
            coinjoin: createReducer({ accounts: [] }, () => ({})),
            discovery: createReducer([], () => ({})),
            settings: createReducer(
                {
                    localCurrency: 'usd',
                    lastUsedFeeLevel: {},
                    debug: {},
                    bitcoinAmountUnit: PROTO.AmountUnit.BITCOIN,
                },
                () => ({}),
            ),
            blockchain: createReducer(
                {
                    btc: {},
                    eth: {},
                    xrp: {},
                    sol: { blockHash: 'BuKJXfBwb5BUXK7wACFCBpTHKyzcSfnAXG2NpyHJQhcX' },
                },
                () => ({}),
            ),
            fees: createReducer(fees, () => ({})),
            fiat: createReducer(
                {
                    current: {
                        'btc-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 1693235607743000,
                            fetchFetchAttemptTimestamp: 1693235607743,
                            rate: 1,
                            ticker: {
                                symbol: 'btc',
                            },
                        },
                        'eth-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 169323560946000,
                            fetchFetchAttemptTimestamp: 1693235609465,
                            rate: 1,
                            ticker: {
                                symbol: 'eth',
                            },
                        },
                        'xrp-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 1693235609467000,
                            fetchFetchAttemptTimestamp: 1693235609467,
                            rate: 1,
                            ticker: {
                                symbol: 'xrp',
                            },
                        },
                    },
                    lastWeek: {
                        'btc-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 1693235707743000,
                            fetchFetchAttemptTimestamp: 1693235707743,
                            rate: 1,
                            ticker: {
                                symbol: 'btc',
                            },
                        },
                        'eth-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 1693235709465000,
                            fetchFetchAttemptTimestamp: 1693235709465,
                            rate: 1,
                            ticker: {
                                symbol: 'eth',
                            },
                        },
                        'xrp-usd': {
                            error: null,
                            isLoading: false,
                            lastTickerTimestamp: 1693235709467000,
                            fetchFetchAttemptTimestamp: 1693235709467,
                            rate: 1,
                            ticker: {
                                symbol: 'xrp',
                            },
                        },
                    },
                },
                () => ({}),
            ),
            transactions: createReducer(
                {
                    transactions: {},
                },
                () => ({}),
            ),
        }),
        protocol: createReducer({ sendForm: {} }, () => ({})),
        messageSystem: createReducer(
            {
                validMessages: {
                    banner: [],
                    context: [],
                    modal: [],
                    feature: [],
                },
                dismissedMessages: {},
            },
            () => ({}),
        ),
        window: createReducer({}, () => ({})),
        guide: createReducer({}, () => ({})),
        metadata: createReducer(
            { enabled: false, providers: [], selectedProvider: {} },
            () => ({}),
        ),
        router: createReducer({}, () => ({})),
        modal: createReducer({}, () => ({})),
    });

const DEFAULT_DRAFT = {
    ...DEFAULT_VALUES,
    outputs: [{ ...DEFAULT_PAYMENT }],
    options: [...DEFAULT_VALUES.options],
    selectedUtxos: [],
};

const getDraft = (draft?: any) => ({
    'xpub-btc-1stTestnetAddress@device_id:0': {
        ...DEFAULT_DRAFT,
        outputs: [
            {
                ...DEFAULT_PAYMENT,
                address: 'A',
                amount: '1',
            },
        ],
        ...draft,
    },
    '0xdB09b793984B862C430b64B9ed53AcF867cC041F-eth-1stTestnetAddress@device_id:0': {
        ...DEFAULT_DRAFT,
        outputs: [
            {
                ...DEFAULT_PAYMENT,
                address: '0xdB09b793984B862C430b64B9ed53AcF867cC041F',
                amount: '1',
            },
        ],
        ...draft,
    },
    'rAPERVgXZavGgiGv6xBgtiZurirW2yAmY-xrp-1stTestnetAddress@device_id:0': {
        ...DEFAULT_DRAFT,
        outputs: [
            {
                ...DEFAULT_PAYMENT,
                address: 'rAPERVgXZavGgiGv6xBgtiZurirW2yAmY',
                amount: '1',
            },
        ],
        ...draft,
    },
    'ETxHeBBcuw9Yu4dGuP3oXrD12V5RECvmi8ogQ9PkjyVF-sol-1stTestnetAddress@device_id:0': {
        ...DEFAULT_DRAFT,
        outputs: [
            {
                ...DEFAULT_PAYMENT,
                address: 'ETxHeBBcuw9Yu4dGuP3oXrD12V5RECvmi8ogQ9PkjyVF',
                amount: '1',
            },
        ],
        ...draft,
    },
});

export const addingOutputs = [
    {
        description: 'Add/Remove/Reset outputs without draft',
        initial: {
            outputs: [{ address: '' }],
        },
        actions: [
            {
                type: 'click',
                element: 'add-output',
                result: {
                    formValues: {
                        outputs: [{ address: '' }, { address: '' }],
                    },
                },
            },
            {
                type: 'click',
                element: 'outputs.0.remove',
                result: {
                    formValues: {
                        outputs: [{ address: '' }],
                    },
                },
            },
        ],
    },
    {
        description: 'Add/Remove/Reset outputs with draft and set-max in second output',
        store: {
            send: {
                drafts: {
                    'xpub-btc-1stTestnetAddress@device_id:0': {
                        ...DEFAULT_DRAFT,
                        outputs: [
                            {
                                ...DEFAULT_PAYMENT,
                                address: 'A',
                                amount: '1',
                            },
                            {
                                ...DEFAULT_PAYMENT,
                                address: 'B',
                                amount: '2',
                            },
                        ],
                        setMaxOutputId: 1,
                    },
                },
            },
        },
        initial: {
            outputs: [{ address: 'A' }, { address: 'B' }],
        },
        actions: [
            {
                type: 'click',
                element: 'add-output',
                result: {
                    formValues: {
                        outputs: [{ address: 'A' }, { address: 'B' }, { address: '' }],
                        setMaxOutputId: 1,
                    },
                },
            },
            {
                type: 'click',
                element: 'outputs.0.remove',
                result: {
                    formValues: {
                        outputs: [{ address: 'B' }, { address: '' }],
                        setMaxOutputId: 0,
                    },
                },
            },
            {
                type: 'click',
                element: 'clear-form',
                result: {
                    formValues: {
                        outputs: [{ address: '' }],
                        setMaxOutputId: undefined,
                    },
                },
            },
        ],
    },
    {
        description: 'Add/Remove/Reset outputs with draft and set-max in first output',
        store: {
            send: {
                drafts: {
                    'xpub-btc-1stTestnetAddress@device_id:0': {
                        ...DEFAULT_DRAFT,
                        outputs: [
                            {
                                ...DEFAULT_PAYMENT,
                                address: 'A',
                                amount: '1',
                            },
                            {
                                ...DEFAULT_PAYMENT,
                                address: 'B',
                                amount: '2',
                            },
                        ],
                        setMaxOutputId: 0,
                    },
                },
            },
        },
        initial: {
            outputs: [{ address: 'A' }, { address: 'B' }],
        },
        actions: [
            {
                type: 'click',
                element: 'add-output',
                result: {
                    formValues: {
                        outputs: [{ address: 'A' }, { address: 'B' }, { address: '' }],
                        setMaxOutputId: 0,
                    },
                },
            },
            {
                type: 'click',
                element: 'outputs.0.remove',
                result: {
                    formValues: {
                        outputs: [{ address: 'B' }, { address: '' }],
                        setMaxOutputId: undefined,
                    },
                },
            },
            {
                type: 'click',
                element: 'clear-form',
                result: {
                    formValues: {
                        outputs: [{ address: '' }],
                        setMaxOutputId: undefined,
                    },
                },
            },
        ],
    },
];

export const drafts = [
    {
        description: 'Load draft and compose without errors',
        connect: {
            success: true,
            payload: [],
        },
    },
    {
        description: 'Load draft and compose with errors',
    },
];

export const composeDebouncedTransaction = [
    {
        description: 'compose with validation errors (Address not set)',
        actions: [
            {
                type: 'input',
                element: 'outputs.0.address',
                value: 'X',
                result: {
                    errors: {
                        outputs: [{ address: { type: 'valid' } }],
                    },
                },
            },
            {
                type: 'input',
                element: 'outputs.0.address',
                result: {
                    errors: {
                        outputs: [{ address: { type: 'required' } }],
                    },
                },
            },
        ],
        finalResult: {
            composeTransactionCalls: 0,
            composedLevels: undefined,
            errors: {
                outputs: [{ address: { type: 'required' } }],
            },
        },
    },
    {
        description: 'compose with validation errors (Address invalid)',
        actions: [{ type: 'input', element: 'outputs.0.address', value: 'FOO', delay: 1 }],
        finalResult: {
            composeTransactionCalls: 0,
            composedLevels: undefined,
            errors: {
                outputs: [{ address: { type: 'valid' } }],
            },
        },
    },
    {
        description: '@trezor/connect call respond with success:false',
        connect: {
            success: false,
            payload: { error: 'error' },
        },
        actions: [{ type: 'input', element: 'outputs.0.amount', value: '1' }],
        finalResult: {
            composeTransactionCalls: 1,
            composedLevels: undefined,
        },
    },
    {
        description: 'Fast typing, one @trezor/connect call',
        connect: {
            success: true,
            payload: [
                {
                    type: 'nonfinal',
                },
            ],
        },
        actions: [{ type: 'input', element: 'outputs.0.amount', value: '111', delay: 100 }],
        finalResult: {
            composeTransactionCalls: 1,
            composedLevels: {
                normal: {
                    type: 'nonfinal',
                },
            },
        },
    },
    {
        description: 'Slow typing, multiple @trezor/connect calls, only last call gets processed',
        connect: [
            {
                success: true,
                payload: [{ type: 'nonfinal', totalSpent: '100000000' }],
            },
            {
                // delay in @trezor/connect response greater than typing delay
                // basically it means: return this response AFTER third call to connect, this response should be ignored
                delay: 500,
                success: true,
                payload: [{ type: 'nonfinal', totalSpent: '1100000000' }],
            },
            {
                success: true,
                payload: [{ type: 'nonfinal', totalSpent: '11100000000' }],
            }, // delay in @trezor/connect response, greater than typing delay
        ],
        actions: [{ type: 'input', element: 'outputs.0.amount', value: '111', delay: 310 }], // delay greater than composeDebounced timeout
        finalResult: {
            composeTransactionCalls: 3,
            composedLevels: {
                normal: {
                    type: 'nonfinal',
                    totalSpent: '11100000000',
                },
            },
        },
    },
];

export const setMax = [
    {
        description: 'setMax: utxos are excluded because of insufficient anonymity',
        store: {
            selectedAccount: {
                ...BTC_ACCOUNT,
                account: {
                    ...BTC_ACCOUNT.account,
                    accountType: 'coinjoin',
                    addresses: {
                        ...BTC_ACCOUNT.account.addresses,
                        anonymitySet: {
                            AA: 0,
                            BB: 50,
                            DD: 50,
                        },
                    },
                },
            },
            coinjoin: {
                accounts: [
                    {
                        targetAnonymity: 50,
                        key: 'xpub-btc-1stTestnetAddress@device_id:0',
                    },
                ],
            },
        },
        actions: [
            {
                type: 'click',
                element: 'coin-control-button',
            },
            {
                type: 'hover',
                element: 'outputs.0.amount',
            },
            {
                type: 'click',
                element: 'outputs.0.setMax',
                result: {
                    composeTransactionCalls: 1,
                    composeTransactionParams: {
                        outputs: [{ type: 'send-max-noaddress' }],
                    },
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [{ address: '', amount: '312.499995', fiat: '312.50' }],
                    },
                    composedLevels: {
                        normal: { type: 'nonfinal' },
                    },
                },
            },
        ],
        connect: {
            success: true,
            payload: [
                {
                    type: 'nonfinal',
                    max: '31249999500',
                },
            ],
        },
        finalResult: {
            composeTransactionCalls: 1,
            composeTransactionParams: {
                account: {
                    // only utxos with enough anonymity are used in TrezorConnect.composeTransaction
                    // see sendFormBitcoinActions
                    utxo: [UTXO.BB, UTXO.DD],
                },
            },
        },
    },
    {
        description:
            'setMax: compose from draft (one input), Amount not affected, but Fiat gets recalculated',
        store: {
            send: {
                drafts: {
                    'xpub-btc-1stTestnetAddress@device_id:0': {
                        ...DEFAULT_DRAFT,
                        outputs: [
                            {
                                ...DEFAULT_PAYMENT,
                                address: '',
                                amount: '1',
                                fiat: '2.00',
                            },
                        ],
                        setMaxOutputId: 0,
                    },
                },
            },
        },
        connect: {
            success: true,
            payload: [
                {
                    type: 'nonfinal',
                    max: '100000000',
                },
            ],
        },
        finalResult: {
            composeTransactionCalls: 1,
            composedLevels: {
                normal: {
                    type: 'nonfinal',
                },
            },
            formValues: {
                selectedFee: undefined,
                outputs: [{ address: '', amount: '1', fiat: '1.00' }],
            },
        },
    },
    {
        description:
            'setMax: compose from draft with error on default level, switching to custom level',
        store: {
            send: {
                drafts: {
                    'xpub-btc-1stTestnetAddress@device_id:0': {
                        ...DEFAULT_DRAFT,
                        outputs: [
                            {
                                ...DEFAULT_PAYMENT,
                                address: 'A',
                                amount: '1',
                                fiat: '2.00',
                            },
                        ],
                        setMaxOutputId: 0,
                    },
                },
            },
        },
        connect: [
            {
                success: true,
                payload: [
                    {
                        type: 'error',
                    },
                ],
            },
            {
                success: true,
                payload: [
                    { type: 'error' },
                    { type: 'final', feePerByte: '2', max: '10000000' },
                    { type: 'final', feePerByte: '1', max: '10000001' },
                ],
            },
        ],
        finalResult: {
            composeTransactionCalls: 2,
            composedLevels: { normal: { type: 'error' }, custom: { type: 'final' } },
            formValues: {
                selectedFee: 'custom' as const,
                feePerUnit: '2',
                outputs: [{ amount: '0.1', fiat: '0.10' }],
            },
        },
    },
    {
        description: 'setMax sequence: compose not final (without address) then add address',
        connect: [
            {
                success: true,
                payload: [
                    {
                        type: 'nonfinal',
                        max: '100000000',
                    },
                ],
            },
            {
                success: true,
                payload: [
                    {
                        type: 'final',
                        max: '100000000',
                    },
                ],
            },
        ],
        actions: [
            {
                type: 'hover',
                element: 'outputs.0.amount',
            },
            {
                type: 'click',
                element: 'outputs.0.setMax',
                result: {
                    composeTransactionCalls: 1,
                    composeTransactionParams: {
                        outputs: [{ type: 'send-max-noaddress' }],
                    },
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [{ address: '', amount: '1', fiat: '1.00' }],
                    },
                    composedLevels: {
                        normal: { type: 'nonfinal' },
                    },
                },
            },
            {
                type: 'input',
                element: 'outputs.0.address',
                value: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                result: {
                    composeTransactionParams: {
                        outputs: [
                            { type: 'send-max', address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX' },
                        ],
                    },
                },
            },
        ],
        finalResult: {
            composeTransactionCalls: 2,
            formValues: {
                setMaxOutputId: 0,
                outputs: [
                    { address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX', amount: '1', fiat: '1.00' },
                ],
            },
            composedLevels: {
                normal: { type: 'final' },
            },
        },
    },
    {
        description:
            'setMax sequence: compose final with address, disable setMax, add second output',
        connect: [
            {
                success: true,
                payload: [
                    {
                        type: 'final',
                        max: '100000000',
                    },
                ],
            },
            {
                success: true,
                payload: [
                    {
                        type: 'nonfinal',
                        max: '100000000',
                    },
                ],
            },
            {
                success: true,
                payload: [
                    {
                        type: 'nonfinal',
                        max: '100000000',
                    },
                ],
            },
            {
                success: true,
                payload: [
                    {
                        type: 'final',
                        totalSpent: '120000000',
                    },
                ],
            },
        ],
        actions: [
            {
                type: 'input',
                element: 'outputs.0.address',
                value: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                result: {
                    composeTransactionCalls: 0,
                    composedLevels: undefined,
                },
            },
            {
                type: 'hover',
                element: 'outputs.0.amount',
            },
            {
                type: 'click',
                element: 'outputs.0.setMax',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [
                            {
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '1',
                                fiat: '1.00',
                            },
                        ],
                    },
                    composedLevels: {
                        normal: { type: 'final' },
                    },
                },
            },
            // add second output
            {
                type: 'click',
                element: 'add-output',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [
                            {
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '1',
                                fiat: '1.00',
                            },
                            { address: '' },
                        ],
                    },
                    composedLevels: {
                        normal: { type: 'final' },
                    },
                },
            },
            // fill address in second output
            {
                type: 'input',
                element: 'outputs.1.address',
                value: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                result: {
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [
                            {
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '1',
                                fiat: '1.00',
                            },
                            { address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX', amount: '' },
                        ],
                    },
                    composeTransactionParams: {
                        outputs: [
                            // corner-case: send-max was changed to send-max-noaddress
                            // see sendFormUtils.getBitcoinComposeOutputs
                            {
                                type: 'send-max-noaddress',
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                            },
                        ],
                    },
                },
            },
            // disable send max
            {
                type: 'click',
                element: 'outputs.0.setMax',
                result: {
                    formValues: {
                        setMaxOutputId: undefined,
                    },
                    composeTransactionParams: {
                        outputs: [
                            // corner-case: payment was changed to payment-noaddress
                            // see sendFormUtils.getBitcoinComposeOutputs
                            {
                                type: 'payment-noaddress',
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '100000000',
                            },
                        ],
                    },
                },
            },
            // fill fiat in second output
            {
                type: 'input',
                element: 'outputs.1.fiat',
                value: '0.20',
                result: {
                    formValues: {
                        outputs: [
                            {
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '1',
                                fiat: '1.00',
                            },
                            {
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '0.20000000',
                                fiat: '0.20',
                            },
                        ],
                    },
                    composeTransactionParams: {
                        outputs: [
                            {
                                type: 'payment',
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '100000000',
                            },
                            {
                                type: 'payment',
                                address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                                amount: '20000000',
                            },
                        ],
                    },
                },
            },
            // remove second output
            // {
            //     type: 'click',
            //     element: 'outputs.1.remove',
            //     result: {
            //         composeTransactionParams: {
            //             outputs: [
            //                 { type: 'payment', address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX', amount: '100000000' }
            //             ]
            //         },
            //     },
            // }
        ],
        finalResult: {
            composeTransactionCalls: 4,
            formValues: {
                setMaxOutputId: undefined,
                outputs: [
                    { address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX', amount: '1', fiat: '1.00' },
                    {
                        address: '3AnYTd2FGxJLNKL1AzxfW3FJMntp9D2KKX',
                        amount: '0.20000000',
                        fiat: '0.20',
                    },
                ],
            },
            composedLevels: {
                normal: {
                    type: 'final',
                    totalSpent: '120000000',
                },
            },
        },
    },
    {
        description: 'ETH',
        store: {
            send: {
                drafts: getDraft({
                    setMaxOutputId: 0,
                }),
            },
            selectedAccount: ETH_ACCOUNT,
        },
        finalResult: {
            estimateFeeCalls: 1,
            composedLevels: {
                normal: {
                    type: 'final',
                    fee: '69300000000000',
                    totalSpent: '10000000000000000000',
                },
                custom: undefined,
            },
            formValues: {
                outputs: [{ amount: '9.9999307', fiat: '10.00' }],
            },
        },
    },
    {
        description: 'ETH with token',
        store: {
            send: {
                drafts: getDraft({
                    outputs: [
                        {
                            ...DEFAULT_PAYMENT,
                            address: '0xABCD',
                            token: '0xABCD',
                        },
                    ],
                    setMaxOutputId: 0,
                }),
            },
            selectedAccount: ETH_ACCOUNT,
        },
        finalResult: {
            estimateFeeCalls: 1,
            composedLevels: {
                normal: {
                    type: 'final',
                    fee: '69300000000000',
                    totalSpent: '1000', // tokens
                },
                custom: undefined,
            },
            formValues: {
                outputs: [{ amount: '1', token: '0xABCD' }],
            },
        },
    },
    {
        description: 'XRP',
        store: {
            send: {
                drafts: getDraft({
                    setMaxOutputId: 0,
                }),
            },
            selectedAccount: XRP_ACCOUNT,
        },
        finalResult: {
            getAccountInfoCalls: 1,
            composedLevels: {
                normal: { type: 'final', fee: '12', totalSpent: '100000000' },
                custom: undefined,
            },
            formValues: {
                outputs: [{ amount: '99.999988', fiat: '100.00' }],
            },
        },
    },
    {
        description: 'XRP no address',
        store: {
            send: {
                drafts: getDraft({
                    outputs: [{ ...DEFAULT_PAYMENT }],
                    setMaxOutputId: 0,
                }),
            },
            selectedAccount: XRP_ACCOUNT,
        },
        finalResult: {
            getAccountInfoCalls: 0,
            composedLevels: {
                normal: { type: 'nonfinal', fee: '12', totalSpent: '100000000' },
                custom: undefined,
            },
            formValues: {
                outputs: [{ amount: '99.999988', fiat: '100.00' }],
            },
        },
    },
    {
        description: 'SOL',
        store: {
            send: {
                drafts: getDraft({
                    setMaxOutputId: 0,
                }),
            },
            selectedAccount: SOL_ACCOUNT,
        },
        finalResult: {
            estimateFeeCalls: 1,
            composedLevels: {
                normal: {
                    type: 'final',
                    fee: '10000',
                    totalSpent: '10000000000',
                },
                custom: undefined,
            },
            formValues: {
                outputs: [{ amount: '9.99999' }],
            },
        },
    },
];

export const amountChange = [
    {
        description: 'Amount to Fiat calculation',
        // input amount
        // input amount with error
        // change currency
    },
    {
        description: 'Amount with error',
        // input amount
        // input amount with error
        // change currency
    },
    {
        description: 'Amount to Fiat calculation then Amount with error',
        // input amount
        // input amount with error
        // change currency
    },
    {
        description: 'Fiat to Amount calculation',
        // input fiat
        // input fiat with error
        // change currency
    },
    {
        description: 'Fiat with error',
        // input fiat
        // input fiat with error
        // change currency
    },
    {
        description: 'Fiat to Amount calculation then Fiat with error',
        // input fiat
        // input fiat with error
        // change currency
    },
    {
        description: 'Eth transaction with data (default amount set to 0)',
    },
];

const getComposeResponse = (resp?: any) => ({
    success: true,
    payload: [
        {
            type: 'final',
            totalSpent: '2500000000',
            fee: '100',
            inputs: [{ amount: '12500000000', prev_hash: UTXO.CC.txid, prev_index: 0 }],
            outputs: [
                { address_n: [44, 0, 0, 1, 1], amount: '10000000000' },
                { address: 'A-external', amount: '2499999900' },
            ],
        },
    ],
    ...resp,
});

export const signAndPush = [
    {
        description: 'ETH',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: ETH_ACCOUNT,
        },
        connect: [
            undefined, // estimateFee
            {
                success: true,
                payload: {
                    serializedTx: 'serializedABCD',
                },
            }, // ethereumSignTransaction
            undefined, // pushTransaction
            undefined, // getAccountInfo
            undefined, // blockchainEstimateFee
        ],
        result: {
            formValues: {
                selectedFee: undefined,
                outputs: [{ address: '', amount: '' }], // form was cleared
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'tx-sent', formattedAmount: '1 ETH' }, // BUG ?
                },
                {
                    type: 'mock-redirect',
                },
            ],
        },
    },
    {
        description: 'ETH failed',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: ETH_ACCOUNT,
        },
        result: {
            formValues: {
                selectedFee: undefined,
                outputs: [{ amount: '1' }], // form not cleared
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'sign-tx-error' },
                },
            ],
        },
    },
    {
        description: 'XRP',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: XRP_ACCOUNT,
        },
        connect: [
            undefined, // getAccountInfo address check
            {
                success: true,
                payload: {
                    serializedTx: 'serializedABCD',
                },
            },
        ],
        result: {
            formValues: {
                selectedFee: undefined,
                outputs: [{ address: '', amount: '' }], // form was cleared
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'tx-sent', formattedAmount: '1 XRP' },
                },
                {
                    type: 'mock-redirect',
                },
            ],
        },
    },
    {
        description: 'XRP failed',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: XRP_ACCOUNT,
        },
        result: {
            formValues: {
                selectedFee: undefined,
                outputs: [{ amount: '1' }], // form not cleared
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'sign-tx-error' },
                },
            ],
        },
    },
    {
        description: 'Success with: custom fee, 2 outputs, 0 utxo (ignored)',
        store: {
            send: {
                drafts: getDraft({
                    selectedFee: 'custom' as const,
                    outputs: [
                        {
                            ...DEFAULT_PAYMENT,
                            address: 'A',
                            amount: '1',
                        },
                        {
                            ...DEFAULT_PAYMENT,
                            address: 'B',
                            amount: '2',
                        },
                    ],
                }),
            },
        },
        connect: [
            getComposeResponse({
                payload: [
                    {
                        // normal fee level, not used in this test
                        type: 'final',
                        totalSpent: '2500000000',
                        fee: '200',
                        inputs: [],
                        outputs: [],
                    },
                    {
                        // custom fee level, used
                        type: 'final',
                        totalSpent: '2500000000', // 2200000000 are externals + fee
                        fee: '100',
                        inputs: [
                            { amount: '0', prev_hash: 'should not be used', prev_index: 0 },
                            { amount: '12500000000', prev_hash: UTXO.CC.txid, prev_index: 0 },
                        ],
                        outputs: [
                            { address_n: [44, 0, 0, 1, 1], amount: '10000000000' },
                            { address: 'A-external', amount: '2100000000' },
                            { address: '1-unused', amount: '100000000' },
                            { address: '2-used', amount: '100000000' },
                            { address: '1-change', amount: '100000000' },
                            { address: 'B-external', amount: '99999900' },
                        ],
                    },
                ],
            }),
            {
                success: true,
                payload: {
                    serializedTx: 'serializedABCD',
                    signedTransaction: {
                        txid: 'txid',
                        vin: [],
                        vout: [],
                    },
                },
            },
        ],
        result: {
            formValues: {
                selectedFee: undefined,
                outputs: [{ address: '', amount: '' }], // form was cleared
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'tx-sent', formattedAmount: '24.999999 BTC' },
                },
                {
                    type: accountsActions.updateAccount.type,
                    payload: {
                        // reduced balance
                        availableBalance: '97800000000',
                        formattedBalance: '978',
                        utxo: [
                            // new utxos created by this tx
                            {
                                address: '1-change',
                                amount: '100000000',
                                vout: 4,
                                txid: 'txid',
                                blockHeight: 0,
                                confirmations: 0,
                                path: "m/44'/0'/0'/1/0",
                            },
                            {
                                address: '2-used',
                                amount: '100000000',
                                vout: 3,
                                txid: 'txid',
                                blockHeight: 0,
                                confirmations: 0,
                                path: "m/44'/0'/0'/0/1",
                            },
                            {
                                address: '1-unused',
                                amount: '100000000',
                                vout: 2,
                                txid: 'txid',
                                blockHeight: 0,
                                confirmations: 0,
                                path: "m/44'/0'/0'/0/2",
                            },
                            {
                                address: '2-change',
                                amount: '10000000000',
                                vout: 0,
                                txid: 'txid',
                                blockHeight: 0,
                                confirmations: 0,
                                path: "m/44'/0'/0'/1/1",
                            },
                            // old utxo without used "utxoC"
                            UTXO['00'],
                            UTXO.AA,
                            UTXO.BB,
                            UTXO.DD,
                            UTXO.EE,
                        ],
                    },
                },
                {
                    type: 'mock-redirect',
                },
            ],
        },
    },
    {
        description: 'Error during signing',
        store: {
            send: {
                drafts: getDraft(),
            },
        },
        connect: [
            getComposeResponse(),
            {
                success: false,
                payload: {
                    error: 'signTx error',
                },
            },
        ],
        result: {
            formValues: {
                outputs: [{ address: 'A', amount: '1' }],
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'sign-tx-error' },
                },
            ],
        },
    },
    {
        description: 'Error during signing (cancelled)',
        store: {
            send: {
                drafts: getDraft(),
            },
        },
        connect: [
            getComposeResponse(),
            {
                success: false,
                payload: {
                    error: 'tx-cancelled',
                },
            },
        ],
        result: {
            formValues: {
                outputs: [{ address: 'A', amount: '1' }],
            },
            actions: [], // silent error - no toast
        },
    },
    {
        description: 'Error during pushing',
        store: {
            send: {
                drafts: getDraft(),
            },
        },
        connect: [
            getComposeResponse(),
            {
                success: true,
                payload: {
                    serializedTx: 'serializedABCD',
                },
            },
            {
                success: false,
                payload: {
                    error: 'pushTx error',
                },
            },
        ],
        result: {
            formValues: {
                outputs: [{ address: 'A', amount: '1' }],
            },
            actions: [
                {
                    type: notificationsActions.addToast.type,
                    payload: { type: 'sign-tx-error' },
                },
            ],
        },
    },
];

export const feeChange = [
    {
        description: 'BTC fee changes',
        store: {
            send: {
                drafts: getDraft(),
            },
            fees: {
                btc: {
                    minFee: 1,
                    maxFee: 100,
                    blockHeight: 1,
                    blockTime: 1,
                    // add more levels
                    levels: [
                        { label: 'high', feePerUnit: '40', blocks: 1 },
                        { label: 'normal', feePerUnit: '4', blocks: 1 },
                        { label: 'economy', feePerUnit: '1', blocks: 1 },
                    ],
                },
            },
        },
        connect: {
            success: false,
            payload: {
                success: false,
                payload: {
                    error: 'compose-response-is-irrelevant',
                },
            },
        },
        actionSequence: [
            {
                type: 'click',
                element: 'select-bar/high',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        selectedFee: 'high' as const,
                        feePerUnit: '',
                    },
                },
            },
            {
                type: 'click',
                element: 'select-bar/custom',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '40', // from high level
                    },
                },
            },
            {
                type: 'click',
                element: 'select-bar/custom',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '40', // from high level
                    },
                },
            },
            {
                type: 'click',
                element: 'select-bar/economy',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        selectedFee: 'economy' as const,
                        feePerUnit: '40', // did not changed
                    },
                },
            },
            {
                type: 'click',
                element: 'select-bar/custom',
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '1', // from low level
                    },
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: '', // reset value
                result: {
                    composeTransactionCalls: 1,
                    formValues: {
                        feePerUnit: '',
                    },
                    errors: {
                        feePerUnit: { type: 'required' },
                    },
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: '2', // add new custom
                result: {
                    composeTransactionCalls: 2,
                    formValues: {
                        feePerUnit: '2',
                    },
                    errors: {},
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: 'a', // try to add invalid
                result: {
                    composeTransactionCalls: 2,
                    formValues: {
                        feePerUnit: '2',
                    },
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: '00000', //  trigger error
                result: {
                    composeTransactionCalls: 2,
                    formValues: {
                        feePerUnit: '200000',
                    },
                },
            },
            {
                type: 'click',
                element: 'select-bar/normal',
                result: {
                    composeTransactionCalls: 3, // called after fee level change from custom with error
                    formValues: {
                        selectedFee: 'normal' as const,
                        feePerUnit: '',
                    },
                },
            },
        ],
        finalResult: {
            composeTransactionCalls: 3,
        },
    },
    {
        description: 'ETH fee limit changes',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: ETH_ACCOUNT,
        },
        // blockchainEstimateFee responses
        connect: [
            {
                success: false,
                payload: {
                    error: 'irrelevant',
                },
            },
            {
                success: true,
                payload: {
                    levels: [{ feeLimit: '41000' }],
                },
            },
            {
                success: true,
                payload: {
                    levels: [{ feeLimit: '21009' }],
                },
            },
            {
                success: true,
                payload: { levels: [{}] },
            },
        ],
        actionSequence: [
            {
                type: 'click',
                element: 'select-bar/custom',
                result: {
                    estimateFeeCalls: 1,
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '3.3',
                        feeLimit: '21000', // default
                        estimatedFeeLimit: '21000',
                    },
                },
            },
            {
                type: 'input',
                element: 'outputs.0.amount',
                value: '.1',
                expectRerender: true, // caused by feeLimit set in useFees sub hook
                result: {
                    estimateFeeCalls: 2,
                    formValues: {
                        outputs: [{ amount: '1.1' }],
                        feePerUnit: '3.3',
                        feeLimit: '41000', // feeLimit is switched automatically
                        estimatedFeeLimit: '41000',
                    },
                },
            },
            // trigger "gas limit below recommended" error
            {
                type: 'input',
                element: 'feeLimit',
                value: '{backspace}',
                result: {
                    estimateFeeCalls: 2,
                    formValues: {
                        outputs: [{ amount: '1.1' }],
                        feePerUnit: '3.3',
                        feeLimit: '4100',
                        estimatedFeeLimit: '41000',
                    },
                    errors: {
                        feeLimit: { type: 'feeLimit' },
                    },
                },
            },
            // im not currently able to click on error button
            // increasing fee level to accepted value
            {
                type: 'input',
                element: 'feeLimit',
                value: '0',
                expectRerender: true, // caused by feeLimit set in useFees sub hook
                result: {
                    estimateFeeCalls: 3,
                    formValues: {
                        outputs: [{ amount: '1.1' }],
                        selectedFee: 'custom' as const,
                        feePerUnit: '3.3',
                        feeLimit: '21009', // feeLimit is switched automatically again
                        estimatedFeeLimit: '21009',
                    },
                    composedLevels: {
                        normal: {
                            type: 'final',
                            feeLimit: '21009',
                            estimatedFeeLimit: '21009',
                        },
                        custom: {
                            type: 'final',
                            feeLimit: '41000', // custom composed tx with higher level (as requested)
                            estimatedFeeLimit: '21009',
                        },
                    },
                    errors: {
                        feeLimit: undefined,
                    }, // no errors
                },
            },
            // reset value
            {
                type: 'input',
                element: 'feeLimit',
                value: '',
                result: {
                    estimateFeeCalls: 3,
                    formValues: {
                        feeLimit: '',
                        estimatedFeeLimit: '21009',
                    },
                    composedLevels: undefined, // no levels
                    errors: {
                        feeLimit: { type: 'required' }, // not set error
                    },
                },
            },
            {
                type: 'input',
                element: 'feeLimit',
                value: '21', // too low
                result: {
                    formValues: {
                        feeLimit: '21',
                    },
                    errors: {
                        feeLimit: { type: 'feeLimit' },
                    },
                },
            },
            {
                type: 'input',
                element: 'feeLimit',
                value: 'a', // try to add invalid
                result: {
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '3.3',
                        feeLimit: '21',
                        estimatedFeeLimit: '21009',
                    },
                },
            },
            {
                type: 'click',
                element: 'send/open-ethereum-data',
            },
            {
                type: 'click',
                element: 'send/close-ethereum-data',
            },
            // switch back to normal
            {
                type: 'click',
                element: 'select-bar/normal',
                result: {
                    estimateFeeCalls: 4, // called after fee level change
                    formValues: {
                        selectedFee: 'normal' as const,
                        feePerUnit: '',
                        feeLimit: '',
                        estimatedFeeLimit: undefined, // because of missing response from connect
                    },
                    composedLevels: {
                        normal: {
                            type: 'final',
                            fee: '69300000000000',
                            feePerByte: '3.3',
                            feeLimit: '21000', // default
                        },
                        custom: undefined, // no custom level build
                    },
                    errors: {
                        feeLimit: undefined, // no levels,
                    }, // no errors
                },
            },
        ],
        finalResult: {
            estimateFeeCalls: 4,
        },
    },
    {
        description: 'XRP fee changes',
        store: {
            send: {
                drafts: getDraft(),
            },
            selectedAccount: XRP_ACCOUNT,
        },
        // getAccountInfo responses
        connect: [
            {
                success: true,
                payload: {
                    empty: true,
                    misc: { reserve: '20000000' },
                },
            },
        ],
        actionSequence: [
            {
                type: 'click',
                element: 'select-bar/custom',
                result: {
                    getAccountInfoCalls: 1,
                    formValues: {
                        selectedFee: 'custom' as const,
                        feePerUnit: '12',
                    },
                    composedLevels: {
                        normal: {
                            type: 'error', // not enough to cover reserve
                        },
                    },
                    errors: {
                        outputs: [
                            {
                                amount: { type: 'compose' }, // AMOUNT_IS_LESS_THAN_RESERVE
                            },
                        ],
                    },
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: '', // reset value
                result: {
                    getAccountInfoCalls: 1,
                    formValues: {
                        feePerUnit: '',
                    },
                    composedLevels: undefined,
                    errors: {
                        feePerUnit: { type: 'required' },
                    },
                },
            },
            {
                type: 'input',
                element: 'feePerUnit',
                value: '10',
                result: {
                    getAccountInfoCalls: 2,
                    formValues: {
                        feePerUnit: '10',
                    },
                    composedLevels: {
                        normal: {
                            type: 'final',
                            fee: '12', // default
                        },
                        custom: {
                            type: 'final',
                            fee: '10', // default
                        },
                    },
                    errors: {
                        feePerUnit: undefined,
                    },
                },
            },
        ],
        finalResult: {
            getAccountInfoCalls: 2,
        },
    },
];

export const amountUnitChange = [
    {
        description: 'compose with satoshi AmountUnit',
        connect: [
            {
                success: true,
                payload: [
                    {
                        type: 'final',
                        max: '100000000',
                    },
                ],
            },
        ],
        store: {
            bitcoinAmountUnit: PROTO.AmountUnit.SATOSHI,
        },
        actions: [{ type: 'input', element: 'outputs.0.amount', value: '111' }],
        finalResult: {
            composeTransactionCalls: 1,
            formValues: {
                selectedFee: undefined,
                outputs: [{ address: '', amount: '111' }],
            },
        },
    },
    {
        description: 'setMax with satoshi AmountUnit',
        connect: [
            {
                success: true,
                payload: [
                    {
                        type: 'nonfinal',
                        max: '100000000',
                    },
                ],
            },
        ],
        actions: [
            {
                type: 'hover',
                element: 'outputs.0.amount',
            },
            {
                type: 'click',
                element: 'outputs.0.setMax',
                result: {
                    composeTransactionCalls: 1,
                    composeTransactionParams: {
                        outputs: [{ type: 'send-max-noaddress' }],
                    },
                    formValues: {
                        setMaxOutputId: 0,
                        outputs: [{ address: '', amount: '1', fiat: '1.00' }],
                    },
                    composedLevels: {
                        normal: { type: 'nonfinal' },
                    },
                },
            },
        ],
    },
];
