import styled from 'styled-components';

import { borders, spacingsPx, spacings } from '@trezor/theme';
import { selectCurrentFiatRates } from '@suite-common/wallet-core';
import {
    getAccountTotalStakingBalance,
    getAccountTokensFiatBalance,
} from '@suite-common/wallet-utils';
import { Column } from '@trezor/components';

import { Account } from 'src/types/wallet';
import { useSelector } from 'src/hooks/suite';
import { selectLocalCurrency } from 'src/reducers/wallet/settingsReducer';
import { selectRouteName } from 'src/reducers/suite/routerReducer';

import { AccountItem } from './AccountItem/AccountItem';
import { useIsSidebarCollapsed } from '../../../suite/layouts/SuiteLayout/Sidebar/utils';

const Section = styled.div<{ $selected?: boolean; $isSidebarCollapsed?: boolean }>`
    display: flex;
    flex-direction: column;
    position: relative;
    border-radius: ${borders.radii.md};

    outline: 1px solid
        ${({ theme, $selected }) => ($selected ? theme.borderElevation0 : 'transparent')};
    padding: ${spacingsPx.xxs};
    margin: 0 -${spacingsPx.xxs};

    &::before {
        content: '';
        position: absolute;
        top: 24px;
        bottom: 28px;
        left: ${({ $isSidebarCollapsed }) => ($isSidebarCollapsed ? '50%' : '24px')};
        border-left: 2px dotted ${({ theme }) => theme.borderDashed};
    }
`;

interface AccountItemsGroupProps {
    account: Account;
    accountLabel?: string;
    selected: boolean;
    showStaking: boolean;
    tokens?: Account['tokens'];
    dataTestKey?: string;
}

export const AccountItemsGroup = ({
    account,
    accountLabel,
    selected,
    showStaking,
    tokens,
    dataTestKey,
}: AccountItemsGroupProps) => {
    const isSidebarCollapsed = useIsSidebarCollapsed();
    const stakingBalance = getAccountTotalStakingBalance(account);

    const routeName = useSelector(selectRouteName);
    const localCurrency = useSelector(selectLocalCurrency);
    const rates = useSelector(selectCurrentFiatRates);

    const tokensFiatBalance = getAccountTokensFiatBalance(account, localCurrency, rates, tokens);

    const tokensRoutes = ['wallet-tokens', 'wallet-tokens-hidden'];

    return (
        <Section $selected={selected} $isSidebarCollapsed={isSidebarCollapsed}>
            <Column gap={spacings.xxs}>
                <AccountItem
                    type="coin"
                    account={account}
                    isSelected={
                        selected &&
                        (routeName === 'wallet-index' ||
                            (routeName === 'wallet-staking' && !showStaking))
                    }
                    accountLabel={accountLabel}
                    formattedBalance={account.formattedBalance}
                    isGroup
                    isGroupSelected={selected}
                    dataTestKey={dataTestKey}
                />
                {showStaking && (
                    <AccountItem
                        account={account}
                        type="staking"
                        isSelected={selected && routeName === 'wallet-staking'}
                        formattedBalance={stakingBalance ?? '0'}
                        isGroup
                        isGroupSelected={selected}
                        dataTestKey={`${dataTestKey}/staking`}
                    />
                )}
                {!!tokens?.length && (
                    <AccountItem
                        account={account}
                        type="tokens"
                        isSelected={selected && tokensRoutes.includes(routeName || '')}
                        formattedBalance={account.formattedBalance}
                        isGroup
                        isGroupSelected={selected}
                        customFiatValue={tokensFiatBalance}
                        tokens={tokens}
                        dataTestKey={`${dataTestKey}/tokens`}
                    />
                )}
            </Column>
        </Section>
    );
};
