import styled from 'styled-components';

import { Button, Tooltip } from '@trezor/components';
import { BigNumber } from '@trezor/utils/src/bigNumber';
import { MIN_ETH_AMOUNT_FOR_STAKING } from '@suite-common/wallet-constants';
import { getNetworkDisplaySymbol, NetworkSymbol } from '@suite-common/wallet-config';

import { Translation } from 'src/components/suite';

const Flex = styled.div`
    display: flex;
    gap: 4px;
`;

// eslint-disable-next-line local-rules/no-override-ds-component
const TinyButton = styled(Button).attrs(props => ({
    ...props,
    variant: 'tertiary',
    type: 'button',
    size: 'tiny',
}))``;

interface FormFractionButtonsProps {
    setRatioAmount: (divisor: number) => void;
    setMax: () => void;
    isDisabled?: boolean;
    symbol: NetworkSymbol;
    totalAmount?: number | string;
    decimals?: number;
}

export const FormFractionButtons = ({
    setRatioAmount,
    setMax,
    isDisabled = false,
    symbol,
    totalAmount,
    decimals,
}: FormFractionButtonsProps) => {
    const isFractionButtonDisabled = (divisor: number) => {
        if (!totalAmount || !decimals) return false;

        return new BigNumber(totalAmount)
            .dividedBy(divisor)
            .decimalPlaces(decimals)
            .lte(MIN_ETH_AMOUNT_FOR_STAKING);
    };
    const is10PercentDisabled = isDisabled || isFractionButtonDisabled(10);
    const is25PercentDisabled = isDisabled || isFractionButtonDisabled(4);
    const is50PercentDisabled = isDisabled || isFractionButtonDisabled(2);
    const isMaxDisabled =
        isDisabled || new BigNumber(totalAmount || '0').lt(MIN_ETH_AMOUNT_FOR_STAKING);

    const displaySymbol = getNetworkDisplaySymbol(symbol);

    return (
        <Flex>
            <Tooltip
                content={
                    is10PercentDisabled && (
                        <Translation
                            id="TR_STAKE_MIN_AMOUNT_TOOLTIP"
                            values={{
                                amount: MIN_ETH_AMOUNT_FOR_STAKING.toString(),
                                networkSymbol: displaySymbol,
                            }}
                        />
                    )
                }
                cursor="pointer"
            >
                <TinyButton isDisabled={is10PercentDisabled} onClick={() => setRatioAmount(10)}>
                    10%
                </TinyButton>
            </Tooltip>
            <Tooltip
                content={
                    is25PercentDisabled && (
                        <Translation
                            id="TR_STAKE_MIN_AMOUNT_TOOLTIP"
                            values={{
                                amount: MIN_ETH_AMOUNT_FOR_STAKING.toString(),
                                networkSymbol: displaySymbol,
                            }}
                        />
                    )
                }
                cursor="pointer"
            >
                <TinyButton isDisabled={is25PercentDisabled} onClick={() => setRatioAmount(4)}>
                    25%
                </TinyButton>
            </Tooltip>
            <Tooltip
                content={
                    is50PercentDisabled && (
                        <Translation
                            id="TR_STAKE_MIN_AMOUNT_TOOLTIP"
                            values={{
                                amount: MIN_ETH_AMOUNT_FOR_STAKING.toString(),
                                networkSymbol: displaySymbol,
                            }}
                        />
                    )
                }
                cursor="pointer"
            >
                <TinyButton isDisabled={is50PercentDisabled} onClick={() => setRatioAmount(2)}>
                    50%
                </TinyButton>
            </Tooltip>
            <Tooltip
                content={
                    isMaxDisabled && (
                        <Translation
                            id="TR_STAKE_MIN_AMOUNT_TOOLTIP"
                            values={{
                                amount: MIN_ETH_AMOUNT_FOR_STAKING.toString(),
                                networkSymbol: displaySymbol,
                            }}
                        />
                    )
                }
                cursor="pointer"
            >
                <TinyButton isDisabled={isDisabled || isMaxDisabled} onClick={setMax}>
                    <Translation id="TR_STAKE_MAX" />
                </TinyButton>
            </Tooltip>
        </Flex>
    );
};
