import {
    ExchangeListResponse,
    ExchangeProviderInfo,
    ExchangeTradeQuoteRequest,
    ExchangeTrade,
    CryptoId,
} from 'invity-api';

import { Account } from 'src/types/wallet';
import { Dispatch } from 'src/types/suite';
import invityAPI from 'src/services/suite/invityAPI';
import * as modalActions from 'src/actions/suite/modalActions';
import { verifyAddress as verifyExchangeAddress } from 'src/actions/wallet/coinmarket/coinmarketCommonActions';

import { COINMARKET_EXCHANGE, COINMARKET_COMMON } from './constants';

export interface ExchangeInfo {
    exchangeList?: ExchangeListResponse;
    providerInfos: { [name: string]: ExchangeProviderInfo };
    buySymbols: Set<CryptoId>;
    sellSymbols: Set<CryptoId>;
}

export type CoinmarketExchangeAction =
    | { type: typeof COINMARKET_EXCHANGE.SAVE_EXCHANGE_INFO; exchangeInfo: ExchangeInfo }
    | {
          type: typeof COINMARKET_EXCHANGE.SAVE_QUOTE_REQUEST;
          request: ExchangeTradeQuoteRequest;
      }
    | { type: typeof COINMARKET_EXCHANGE.SAVE_TRANSACTION_ID; transactionId: string | undefined }
    | { type: typeof COINMARKET_EXCHANGE.SET_IS_FROM_REDIRECT; isFromRedirect: boolean }
    | { type: typeof COINMARKET_EXCHANGE.VERIFY_ADDRESS; addressVerified: string }
    | {
          type: typeof COINMARKET_EXCHANGE.SAVE_QUOTES;
          quotes: ExchangeTrade[];
      }
    | {
          type: typeof COINMARKET_EXCHANGE.SAVE_QUOTE;
          quote: ExchangeTrade | undefined;
      }
    | { type: typeof COINMARKET_EXCHANGE.SET_COINMARKET_ACCOUNT; account: Account | undefined }
    | {
          type: typeof COINMARKET_COMMON.SAVE_TRADE;
          date: string;
          key?: string;
          tradeType: 'exchange';
          data: ExchangeTrade;
          account: {
              symbol: Account['symbol'];
              descriptor: Account['descriptor'];
              accountIndex: Account['index'];
              accountType: Account['accountType'];
          };
      };

export const loadExchangeInfo = async (): Promise<ExchangeInfo> => {
    const exchangeList = await invityAPI.getExchangeList();

    if (!exchangeList || exchangeList.length === 0) {
        return { providerInfos: {}, buySymbols: new Set(), sellSymbols: new Set() };
    }

    const providerInfos: { [name: string]: ExchangeProviderInfo } = {};
    exchangeList.forEach(e => (providerInfos[e.name] = e));

    // merge symbols supported by at least one partner
    const buySymbolsArray: CryptoId[] = [];
    const sellSymbolsArray: CryptoId[] = [];
    exchangeList.forEach(p => {
        if (p.buyTickers) {
            buySymbolsArray.push(...p.buyTickers);
        }
        if (p.sellTickers) {
            sellSymbolsArray.push(...p.sellTickers);
        }
    });

    const buySymbols = new Set<CryptoId>(buySymbolsArray);
    const sellSymbols = new Set<CryptoId>(sellSymbolsArray);

    return {
        exchangeList,
        providerInfos,
        buySymbols,
        sellSymbols,
    };
};

export const saveExchangeInfo = (exchangeInfo: ExchangeInfo): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SAVE_EXCHANGE_INFO,
    exchangeInfo,
});

// this is only a wrapper for `openDeferredModal` since it doesn't work with `bindActionCreators`
// used in useCoinmarketExchangeOffers
export const openCoinmarketExchangeConfirmModal =
    (provider?: string, isDex?: boolean, fromCryptoCurrency?: string, toCryptoCurrency?: string) =>
    (dispatch: Dispatch) =>
        dispatch(
            modalActions.openDeferredModal({
                type: isDex ? 'coinmarket-exchange-dex-terms' : 'coinmarket-exchange-terms',
                provider,
                fromCryptoCurrency,
                toCryptoCurrency,
            }),
        );

export const saveTrade = (
    exchangeTrade: ExchangeTrade,
    account: Account,
    date: string,
): CoinmarketExchangeAction => ({
    type: COINMARKET_COMMON.SAVE_TRADE,
    tradeType: 'exchange',
    key: exchangeTrade.orderId,
    date,
    data: exchangeTrade,
    account: {
        descriptor: account.descriptor,
        symbol: account.symbol,
        accountType: account.accountType,
        accountIndex: account.index,
    },
});

export const saveQuoteRequest = (request: ExchangeTradeQuoteRequest): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SAVE_QUOTE_REQUEST,
    request,
});

export const saveTransactionId = (transactionId: string | undefined): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SAVE_TRANSACTION_ID,
    transactionId,
});

export const saveQuotes = (quotes: ExchangeTrade[]): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SAVE_QUOTES,
    quotes,
});

export const saveSelectedQuote = (quote: ExchangeTrade | undefined): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SAVE_QUOTE,
    quote,
});

export const setIsFromRedirect = (isFromRedirect: boolean): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SET_IS_FROM_REDIRECT,
    isFromRedirect,
});

export const verifyAddress = (account: Account, address?: string, path?: string) =>
    verifyExchangeAddress(account, address, path, COINMARKET_EXCHANGE.VERIFY_ADDRESS);

export const setCoinmarketExchangeAccount = (
    account: Account | undefined,
): CoinmarketExchangeAction => ({
    type: COINMARKET_EXCHANGE.SET_COINMARKET_ACCOUNT,
    account,
});
