import styled, { css } from 'styled-components';

import { AssetLogo, useElevation } from '@trezor/components';
import { getCoingeckoId, NetworkSymbol } from '@suite-common/wallet-config';
import { TokenInfo } from '@trezor/connect';
import { getContractAddressForNetworkSymbol } from '@suite-common/wallet-utils';
import { borders, Elevation, mapElevationToBackground, mapElevationToBorder } from '@trezor/theme';

export type TokenIconSetProps = {
    symbol: NetworkSymbol;
    tokens: TokenInfo[];
};

const IconContainer = styled.div<{ $length: number }>`
    width: 24px;
    justify-content: center;
    display: flex;
    align-items: center;
    ${({ $length }) =>
        $length > 1 &&
        css`
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(${$length > 1 ? '1px' : '6px'}, 6px));
            direction: rtl;
            justify-items: center;
        `}
`;

const TokenIconPlaceholder = styled.div<{ $elevation: Elevation }>`
    width: 20px;
    height: 20px;
    border-radius: ${borders.radii.full};
    border: 1px solid ${mapElevationToBorder};
    background: ${mapElevationToBackground};
`;

/**
 * @param tokens - provide already sorted tokens (for example by fiat value).
 */
export const TokenIconSet = ({ symbol, tokens }: TokenIconSetProps) => {
    const { elevation } = useElevation();
    const { length } = tokens;

    if (length === 0) {
        return null;
    }

    const visibleTokens = tokens.slice(0, 3).reverse();

    const coingeckoId = getCoingeckoId(symbol);

    return (
        <IconContainer $length={length}>
            {length > 3 && <TokenIconPlaceholder $elevation={elevation} />}
            {visibleTokens.map(token => (
                <AssetLogo
                    key={token.contract}
                    size={20}
                    coingeckoId={coingeckoId ?? ''}
                    contractAddress={getContractAddressForNetworkSymbol(symbol, token.contract)}
                    placeholder={token.symbol?.toUpperCase() ?? ''}
                    placeholderWithTooltip={false}
                />
            ))}
        </IconContainer>
    );
};
