import { test, expect, Page, devices } from '@playwright/test';

import { ensureDirectoryExists } from '@trezor/node-utils';
import { TrezorUserEnvLink } from '@trezor/trezor-user-env-link';

import { formatUrl, log, waitAndClick } from '../support/helpers';

const url = process.env.URL || 'http://localhost:8088/';

let dir: string;
let popup: Page;

const iPhone = devices['iPhone 13 Pro'];
const android = devices['Pixel 5'];
const safari = devices['Desktop Safari'];
test.beforeAll(async () => {
    dir = await ensureDirectoryExists('./e2e/screenshots/unsupported-browser');
});

const openPopup = async (page: Page) =>
    (
        await Promise.all([
            // It is important to call waitForEvent before click to set up waiting.
            page.waitForEvent('popup'),
            // Opens popup.
            page.click("div[data-testid='@api-playground/collapsible-box']"),
            page.click("button[data-testid='@submit-button']"),
        ])
    )[0];

test.beforeAll(async () => {
    await TrezorUserEnvLink.connect();
    await TrezorUserEnvLink.stopBridge();
    await TrezorUserEnvLink.stopEmu();
});

test('unsupported browser', async ({ browser }) => {
    const context = await browser.newContext({
        ...safari,
    });
    const page = await context.newPage();
    await page.goto(formatUrl(url, `methods/bitcoin/getPublicKey/`));
    await waitAndClick(page, ['@api-playground/collapsible-box']);
    await page.waitForSelector("button[data-testid='@submit-button']", { state: 'visible' });
    popup = await openPopup(page);
    await popup.waitForSelector('text=Unsupported browser');
    await popup.screenshot({ path: `${dir}/browser-not-supported.png` });
    await popup.close({ runBeforeUnload: true });
    await page.close();
    await context.close();
});

test('outdated browser', async ({ browser }) => {
    const context = await browser.newContext({
        userAgent:
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:100.0) Gecko/20100101 Firefox/50.0',
    });
    const page = await context.newPage();
    await page.goto(formatUrl(url, `methods/bitcoin/getPublicKey/`));
    await waitAndClick(page, ['@api-playground/collapsible-box']);
    await page.waitForSelector("button[data-testid='@submit-button']", { state: 'visible' });
    popup = await openPopup(page);
    await popup.waitForLoadState('load');
    await popup.waitForSelector('text=Outdated browser');
    // no react is rendering yet only browser check
    expect(await popup.locator('#reactRenderIn').count()).toEqual(0);
    await popup.screenshot({ path: `${dir}/outdated-browser-1.png` });
    await popup.click('text=I acknowledge and wish to continue');
    // only after this check react renders
    await popup.waitForSelector('#reactRenderIn');
    log('clicking on analytics continue button');
    await waitAndClick(popup, ['@analytics/continue-button']);
    // In Firefox it should display Install Bridge page.
    await popup.getByRole('heading', { name: "Browser can't communicate with device" });
    await popup.close({ runBeforeUnload: true });
    await page.close();
    await context.close();
});

// test mobile browsers
test(`env: web, device: mobile/iPhone => not allowed `, async ({ browser }) => {
    const context = await browser.newContext({
        ...iPhone,
    });
    const page = await context.newPage();
    await page.goto(formatUrl(url, `methods/bitcoin/getPublicKey/`));

    popup = await openPopup(page);
    // unfortunately webusb now does not work for connect-popup, so mobile chrome won't run even if it technically could
    await popup.waitForSelector('text=Smartphones not supported yet');
    await popup.screenshot({ path: `${dir}/mobile-iphone-not-supported.png` });

    await popup.click('text=Close');
    await page.close();
    await context.close();
});

test(`env: web, device: mobile/Android => allowed `, async ({ browser }) => {
    const context = await browser.newContext({
        ...android,
    });
    const page = await context.newPage();
    await page.goto(formatUrl(url, `methods/bitcoin/getPublicKey/`));

    popup = await openPopup(page);

    await popup.waitForSelector('text=Connect Trezor to continue');
    await popup.screenshot({ path: `${dir}/mobile-android-supported.png` });

    await page.close();
    await context.close();
});
