import { ReactNode } from 'react';

import styled, { css } from 'styled-components';

import {
    Elevation,
    mapElevationToBackground,
    mapElevationToBorder,
    nextElevation,
} from '@trezor/theme';

import { useElevation } from '../ElevationContext/ElevationContext';
import { useTable } from './Table';
import { useTableHeader } from './TableHeader';

export const Row = styled.tr<{
    $elevation: Elevation;
    $isCollapsed: boolean;
    $isHighlighted: boolean;
    $isHeader: boolean;
    $hasBorderTop: boolean;
}>`
    ${({ $hasBorderTop, theme, $elevation }) =>
        $hasBorderTop &&
        css`
            border-top: 1px solid ${mapElevationToBorder({ theme, $elevation })};
        `}

    &:first-child {
        border-top: 0;
    }

    ${({ $isHighlighted, theme, $elevation, $isHeader }) =>
        $isHighlighted &&
        !$isHeader &&
        css`
            &:hover {
                background-color: ${mapElevationToBackground({
                    theme,
                    $elevation: nextElevation[$elevation],
                })};

                & > td:first-child {
                    background: linear-gradient(
                        to right,
                        ${mapElevationToBackground({
                                theme,
                                $elevation: nextElevation[$elevation],
                            })}
                            90%,
                        rgba(0 0 0 / 0%)
                    );
                }
            }
        `}

    ${({ $isCollapsed }) =>
        $isCollapsed &&
        css`
            visibility: collapse;
            border-top: 1;
            opacity: 0;
        `}

        ${({ onClick }) =>
        onClick &&
        css`
            &:hover {
                cursor: pointer;
            }
        `}
`;

export interface TableRowProps {
    children: ReactNode;
    isCollapsed?: boolean;
    isHighlightedOnHover?: boolean;
    onClick?: () => void;
    onHover?: (isHovering: boolean) => void;
    hasBorderTop?: boolean;
}

export const TableRow = ({
    children,
    isCollapsed = false,
    onClick,
    onHover,
    isHighlightedOnHover,
    hasBorderTop,
}: TableRowProps) => {
    const { elevation } = useElevation();
    const isHeader = useTableHeader();
    const { isRowHighlightedOnHover, hasBorders } = useTable();

    return (
        <Row
            $elevation={elevation}
            $isCollapsed={isCollapsed}
            $isHighlighted={isHighlightedOnHover ?? isRowHighlightedOnHover}
            $isHeader={isHeader}
            $hasBorderTop={hasBorderTop ?? hasBorders}
            onClick={onClick}
            onMouseEnter={() => onHover?.(true)}
            onMouseLeave={() => onHover?.(false)}
        >
            {children}
        </Row>
    );
};
