import React, { FC, Suspense, useContext, useId } from 'react';
import styled, { useTheme } from 'styled-components';

const TonkeeperLottieIcon = React.lazy(() => import('./lottie/TonkeeperLottie'));

export const DarkThemeContext = React.createContext<boolean>(true);

const TonkeeperSvgIcon: FC<{ size?: string }> = ({ size = '128' }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width={size}
            height={size}
            viewBox="0 0 128 128"
            fill="none"
        >
            <path opacity="0.5" d="M64 58L118 34L64 118V58Z" fill="currentColor" />
            <path opacity="0.75" d="M64 58L10 34L64 118V58Z" fill="currentColor" />
            <path d="M64 58L10 34L64 10L118 34L64 58Z" fill="currentColor" />
        </svg>
    );
};
export const TonkeeperIcon: FC<{
    width?: string;
    height?: string;
    loop?: boolean;
}> = ({ width = '128', height = '128', loop = false }) => {
    const isDark = useContext(DarkThemeContext);
    if (isDark) {
        return (
            <Suspense fallback={<div style={{ width, height }}></div>}>
                <TonkeeperLottieIcon width={width} height={height} loop={loop} />
            </Suspense>
        );
    } else {
        return <TonkeeperSvgIcon size={width} />;
    }
};

export const ChevronLeftIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="1rem"
            height="1rem"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                d="M10.3075 3.50173C10.5846 3.19385 10.5596 2.71963 10.2517 2.44254C9.94384 2.16544 9.46962 2.1904 9.19253 2.49828L4.69253 7.49828C4.43582 7.78351 4.43582 8.2165 4.69253 8.50173L9.19253 13.5017C9.46962 13.8096 9.94384 13.8346 10.2517 13.5575C10.5596 13.2804 10.5846 12.8062 10.3075 12.4983L6.25902 8.00001L10.3075 3.50173Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ChevronRightIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="1rem"
            height="1rem"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                d="M6.31997 2.38564C5.98063 2.62318 5.8981 3.09082 6.13564 3.43016L9.33457 8.00006L6.13564 12.57C5.8981 12.9093 5.98063 13.377 6.31997 13.6145C6.6593 13.852 7.12695 13.7695 7.36449 13.4302L10.8645 8.43016C11.0453 8.17192 11.0453 7.82821 10.8645 7.56997L7.36449 2.56997C7.12695 2.23063 6.6593 2.1481 6.31997 2.38564Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ChevronDownIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                d="M3.69495 6.64626C3.36148 6.40055 2.89197 6.47168 2.64626 6.80515C2.40055 7.13861 2.47168 7.60813 2.80515 7.85384L7.55515 11.3538C7.81972 11.5488 8.18038 11.5488 8.44495 11.3538L13.1949 7.85384C13.5284 7.60813 13.5996 7.13861 13.3538 6.80515C13.1081 6.47168 12.6386 6.40055 12.3052 6.64626L8.00005 9.81844L3.69495 6.64626Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const DoneIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M14.2803 3.22065C14.5732 3.51354 14.5732 3.98841 14.2803 4.28131L6.28033 12.2813C5.98744 12.5742 5.51256 12.5742 5.21967 12.2813L1.71967 8.78131C1.42678 8.48841 1.42678 8.01354 1.71967 7.72065C2.01256 7.42775 2.48744 7.42775 2.78033 7.72065L5.75 10.6903L13.2197 3.22065C13.5126 2.92775 13.9874 2.92775 14.2803 3.22065Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CheckIcon: FC<{ width?: string; height?: string }> = ({
    width = '28',
    height = '28'
}) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width={width}
            height={height}
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                d="M22.2071 9.70711C22.5976 9.31658 22.5976 8.68342 22.2071 8.29289C21.8166 7.90237 21.1834 7.90237 20.7929 8.29289L10.5 18.5858L6.70711 14.7929C6.31658 14.4024 5.68342 14.4024 5.29289 14.7929C4.90237 15.1834 4.90237 15.8166 5.29289 16.2071L9.79289 20.7071C10.1834 21.0976 10.8166 21.0976 11.2071 20.7071L22.2071 9.70711Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CloseIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
            color={theme[color || 'iconSecondary']}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2.46967 2.46967C2.76256 2.17678 3.23744 2.17678 3.53033 2.46967L8 6.93934L12.4697 2.46967C12.7626 2.17678 13.2374 2.17678 13.5303 2.46967C13.8232 2.76256 13.8232 3.23744 13.5303 3.53033L9.06066 8L13.5303 12.4697C13.8232 12.7626 13.8232 13.2374 13.5303 13.5303C13.2374 13.8232 12.7626 13.8232 12.4697 13.5303L8 9.06066L3.53033 13.5303C3.23744 13.8232 2.76256 13.8232 2.46967 13.5303C2.17678 13.2374 2.17678 12.7626 2.46967 12.4697L6.93934 8L2.46967 3.53033C2.17678 3.23744 2.17678 2.76256 2.46967 2.46967Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const XmarkIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8 16C12.4183 16 16 12.4183 16 8C16 3.58172 12.4183 0 8 0C3.58172 0 0 3.58172 0 8C0 12.4183 3.58172 16 8 16ZM5.78033 4.71967C5.48744 4.42678 5.01256 4.42678 4.71967 4.71967C4.42678 5.01256 4.42678 5.48744 4.71967 5.78033L6.93934 8L4.71967 10.2197C4.42678 10.5126 4.42678 10.9874 4.71967 11.2803C5.01256 11.5732 5.48744 11.5732 5.78033 11.2803L8 9.06066L10.2197 11.2803C10.5126 11.5732 10.9874 11.5732 11.2803 11.2803C11.5732 10.9874 11.5732 10.5126 11.2803 10.2197L9.06066 8L11.2803 5.78033C11.5732 5.48744 11.5732 5.01256 11.2803 4.71967C10.9874 4.42678 10.5126 4.42678 10.2197 4.71967L8 6.93934L5.78033 4.71967Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ClockIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <rect width="16" height="16" rx="8" fill="currentColor" />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8 4.25C8.41421 4.25 8.75 4.58579 8.75 5V7.68934L10.5303 9.46967C10.8232 9.76256 10.8232 10.2374 10.5303 10.5303C10.2374 10.8232 9.76256 10.8232 9.46967 10.5303L7.46967 8.53033C7.32902 8.38968 7.25 8.19891 7.25 8V5C7.25 4.58579 7.58579 4.25 8 4.25Z"
                fill="white"
            />
        </svg>
    );
};

export const GearIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="128"
            height="128"
            viewBox="0 0 128 128"
            fill="none"
        >
            <path
                opacity="0.32"
                d="M84 64C84 75.0457 75.0457 84 64 84C52.9543 84 44 75.0457 44 64C44 52.9543 52.9543 44 64 44C75.0457 44 84 52.9543 84 64Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M28.3199 27.8L28.3199 27.8L14.8099 51.2C12.1177 55.8631 10.7715 58.1946 10.2447 60.6734C9.77847 62.8666 9.77847 65.1334 10.2447 67.3266C10.7715 69.8054 12.1177 72.1369 14.8099 76.8L28.3199 100.2C31.0121 104.863 32.3582 107.195 34.2415 108.89C35.9078 110.391 37.8708 111.524 40.0033 112.217C42.4134 113 45.1057 113 50.4901 113H77.5101C82.8946 113 85.5868 113 87.997 112.217C90.1294 111.524 92.0925 110.391 93.7588 108.89C95.642 107.195 96.9881 104.863 99.6804 100.2L113.19 76.8L113.19 76.7999C115.883 72.1369 117.229 69.8053 117.756 67.3266C118.222 65.1334 118.222 62.8666 117.756 60.6734C117.229 58.1946 115.883 55.8631 113.19 51.2001L113.19 51.2L99.6804 27.8C96.9881 23.1369 95.642 20.8054 93.7588 19.1097C92.0925 17.6093 90.1294 16.476 87.997 15.7831C85.5868 15 82.8946 15 77.5101 15H50.4901C45.1057 15 42.4134 15 40.0033 15.7831C37.8708 16.476 35.9078 17.6093 34.2415 19.1097C32.3582 20.8054 31.0121 23.1369 28.3199 27.8ZM63.9999 84C75.0456 84 83.9999 75.0457 83.9999 64C83.9999 52.9543 75.0456 44 63.9999 44C52.9542 44 43.9999 52.9543 43.9999 64C43.9999 75.0457 52.9542 84 63.9999 84Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CheckMarkIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="128"
            height="128"
            viewBox="0 0 128 128"
            fill="none"
        >
            <path
                opacity="0.32"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M86.8284 48.1716C88.3905 49.7337 88.3905 52.2663 86.8284 53.8284L58.8284 81.8284C57.2663 83.3905 54.7337 83.3905 53.1716 81.8284L41.1716 69.8284C39.6095 68.2663 39.6095 65.7337 41.1716 64.1716C42.7337 62.6095 45.2663 62.6095 46.8284 64.1716L56 73.3431L81.1716 48.1716C82.7337 46.6095 85.2663 46.6095 86.8284 48.1716Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M64 116C92.7188 116 116 92.7188 116 64C116 35.2812 92.7188 12 64 12C35.2812 12 12 35.2812 12 64C12 92.7188 35.2812 116 64 116ZM86.8284 53.8284C88.3905 52.2663 88.3905 49.7337 86.8284 48.1716C85.2663 46.6095 82.7337 46.6095 81.1716 48.1716L56 73.3431L46.8284 64.1716C45.2663 62.6095 42.7337 62.6095 41.1716 64.1716C39.6095 65.7337 39.6095 68.2663 41.1716 69.8284L53.1716 81.8284C54.7337 83.3905 57.2663 83.3905 58.8284 81.8284L86.8284 53.8284Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const WriteIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="128"
            height="128"
            viewBox="0 0 128 128"
            fill="none"
        >
            <path
                opacity="0.32"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M11.2867 28.1587C10.541 30.9415 11.2397 34.2284 12.637 40.8022L25.278 100.274L92.9104 65.4329L81.4603 76.883C79.827 78.5163 78.5117 80.4391 77.5814 82.5534L76.5951 84.795C75.8677 86.4482 77.552 88.1325 79.2052 87.4051L81.4468 86.4188C83.561 85.4885 85.4839 84.1732 87.1172 82.5398L125.456 44.201C127.018 42.6389 127.018 40.1063 125.456 38.5442C123.894 36.9821 121.361 36.9821 119.799 38.5442L105.222 53.1218L98.714 22.5059L98.7139 22.5059C97.3167 15.9322 96.618 12.6453 94.805 10.4064C93.2102 8.43697 91.0328 7.023 88.585 6.36711C85.8023 5.62147 82.5154 6.32012 75.9416 7.71741L27.4255 18.0298C20.8517 19.4271 17.5648 20.1258 15.3259 21.9388C13.3565 23.5336 11.9426 25.7109 11.2867 28.1587Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M21.3079 18.5521C20 21.1191 20 24.4794 20 31.2V96.8C20 103.521 20 106.881 21.3079 109.448C22.4584 111.706 24.2942 113.542 26.5521 114.692C29.1191 116 32.4794 116 39.2 116H88.8C95.5206 116 98.8809 116 101.448 114.692C103.706 113.542 105.542 111.706 106.692 109.448C108 106.881 108 103.521 108 96.8V61.657L87.1171 82.5398C85.4838 84.1732 83.561 85.4885 81.4467 86.4188L79.2051 87.4051C77.5519 88.1325 75.8676 86.4482 76.595 84.795L77.5813 82.5534C78.5116 80.4391 79.827 78.5163 81.4603 76.883L108 50.3433V31.2C108 24.4794 108 21.1191 106.692 18.5521C105.542 16.2942 103.706 14.4584 101.448 13.3079C98.8809 12 95.5206 12 88.8 12H39.2C32.4794 12 29.1191 12 26.5521 13.3079C24.2942 14.4584 22.4584 16.2942 21.3079 18.5521Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SpinnerIcon: FC<{ className?: string }> = ({ className }) => (
    <StyledSpinner viewBox="0 0 50 50" width="1em" height="1em" className={className}>
        <circle className="path" cx="25" cy="25" r="20" fill="none" strokeWidth="4" />
    </StyledSpinner>
);

const StyledSpinner = styled.svg`
    animation: rotate 1s linear infinite;
    width: 1em;
    height: 1em;

    & .path {
        stroke: currentColor;
        stroke-linecap: round;
        animation: dash 1.5s ease-in-out infinite;
    }

    @keyframes rotate {
        100% {
            transform: rotate(360deg);
        }
    }
    @keyframes dash {
        0% {
            stroke-dasharray: 1, 150;
            stroke-dashoffset: 0;
        }
        50% {
            stroke-dasharray: 90, 150;
            stroke-dashoffset: -35;
        }
        100% {
            stroke-dasharray: 90, 150;
            stroke-dashoffset: -124;
        }
    }
`;

export const ToncoinIcon: FC<{ width?: string; height?: string }> = ({
    width = '44',
    height = '44'
}) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width={width}
            height={height}
            viewBox="0 0 44 44"
            fill="none"
        >
            <rect width="44" height="44" rx="22" fill="#0088CC" />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M23.1557 31.4672L30.0161 19.9928C30.7071 18.8369 31.0526 18.259 31.1463 17.7999C31.4212 16.4532 30.6595 15.1104 29.3626 14.6552C28.9205 14.5 28.2471 14.5 26.9005 14.5H17.0997C15.7531 14.5 15.0797 14.5 14.6376 14.6552C13.3407 15.1104 12.579 16.4532 12.8539 17.7999C12.9476 18.259 13.2931 18.8369 13.9841 19.9928L20.8445 31.4672C21.0894 31.8768 21.2119 32.0816 21.3417 32.1913C21.7218 32.5126 22.2784 32.5126 22.6585 32.1913C22.7884 32.0816 22.9108 31.8768 23.1557 31.4672ZM21.0001 16.5V27.9L15.1768 17.9916C14.9921 17.6774 14.8997 17.5202 14.875 17.3953C14.8027 17.029 15.0108 16.6653 15.3633 16.542C15.4835 16.5 15.6658 16.5 16.0303 16.5H21.0001ZM23.0001 27.9V16.5H27.9699C28.3345 16.5 28.5167 16.5 28.6369 16.542C28.9894 16.6653 29.1975 17.029 29.1252 17.3953C29.1005 17.5202 29.0081 17.6774 28.8235 17.9916L23.0001 27.9Z"
                fill="white"
            />
        </svg>
    );
};

export const CheckboxIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="12"
            height="9"
            viewBox="0 0 12 9"
            fill="none"
        >
            <path
                d="M11.7071 1.70711C12.0976 1.31658 12.0976 0.683417 11.7071 0.292893C11.3166 -0.0976311 10.6834 -0.0976311 10.2929 0.292893L4 6.58579L1.70711 4.29289C1.31658 3.90237 0.683418 3.90237 0.292893 4.29289C-0.0976311 4.68342 -0.0976311 5.31658 0.292893 5.70711L3.29289 8.70711C3.68342 9.09763 4.31658 9.09763 4.70711 8.70711L11.7071 1.70711Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const DownIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="12"
            height="6"
            viewBox="0 0 12 6"
            fill="none"
        >
            <path
                d="M1.69495 0.646258C1.36148 0.400546 0.891968 0.471684 0.646257 0.805149C0.400546 1.13861 0.471685 1.60813 0.80515 1.85384L5.55515 5.35384C5.81972 5.54879 6.18038 5.54879 6.44495 5.35384L11.1949 1.85384C11.5284 1.60813 11.5996 1.13861 11.3538 0.805149C11.1081 0.471684 10.6386 0.400546 10.3052 0.646258L6.00005 3.81844L1.69495 0.646258Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PlusIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                d="M8.75 1.75C8.75 1.33579 8.41421 1 8 1C7.58579 1 7.25 1.33579 7.25 1.75V7.25H1.75C1.33579 7.25 1 7.58579 1 8C1 8.41421 1.33579 8.75 1.75 8.75H7.25V14.25C7.25 14.6642 7.58579 15 8 15C8.41421 15 8.75 14.6642 8.75 14.25V8.75H14.25C14.6642 8.75 15 8.41421 15 8C15 7.58579 14.6642 7.25 14.25 7.25H8.75V1.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const MinusIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M14 8C14 8.41421 13.6642 8.75 13.25 8.75L2.75 8.75C2.33579 8.75 2 8.41421 2 8C2 7.58579 2.33579 7.25 2.75 7.25L13.25 7.25C13.6642 7.25 14 7.58579 14 8Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PlusIconSmall = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                d="M8.75 2.75C8.75 2.33579 8.41421 2 8 2C7.58579 2 7.25 2.33579 7.25 2.75V7.25H2.75C2.33579 7.25 2 7.58579 2 8C2 8.41421 2.33579 8.75 2.75 8.75H7.25V13.25C7.25 13.6642 7.58579 14 8 14C8.41421 14 8.75 13.6642 8.75 13.25V8.75H13.25C13.6642 8.75 14 8.41421 14 8C14 7.58579 13.6642 7.25 13.25 7.25H8.75V2.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SettingsIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M5.99116 1.0011L5.89497 1.00104C5.39263 1.00054 4.97766 1.00014 4.59384 1.12651C4.25688 1.23747 3.94738 1.41887 3.68593 1.65866C3.38813 1.93179 3.18573 2.29406 2.94071 2.73259L2.89377 2.81655L0.961754 6.26655L0.916807 6.34669L0.916805 6.3467C0.681201 6.76648 0.486501 7.11339 0.407652 7.49415C0.3384 7.82857 0.3384 8.17363 0.407652 8.50805C0.486501 8.88881 0.681202 9.23572 0.916807 9.65551L0.961754 9.73565L2.89377 13.1857L2.94071 13.2696C3.18573 13.7081 3.38813 14.0704 3.68593 14.3435C3.94738 14.5833 4.25688 14.7647 4.59384 14.8757C4.97766 15.0021 5.39263 15.0017 5.89497 15.0012L5.99116 15.0011H10.0089L10.1051 15.0012C10.6074 15.0017 11.0224 15.0021 11.4062 14.8757C11.7432 14.7647 12.0527 14.5833 12.3141 14.3435C12.6119 14.0704 12.8143 13.7081 13.0593 13.2696L13.1063 13.1856L15.0383 9.73563L15.0832 9.65549C15.3188 9.23571 15.5135 8.8888 15.5923 8.50805C15.6616 8.17363 15.6616 7.82857 15.5923 7.49415C15.5135 7.1134 15.3188 6.7665 15.0832 6.34671L15.0383 6.26657L13.1063 2.81657L13.0593 2.73261L13.0593 2.73261C12.8143 2.29407 12.6119 1.9318 12.3141 1.65867C12.0527 1.41888 11.7432 1.23747 11.4062 1.12652C11.0224 1.00014 10.6074 1.00054 10.1051 1.00104L10.0089 1.0011H5.99116ZM5.06298 2.55127C5.18817 2.51004 5.34233 2.5011 5.99116 2.5011H10.0089C10.6577 2.5011 10.8119 2.51004 10.9371 2.55127C11.0718 2.59565 11.1956 2.66821 11.3002 2.76413C11.3974 2.85322 11.4805 2.98336 11.7975 3.54947L13.7295 6.99947C14.0329 7.54135 14.0978 7.67402 14.1235 7.79832C14.1512 7.93209 14.1512 8.07011 14.1235 8.20388C14.0978 8.32818 14.0329 8.46085 13.7295 9.00273L11.7975 12.4527C11.4805 13.0188 11.3974 13.149 11.3002 13.2381C11.1956 13.334 11.0718 13.4066 10.9371 13.4509C10.8119 13.4922 10.6577 13.5011 10.0089 13.5011H5.99116C5.34233 13.5011 5.18817 13.4922 5.06298 13.4509C4.92819 13.4066 4.80439 13.334 4.69981 13.2381C4.60267 13.149 4.51954 13.0188 4.20252 12.4527L2.27051 9.00274C1.96705 8.46085 1.90223 8.32818 1.87649 8.20388C1.84879 8.07011 1.84879 7.93209 1.87649 7.79832C1.90223 7.67402 1.96705 7.54135 2.27051 6.99946L4.20252 3.54946C4.51954 2.98336 4.60267 2.85322 4.69981 2.76413C4.80439 2.66821 4.92819 2.59565 5.06298 2.55127ZM6.50002 8.0011C6.50002 7.17267 7.17159 6.5011 8.00002 6.5011C8.82844 6.5011 9.50002 7.17267 9.50002 8.0011C9.50002 8.82953 8.82844 9.5011 8.00002 9.5011C7.17159 9.5011 6.50002 8.82953 6.50002 8.0011ZM8.00002 5.0011C6.34316 5.0011 5.00002 6.34425 5.00002 8.0011C5.00002 9.65795 6.34316 11.0011 8.00002 11.0011C9.65687 11.0011 11 9.65795 11 8.0011C11 6.34425 9.65687 5.0011 8.00002 5.0011Z"
                fill="white"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.99986 5.00108C6.34301 5.00108 4.99986 6.34423 4.99986 8.00108C4.99986 9.65794 6.34301 11.0011 7.99986 11.0011C9.65672 11.0011 10.9999 9.65794 10.9999 8.00108C10.9999 6.34423 9.65672 5.00108 7.99986 5.00108ZM6.49986 8.00108C6.49986 7.17265 7.17144 6.50108 7.99986 6.50108C8.82829 6.50108 9.49986 7.17265 9.49986 8.00108C9.49986 8.82951 8.82829 9.50108 7.99986 9.50108C7.17144 9.50108 6.49986 8.82951 6.49986 8.00108Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M6.1227 1.00103C5.54082 1.00057 5.08365 1.00021 4.66203 1.13904C4.29137 1.26109 3.95092 1.46063 3.66332 1.7244C3.33619 2.02444 3.11313 2.4235 2.82922 2.93141L1.02305 6.1567C0.750249 6.64297 0.53584 7.02516 0.449223 7.44344C0.373045 7.8113 0.373045 8.19087 0.449223 8.55873C0.53584 8.977 0.75025 9.35919 1.02305 9.84547L2.82922 13.0708C3.11313 13.5787 3.33619 13.9777 3.66332 14.2778C3.95092 14.5415 4.29137 14.7411 4.66203 14.8631C5.08366 15.002 5.54082 15.0016 6.1227 15.0011H9.87701C10.4589 15.0016 10.9161 15.002 11.3377 14.8631C11.7084 14.7411 12.0488 14.5415 12.3364 14.2778C12.6635 13.9777 12.8866 13.5787 13.1705 13.0707L14.9767 9.84544C15.2494 9.35917 15.4638 8.97699 15.5505 8.55872C15.6266 8.19086 15.6266 7.8113 15.5505 7.44344C15.4638 7.02517 15.2494 6.64298 14.9766 6.15671L13.1705 2.93142C12.8866 2.42351 12.6635 2.02445 12.3364 1.72441C12.0488 1.46064 11.7084 1.26109 11.3377 1.13904C10.9161 1.00021 10.4589 1.00057 9.87702 1.00103H6.1227ZM5.13116 2.56379C5.29445 2.51002 5.49128 2.50108 6.22545 2.50108H9.77426C10.5084 2.50108 10.7053 2.51002 10.8686 2.56379C11.037 2.61927 11.1918 2.70997 11.3225 2.82987C11.4492 2.94607 11.5532 3.11344 11.9119 3.75401L13.6199 6.80401C13.9633 7.4172 14.0481 7.58552 14.0816 7.74761C14.1163 7.91482 14.1163 8.08735 14.0816 8.25455C14.0481 8.41664 13.9633 8.58497 13.6199 9.19815L11.9119 12.2482C11.5532 12.8887 11.4492 13.0561 11.3225 13.1723C11.1918 13.2922 11.037 13.3829 10.8686 13.4384C10.7053 13.4921 10.5084 13.5011 9.77426 13.5011H6.22545C5.49128 13.5011 5.29445 13.4921 5.13116 13.4384C4.96268 13.3829 4.80793 13.2922 4.67721 13.1723C4.55051 13.0561 4.44654 12.8887 4.08782 12.2482L2.3798 9.19817C2.03641 8.58497 1.95162 8.41665 1.91806 8.25456C1.88343 8.08735 1.88343 7.91482 1.91806 7.74761C1.95162 7.58552 2.03641 7.41719 2.3798 6.804L4.08782 3.754C4.44654 3.11343 4.55051 2.94606 4.67721 2.82986C4.80793 2.70997 4.96268 2.61927 5.13116 2.56379Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ReorderIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28px"
            height="28px"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                d="M6 8.75C6 8.33579 6.33579 8 6.75 8H21.25C21.6642 8 22 8.33579 22 8.75C22 9.16421 21.6642 9.5 21.25 9.5H6.75C6.33579 9.5 6 9.16421 6 8.75Z"
                fill="currentColor"
            />
            <path
                d="M6 14C6 13.5858 6.33579 13.25 6.75 13.25H21.25C21.6642 13.25 22 13.5858 22 14C22 14.4142 21.6642 14.75 21.25 14.75H6.75C6.33579 14.75 6 14.4142 6 14Z"
                fill="currentColor"
            />
            <path
                d="M6.75 18.5C6.33579 18.5 6 18.8358 6 19.25C6 19.6642 6.33579 20 6.75 20H21.25C21.6642 20 22 19.6642 22 19.25C22 18.8358 21.6642 18.5 21.25 18.5H6.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const EllipsisIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2.5 9.5C3.32843 9.5 4 8.82843 4 8C4 7.17157 3.32843 6.5 2.5 6.5C1.67157 6.5 1 7.17157 1 8C1 8.82843 1.67157 9.5 2.5 9.5ZM9.5 8C9.5 8.82843 8.82843 9.5 8 9.5C7.17157 9.5 6.5 8.82843 6.5 8C6.5 7.17157 7.17157 6.5 8 6.5C8.82843 6.5 9.5 7.17157 9.5 8ZM15 8C15 8.82843 14.3284 9.5 13.5 9.5C12.6716 9.5 12 8.82843 12 8C12 7.17157 12.6716 6.5 13.5 6.5C14.3284 6.5 15 7.17157 15 8Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const VerificationIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <g clipPath="url(#clip0_3608_14949)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M7.45739 0.627004C7.19814 0.727617 6.96231 0.941576 6.49066 1.36949C6.26848 1.57108 6.15739 1.67187 6.03317 1.74698C5.86594 1.84811 5.68056 1.91558 5.48744 1.94561C5.34401 1.96791 5.19412 1.96211 4.89435 1.95051C4.25798 1.92587 3.93979 1.91356 3.67652 2.00313C3.322 2.12375 3.02542 2.37261 2.84507 2.7008C2.71114 2.94452 2.66802 3.26001 2.58177 3.89098C2.54115 4.18822 2.52083 4.33684 2.47396 4.47422C2.41086 4.65919 2.31222 4.83004 2.18358 4.97717C2.08804 5.08645 1.96949 5.17835 1.73239 5.36216C1.22907 5.75233 0.977414 5.94742 0.83331 6.18527C0.639262 6.50555 0.572032 6.88683 0.644835 7.25416C0.6989 7.52695 0.868659 7.79635 1.20818 8.33514C1.36811 8.58895 1.44808 8.71586 1.50048 8.85122C1.57104 9.03348 1.60529 9.22776 1.60133 9.42316C1.59839 9.56828 1.56665 9.71489 1.50316 10.0081C1.3684 10.6305 1.30102 10.9417 1.34351 11.2166C1.40073 11.5866 1.59432 11.9219 1.8862 12.1565C2.10296 12.3307 2.40617 12.428 3.01259 12.6225C3.29825 12.7141 3.44109 12.7599 3.56824 12.8299C3.73944 12.9242 3.89056 13.051 4.01312 13.2032C4.10415 13.3163 4.17407 13.449 4.3139 13.7144C4.61075 14.2778 4.75919 14.5596 4.9684 14.7428C5.25012 14.9895 5.61393 15.1219 5.98832 15.114C6.26636 15.1081 6.56114 14.9877 7.1507 14.7469C7.42843 14.6335 7.5673 14.5768 7.70971 14.5487C7.90144 14.5109 8.09872 14.5109 8.29046 14.5487C8.43287 14.5768 8.57174 14.6335 8.84947 14.7469C9.43903 14.9877 9.73381 15.1081 10.0118 15.114C10.3862 15.1219 10.7501 14.9895 11.0318 14.7428C11.241 14.5596 11.3894 14.2779 11.6863 13.7144C11.8261 13.449 11.896 13.3163 11.987 13.2032C12.1096 13.051 12.2607 12.9242 12.4319 12.8299C12.5591 12.7599 12.7019 12.7141 12.9876 12.6225C13.594 12.428 13.8972 12.3307 14.114 12.1565C14.4059 11.9219 14.5994 11.5866 14.6567 11.2166C14.6992 10.9417 14.6318 10.6305 14.497 10.0081C14.4335 9.71491 14.4018 9.56828 14.3988 9.42316C14.3949 9.22776 14.4291 9.03348 14.4997 8.85122C14.5521 8.71586 14.6321 8.58895 14.792 8.33514C15.1315 7.79635 15.3013 7.52695 15.3553 7.25416C15.4281 6.88683 15.3609 6.50555 15.1669 6.18527C15.0228 5.94742 14.7711 5.75233 14.2678 5.36216C14.0307 5.17835 13.9121 5.08645 13.8166 4.97717C13.688 4.83004 13.5893 4.65919 13.5262 4.47422C13.4793 4.33684 13.459 4.18822 13.4184 3.89098C13.3321 3.26001 13.289 2.94452 13.1551 2.7008C12.9748 2.37261 12.6782 2.12375 12.3236 2.00313C12.0604 1.91356 11.7422 1.92587 11.1058 1.95051C10.806 1.96211 10.6562 1.96791 10.5127 1.94561C10.3196 1.91558 10.1342 1.84811 9.96699 1.74698C9.84278 1.67187 9.73169 1.57108 9.50951 1.36949C9.03786 0.941576 8.80203 0.727617 8.54277 0.627004C8.19366 0.49152 7.8065 0.49152 7.45739 0.627004ZM11.0304 7.03036C11.3233 6.73746 11.3233 6.26259 11.0304 5.9697C10.7375 5.6768 10.2626 5.6768 9.96975 5.9697L7.00008 8.93937L5.78041 7.7197C5.48752 7.4268 5.01265 7.4268 4.71975 7.7197C4.42686 8.01259 4.42686 8.48746 4.71975 8.78036L6.46975 10.5304C6.76265 10.8233 7.23752 10.8233 7.53041 10.5304L11.0304 7.03036Z"
                    fill="#8994A3"
                />
            </g>
            <defs>
                <clipPath id="clip0_3608_14949">
                    <rect width="16" height="16" fill="white" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const CheckmarkCircleIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="32"
            height="32"
            viewBox="0 0 32 32"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M16 30C23.732 30 30 23.732 30 16C30 8.26801 23.732 2 16 2C8.26801 2 2 8.26801 2 16C2 23.732 8.26801 30 16 30ZM22.0607 14.0607C22.6464 13.4749 22.6464 12.5251 22.0607 11.9393C21.4749 11.3536 20.5251 11.3536 19.9393 11.9393L14 17.8787L11.8107 15.6893C11.2249 15.1036 10.2751 15.1036 9.68934 15.6893C9.10355 16.2751 9.10355 17.2249 9.68934 17.8107L12.9393 21.0607C13.5251 21.6464 14.4749 21.6464 15.0607 21.0607L22.0607 14.0607Z"
                fill="currentColor"
            />
            <path
                opacity="0.32"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M22.0607 11.9393C22.6464 12.5251 22.6464 13.4749 22.0607 14.0607L15.0607 21.0607C14.4749 21.6464 13.5251 21.6464 12.9393 21.0607L9.68934 17.8107C9.10355 17.2249 9.10355 16.2751 9.68934 15.6893C10.2751 15.1036 11.2249 15.1036 11.8107 15.6893L14 17.8787L19.9393 11.9393C20.5251 11.3536 21.4749 11.3536 22.0607 11.9393Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ExclamationMarkCircleIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="32"
            height="32"
            viewBox="0 0 32 32"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M16 30C23.732 30 30 23.732 30 16C30 8.26801 23.732 2 16 2C8.26801 2 2 8.26801 2 16C2 23.732 8.26801 30 16 30ZM14.0999 9.99751C14.0455 8.91083 14.912 8 16 8C17.088 8 17.9545 8.91083 17.9001 9.9975L17.5749 16.5019C17.5329 17.3411 16.8403 18 16 18C15.1597 18 14.4671 17.3411 14.4251 16.5019L14.0999 9.99751ZM14 22C14 20.8954 14.8954 20 16 20C17.1046 20 18 20.8954 18 22C18 23.1046 17.1046 24 16 24C14.8954 24 14 23.1046 14 22Z"
                fill="currentColor"
            />
            <g opacity="0.32">
                <path
                    d="M16 8C14.912 8 14.0455 8.91083 14.0999 9.99751L14.4251 16.5019C14.4671 17.3411 15.1597 18 16 18C16.8403 18 17.5329 17.3411 17.5749 16.5019L17.9001 9.9975C17.9545 8.91083 17.088 8 16 8Z"
                    fill="currentColor"
                />
                <path
                    d="M16 20C14.8954 20 14 20.8954 14 22C14 23.1046 14.8954 24 16 24C17.1046 24 18 23.1046 18 22C18 20.8954 17.1046 20 16 20Z"
                    fill="currentColor"
                />
            </g>
        </svg>
    );
};

export const StarIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="12"
            height="12"
            viewBox="0 0 12 12"
            fill="none"
        >
            <path
                d="M4.60237 3.36036C5.20058 1.9221 5.49968 1.20297 6.00003 1.20297C6.50037 1.20297 6.79947 1.9221 7.39768 3.36036C7.49017 3.58275 7.53642 3.69394 7.62743 3.76007C7.71845 3.82619 7.83849 3.83582 8.07858 3.85506C9.6313 3.97954 10.4077 4.04178 10.5623 4.51764C10.7169 4.99349 10.1254 5.50018 8.94237 6.51355C8.75945 6.67025 8.66799 6.74859 8.63323 6.85558C8.59846 6.96257 8.62641 7.07972 8.68229 7.314C9.04372 8.82919 9.22444 9.58679 8.81965 9.88088C8.41486 10.175 7.75019 9.769 6.42084 8.95704C6.2153 8.83149 6.11252 8.76872 6.00003 8.76872C5.88753 8.76872 5.78476 8.83149 5.57921 8.95704C4.24986 9.769 3.58519 10.175 3.1804 9.88088C2.77562 9.58679 2.95633 8.82919 3.31776 7.314C3.37364 7.07972 3.40159 6.96257 3.36682 6.85558C3.33206 6.74859 3.2406 6.67025 3.05768 6.51355C1.87467 5.50018 1.28316 4.99349 1.43778 4.51764C1.59239 4.04178 2.36875 3.97954 3.92148 3.85506C4.16156 3.83582 4.28161 3.82619 4.37262 3.76007C4.46363 3.69394 4.50988 3.58275 4.60237 3.36036Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ScanIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                opacity="0.32"
                d="M9 12.2C9 11.0799 9 10.5198 9.21799 10.092C9.40973 9.71569 9.71569 9.40973 10.092 9.21799C10.5198 9 11.0799 9 12.2 9H15.8C16.9201 9 17.4802 9 17.908 9.21799C18.2843 9.40973 18.5903 9.71569 18.782 10.092C19 10.5198 19 11.0799 19 12.2V15.8C19 16.9201 19 17.4802 18.782 17.908C18.5903 18.2843 18.2843 18.5903 17.908 18.782C17.4802 19 16.9201 19 15.8 19H12.2C11.0799 19 10.5198 19 10.092 18.782C9.71569 18.5903 9.40973 18.2843 9.21799 17.908C9 17.4802 9 16.9201 9 15.8V12.2Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.64757 3L8.5 3.00001C9.32843 3.00001 10 3.67158 10 4.50001C10 5.32843 9.32843 6 8.5 6H6.8C6.51998 6 6.37997 6 6.27301 6.0545L6.26693 6.05765C6.17765 6.10469 6.10469 6.17765 6.05765 6.26693L6.0545 6.27301C6 6.37997 6 6.51998 6 6.80001V8.5C6 9.32843 5.32843 10 4.5 10C3.67158 10 3 9.32843 3 8.5L3 7.64757C2.99997 7.1326 2.99993 6.65963 3.03223 6.26426C3.06706 5.838 3.14679 5.37165 3.38148 4.91104C3.71704 4.25247 4.25247 3.71704 4.91104 3.38148C5.37165 3.14679 5.838 3.06706 6.26426 3.03223C6.65963 2.99993 7.1326 2.99997 7.64757 3ZM6.05946 6.26175L6.05834 6.26478C6.05904 6.26265 6.05946 6.26175 6.05946 6.26175ZM6.26175 6.05946C6.26175 6.05946 6.26265 6.05904 6.26478 6.05834L6.26175 6.05946Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M19.5 6C18.6716 6 18 5.32843 18 4.50001C18 3.67158 18.6716 3.00001 19.5 3.00001L20.3524 3C20.8674 2.99997 21.3404 2.99993 21.7358 3.03223C22.162 3.06706 22.6284 3.14679 23.089 3.38148C23.7475 3.71704 24.283 4.25247 24.6185 4.91104C24.8532 5.37165 24.9329 5.838 24.9678 6.26426C25.0001 6.65963 25 7.13258 25 7.64754L25 8.5C25 9.32843 24.3284 10 23.5 10C22.6716 10 22 9.32843 22 8.5L22 6.80001C22 6.51998 22 6.37997 21.9455 6.27301L21.9424 6.26693C21.8953 6.17765 21.8224 6.10469 21.7331 6.05765L21.727 6.0545C21.62 6 21.48 6 21.2 6H19.5ZM21.7383 6.05946L21.7352 6.05835C21.7374 6.05905 21.7383 6.05946 21.7383 6.05946ZM21.9406 6.26176C21.9406 6.26176 21.941 6.26263 21.9417 6.26476L21.9406 6.26176Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M22 19.5C22 18.6716 22.6716 18 23.5 18C24.3284 18 25 18.6716 25 19.5L25 20.3525C25 20.8674 25.0001 21.3404 24.9678 21.7358C24.9329 22.162 24.8532 22.6284 24.6185 23.089C24.283 23.7475 23.7475 24.283 23.089 24.6185C22.6284 24.8532 22.162 24.9329 21.7358 24.9678C21.3404 25.0001 20.8674 25 20.3525 25L19.5 25C18.6716 25 18 24.3284 18 23.5C18 22.6716 18.6716 22 19.5 22L21.2 22C21.48 22 21.62 22 21.727 21.9455L21.7331 21.9424C21.8224 21.8953 21.8953 21.8224 21.9424 21.7331L21.9455 21.727C22 21.62 22 21.48 22 21.2L22 19.5ZM21.9406 21.7383L21.9417 21.7352C21.941 21.7373 21.9406 21.7383 21.9406 21.7383ZM21.7383 21.9406C21.7383 21.9406 21.7374 21.941 21.7353 21.9417L21.7383 21.9406Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8.5 22C9.32843 22 10 22.6716 10 23.5C10 24.3284 9.32843 25 8.5 25L7.64754 25C7.13258 25 6.65963 25.0001 6.26426 24.9678C5.838 24.9329 5.37165 24.8532 4.91104 24.6185C4.25247 24.283 3.71704 23.7475 3.38148 23.089C3.14679 22.6284 3.06706 22.162 3.03223 21.7358C2.99993 21.3404 2.99997 20.8674 3 20.3524L3 19.5C3 18.6716 3.67158 18 4.5 18C5.32843 18 6 18.6716 6 19.5V21.2C6 21.48 6 21.62 6.0545 21.727L6.05765 21.7331C6.10469 21.8224 6.17765 21.8953 6.26693 21.9424L6.27301 21.9455C6.37997 22 6.51998 22 6.8 22L8.5 22ZM6.26175 21.9406L6.26475 21.9417C6.26264 21.941 6.26175 21.9406 6.26175 21.9406ZM6.05946 21.7383C6.05946 21.7383 6.05905 21.7374 6.05835 21.7353L6.05946 21.7383Z"
                fill="currentColor"
            />
            <path
                d="M7.5 12.5C6.67157 12.5 6 13.1716 6 14C6 14.8284 6.67157 15.5 7.5 15.5H20.5C21.3284 15.5 22 14.8284 22 14C22 13.1716 21.3284 12.5 20.5 12.5H7.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SaleIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="29"
            height="29"
            viewBox="0 0 29 29"
            fill="none"
        >
            <g filter="url(#filter0_d_6325_2611)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M15.6567 4H16.5998C18.14 4 18.91 4 19.4983 4.29973C20.0157 4.56338 20.4364 4.98408 20.7001 5.50153C20.9998 6.08978 20.9998 6.85986 20.9998 8.4V9.34315C20.9998 9.95462 20.9998 10.2604 20.9307 10.5481C20.8695 10.8032 20.7685 11.047 20.6314 11.2707C20.4768 11.523 20.2606 11.7392 19.8283 12.1716L16.7717 15.2281C15.8014 16.1984 15.3163 16.6835 14.7569 16.8653C14.2649 17.0251 13.7348 17.0251 13.2427 16.8653C12.6833 16.6835 12.1982 16.1984 11.228 15.2281L9.77168 13.7719C8.80144 12.8016 8.31632 12.3165 8.13456 11.7571C7.97468 11.265 7.97468 10.735 8.13456 10.2429C8.31632 9.6835 8.80144 9.19838 9.77168 8.22814L12.8283 5.17157C13.2606 4.73919 13.4768 4.523 13.7291 4.3684C13.9528 4.23133 14.1967 4.13032 14.4517 4.06908C14.7395 4 15.0452 4 15.6567 4ZM18.3748 7.75C18.9961 7.75 19.4998 7.24632 19.4998 6.625C19.4998 6.00368 18.9961 5.5 18.3748 5.5C17.7535 5.5 17.2498 6.00368 17.2498 6.625C17.2498 7.24632 17.7535 7.75 18.3748 7.75Z"
                    fill="white"
                />
            </g>
            <defs>
                <filter
                    id="filter0_d_6325_2611"
                    x="0.0146484"
                    y="0"
                    width="28.9854"
                    height="28.9852"
                    filterUnits="userSpaceOnUse"
                    colorInterpolationFilters="sRGB"
                >
                    <feFlood floodOpacity="0" result="BackgroundImageFix" />
                    <feColorMatrix
                        in="SourceAlpha"
                        type="matrix"
                        values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"
                        result="hardAlpha"
                    />
                    <feOffset dy="4" />
                    <feGaussianBlur stdDeviation="4" />
                    <feComposite in2="hardAlpha" operator="out" />
                    <feColorMatrix
                        type="matrix"
                        values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.08 0"
                    />
                    <feBlend
                        mode="normal"
                        in2="BackgroundImageFix"
                        result="effect1_dropShadow_6325_2611"
                    />
                    <feBlend
                        mode="normal"
                        in="SourceGraphic"
                        in2="effect1_dropShadow_6325_2611"
                        result="shape"
                    />
                </filter>
            </defs>
        </svg>
    );
};

export const FireBadgeIcon = () => {
    return (
        <svg
            width="32"
            height="32"
            viewBox="0 0 32 32"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
        >
            <g filter="url(#filter0_d_9578_3142)">
                <path
                    d="M24 18C24 21.3137 21.3137 24 18 24C14.6863 24 12 21.3137 12 18C12 14.3911 16.0701 13.0207 16.8659 9.15186C16.9718 8.63724 17.0247 8.37994 17.1549 8.24449C17.2664 8.12847 17.4272 8.05474 17.5878 8.04589C17.7754 8.03557 17.9371 8.1255 18.2604 8.30537C20.5637 9.58683 20.75 12.0683 20.75 12.5C20.75 14.25 19.7462 15.7462 20.5 16.5C21.0752 17.0751 22.0657 16.6655 22.3922 15.5181C22.4634 15.2677 22.499 15.1425 22.592 15.0719C22.6654 15.0161 22.7861 14.9891 22.8763 15.0082C22.9905 15.0324 23.0417 15.0911 23.1441 15.2083C23.5509 15.6739 24 16.5226 24 18Z"
                    fill="#FF4766"
                />
                <path
                    d="M18 24.75C21.7279 24.75 24.75 21.7279 24.75 18C24.75 16.3572 24.2453 15.3289 23.709 14.7149C23.7035 14.7087 23.6976 14.7017 23.691 14.6941C23.5994 14.587 23.3984 14.3522 23.0318 14.2745C22.7283 14.2102 22.3853 14.287 22.1383 14.4747C21.9447 14.6218 21.8467 14.8055 21.7905 14.9418C21.7442 15.0541 21.7069 15.1858 21.677 15.291C21.6749 15.2984 21.6728 15.3057 21.6708 15.3128C21.5536 15.7247 21.343 15.917 21.2133 15.9796C21.1499 16.0102 21.1102 16.0097 21.0944 16.0075C21.0827 16.0059 21.062 16.0013 21.0303 15.9697C21.0087 15.9481 20.9765 15.9113 20.9703 15.7705C20.9631 15.6051 20.9966 15.3774 21.0735 15.0404C21.0989 14.9293 21.129 14.8076 21.1613 14.677C21.3079 14.0844 21.5 13.3077 21.5 12.5C21.5 11.9921 21.2985 9.13742 18.625 7.64997C18.6173 7.6457 18.6096 7.6414 18.6018 7.63708C18.4606 7.55845 18.3067 7.47273 18.163 7.41255C17.9912 7.34058 17.7879 7.28375 17.5466 7.29703C17.1965 7.3163 16.8572 7.47192 16.6142 7.72478C16.4215 7.92531 16.3342 8.16576 16.2817 8.34473C16.2301 8.52057 16.185 8.73983 16.1368 8.9742C16.135 8.98302 16.1332 8.99187 16.1313 9.00073C15.794 10.6405 14.7981 11.7775 13.7037 13.027C13.6592 13.0777 13.6147 13.1286 13.5699 13.1797C12.465 14.4432 11.25 15.8828 11.25 18C11.25 21.7279 14.2721 24.75 18 24.75Z"
                    stroke="white"
                    strokeWidth="1.5"
                />
            </g>
            <defs>
                <filter
                    id="filter0_d_9578_3142"
                    x="4.5"
                    y="2.5451"
                    width="27"
                    height="30.9549"
                    filterUnits="userSpaceOnUse"
                    colorInterpolationFilters="sRGB"
                >
                    <feFlood floodOpacity="0" result="BackgroundImageFix" />
                    <feColorMatrix
                        in="SourceAlpha"
                        type="matrix"
                        values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"
                        result="hardAlpha"
                    />
                    <feOffset dy="2" />
                    <feGaussianBlur stdDeviation="3" />
                    <feComposite in2="hardAlpha" operator="out" />
                    <feColorMatrix
                        type="matrix"
                        values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.08 0"
                    />
                    <feBlend
                        mode="normal"
                        in2="BackgroundImageFix"
                        result="effect1_dropShadow_9578_3142"
                    />
                    <feBlend
                        mode="normal"
                        in="SourceGraphic"
                        in2="effect1_dropShadow_9578_3142"
                        result="shape"
                    />
                </filter>
            </defs>
        </svg>
    );
};

export const CopyIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="17"
            height="16"
            viewBox="0 0 17 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M12.863 11.988C13.8314 11.9619 14.4518 11.8795 14.964 11.6185C15.6225 11.283 16.158 10.7475 16.4935 10.089C16.875 9.34028 16.875 8.36018 16.875 6.4V5.6C16.875 3.63982 16.875 2.65972 16.4935 1.91103C16.158 1.25247 15.6225 0.717034 14.964 0.381477C14.2153 0 13.2352 0 11.275 0H10.475C8.51481 0 7.53472 0 6.78603 0.381477C6.12746 0.717034 5.59203 1.25247 5.25647 1.91103C4.99552 2.42319 4.91307 3.04363 4.88703 4.01203C3.91863 4.03808 3.29819 4.12052 2.78603 4.38148C2.12746 4.71703 1.59203 5.25247 1.25647 5.91103C0.874993 6.65972 0.874992 7.63982 0.874992 9.6V10.4C0.874992 12.3602 0.874993 13.3403 1.25647 14.089C1.59203 14.7475 2.12746 15.283 2.78603 15.6185C3.53472 16 4.51481 16 6.47499 16H7.27499C9.23518 16 10.2153 16 10.964 15.6185C11.6225 15.283 12.158 14.7475 12.4935 14.089C12.7545 13.5768 12.8369 12.9564 12.863 11.988ZM11.275 1.5H10.475C9.47015 1.5 8.80673 1.50117 8.29835 1.5427C7.80748 1.58281 7.59466 1.65295 7.46701 1.71799C7.09069 1.90973 6.78473 2.21569 6.59298 2.59202C6.52795 2.71966 6.4578 2.93248 6.4177 3.42335C6.40364 3.59541 6.39421 3.78521 6.38788 4C6.4167 4 6.44574 4 6.47499 4H7.27499C9.23518 4 10.2153 4 10.964 4.38148C11.6225 4.71703 12.158 5.25247 12.4935 5.91103C12.875 6.65972 12.875 7.63982 12.875 9.6V10.4C12.875 10.4293 12.875 10.4583 12.875 10.4871C13.0898 10.4808 13.2796 10.4714 13.4516 10.4573C13.9425 10.4172 14.1553 10.3471 14.283 10.282C14.6593 10.0903 14.9653 9.78431 15.157 9.40798C15.222 9.28034 15.2922 9.06752 15.3323 8.57665C15.3738 8.06826 15.375 7.40484 15.375 6.4V5.6C15.375 4.59516 15.3738 3.93174 15.3323 3.42335C15.2922 2.93248 15.222 2.71966 15.157 2.59202C14.9653 2.21569 14.6593 1.90973 14.283 1.71799C14.1553 1.65295 13.9425 1.58281 13.4516 1.5427C12.9433 1.50117 12.2798 1.5 11.275 1.5ZM6.47499 5.5H7.27499C8.27984 5.5 8.94326 5.50117 9.45164 5.5427C9.94251 5.58281 10.1553 5.65295 10.283 5.71799C10.6593 5.90973 10.9653 6.21569 11.157 6.59202C11.222 6.71966 11.2922 6.93248 11.3323 7.42335C11.3738 7.93174 11.375 8.59516 11.375 9.6V10.4C11.375 11.4048 11.3738 12.0683 11.3323 12.5766C11.2922 13.0675 11.222 13.2803 11.157 13.408C10.9653 13.7843 10.6593 14.0903 10.283 14.282C10.1553 14.3471 9.94251 14.4172 9.45164 14.4573C8.94326 14.4988 8.27984 14.5 7.27499 14.5H6.47499C5.47015 14.5 4.80673 14.4988 4.29835 14.4573C3.80748 14.4172 3.59465 14.3471 3.46701 14.282C3.09069 14.0903 2.78473 13.7843 2.59298 13.408C2.52794 13.2803 2.4578 13.0675 2.4177 12.5766C2.37616 12.0683 2.37499 11.4048 2.37499 10.4V9.6C2.37499 8.59516 2.37616 7.93174 2.4177 7.42335C2.4578 6.93248 2.52794 6.71966 2.59298 6.59202C2.78473 6.21569 3.09069 5.90973 3.46701 5.71799C3.59465 5.65295 3.80748 5.58281 4.29835 5.5427C4.80673 5.50117 5.47015 5.5 6.47499 5.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const GlobIcon: FC<{ size?: string }> = ({ size = '16' }) => {
    return (
        <svg
            width={size}
            height={size}
            viewBox="0 0 32 32"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M19.4677 17.5C19.235 22.8655 17.7711 27 16 27C14.2289 27 12.765 22.8655 12.5323 17.5H19.4677ZM22.4703 17.5C22.363 20.1939 21.9676 22.6955 21.3337 24.6878C21.2101 25.0761 21.0716 25.464 20.9157 25.8432C24.0971 24.2513 26.3992 21.1624 26.8986 17.5H22.4703ZM26.8986 14.5H22.4703C22.363 11.8061 21.9676 9.30449 21.3337 7.31221C21.2101 6.9239 21.0716 6.53599 20.9157 6.1568C24.0971 7.7487 26.3992 10.8376 26.8986 14.5ZM19.4677 14.5H12.5323C12.765 9.13526 14.2285 5.00121 15.9992 5H16C17.7711 5 19.235 9.13447 19.4677 14.5ZM9.52969 14.5C9.63704 11.8061 10.0324 9.30449 10.6663 7.31221C10.7899 6.9239 10.9284 6.53599 11.0843 6.1568C7.90287 7.7487 5.60079 10.8376 5.10142 14.5H9.52969ZM5.10142 17.5C5.60079 21.1624 7.90287 24.2513 11.0843 25.8432C10.9284 25.464 10.7899 25.0761 10.6663 24.6878C10.0324 22.6955 9.63704 20.1939 9.52969 17.5H5.10142ZM30 16C30 23.732 23.732 30 16 30C8.26801 30 2 23.732 2 16C2 8.26801 8.26801 2 16 2C23.732 2 30 8.26801 30 16Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const MessageIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M3 10.4C3 8.15979 3 7.03969 3.43597 6.18404C3.81947 5.43139 4.43139 4.81947 5.18404 4.43597C6.03969 4 7.15979 4 9.4 4H18.6C20.8402 4 21.9603 4 22.816 4.43597C23.5686 4.81947 24.1805 5.43139 24.564 6.18404C25 7.03969 25 8.15979 25 10.4V15.6C25 17.8402 25 18.9603 24.564 19.816C24.1805 20.5686 23.5686 21.1805 22.816 21.564C21.9603 22 20.8402 22 18.6 22H9L7.23021 23.7698C6.22414 24.7759 5.7211 25.2789 5.31696 25.4102C4.48892 25.6793 3.58819 25.3062 3.19292 24.5304C3 24.1518 3 23.4404 3 22.0176V22V17V10.4ZM8 10.5C8 9.67157 8.67157 9 9.5 9H18.5C19.3284 9 20 9.67157 20 10.5C20 11.3284 19.3284 12 18.5 12H9.5C8.67157 12 8 11.3284 8 10.5ZM8 15.5C8 14.6716 8.67157 14 9.5 14H14.5C15.3284 14 16 14.6716 16 15.5C16 16.3284 15.3284 17 14.5 17H9.5C8.67157 17 8 16.3284 8 15.5Z"
                fill="currentColor"
            />
            <path
                opacity="0.32"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M9.5 9C8.67157 9 8 9.67157 8 10.5C8 11.3284 8.67157 12 9.5 12H18.5C19.3284 12 20 11.3284 20 10.5C20 9.67157 19.3284 9 18.5 9H9.5ZM9.5 14C8.67157 14 8 14.6716 8 15.5C8 16.3284 8.67157 17 9.5 17H14.5C15.3284 17 16 16.3284 16 15.5C16 14.6716 15.3284 14 14.5 14H9.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const NotificationIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                d="M23.9985 9.46572C23.4105 9.80554 22.728 10 22 10C19.7909 10 18 8.20914 18 6C18 5.27203 18.1945 4.58948 18.5343 4.00147C18.2461 4 17.9356 4 17.6 4H10.4C8.15979 4 7.03968 4 6.18404 4.43597C5.43139 4.81947 4.81947 5.43139 4.43597 6.18404C4 7.03969 4 8.15979 4 10.4V17.6C4 19.8402 4 20.9603 4.43597 21.816C4.81947 22.5686 5.43139 23.1805 6.18404 23.564C7.03968 24 8.15979 24 10.4 24H17.6C19.8402 24 20.9603 24 21.816 23.564C22.5686 23.1805 23.1805 22.5686 23.564 21.816C24 20.9603 24 19.8402 24 17.6V10.4C24 10.0644 24 9.75393 23.9985 9.46572Z"
                fill="currentColor"
            />
            <path
                opacity="0.32"
                d="M26 6C26 8.20914 24.2091 10 22 10C19.7909 10 18 8.20914 18 6C18 3.79086 19.7909 2 22 2C24.2091 2 26 3.79086 26 6Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const NotificationOutlineIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M13.5 5C14.8807 5 16 3.88071 16 2.5C16 1.11929 14.8807 0 13.5 0C12.1193 0 11 1.11929 11 2.5C11 3.88071 12.1193 5 13.5 5ZM1.43597 3.18404C1 4.03969 1 5.15979 1 7.4V8.6C1 10.8402 1 11.9603 1.43597 12.816C1.81947 13.5686 2.43139 14.1805 3.18404 14.564C4.03969 15 5.15979 15 7.4 15H8.6C10.8402 15 11.9603 15 12.816 14.564C13.5686 14.1805 14.1805 13.5686 14.564 12.816C15 11.9603 15 10.8402 15 8.6V7.4L15 7.25268V7.25C15 6.83579 14.6642 6.5 14.25 6.5C13.8358 6.5 13.5 6.83579 13.5 7.25V9.5C13.5 10.9001 13.5 11.6002 13.2275 12.135C12.9878 12.6054 12.6054 12.9878 12.135 13.2275C11.6002 13.5 10.9001 13.5 9.5 13.5H6.5C5.09987 13.5 4.3998 13.5 3.86502 13.2275C3.39462 12.9878 3.01217 12.6054 2.77248 12.135C2.5 11.6002 2.5 10.9001 2.5 9.5V6.5C2.5 5.09987 2.5 4.3998 2.77248 3.86502C3.01217 3.39462 3.39462 3.01217 3.86502 2.77248C4.3998 2.5 5.09987 2.5 6.5 2.5H8.75C9.16421 2.5 9.5 2.16421 9.5 1.75C9.5 1.33579 9.16421 1 8.75 1.00001L8.74725 1L8.6 1H7.4C5.15979 1 4.03969 1 3.18404 1.43597C2.43139 1.81947 1.81947 2.43139 1.43597 3.18404Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SlidersIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2 3.5H7.35352C7.67998 2.34575 8.74122 1.5 10 1.5C11.2588 1.5 12.32 2.34575 12.6465 3.5H14C14.4142 3.5 14.75 3.83579 14.75 4.25C14.75 4.66421 14.4142 5 14 5H12.6465C12.32 6.15425 11.2588 7 10 7C8.74122 7 7.67998 6.15425 7.35352 5H2C1.58579 5 1.25 4.66421 1.25 4.25C1.25 3.83579 1.58579 3.5 2 3.5ZM10 3C9.30964 3 8.75 3.55964 8.75 4.25C8.75 4.94036 9.30964 5.5 10 5.5C10.6904 5.5 11.25 4.94036 11.25 4.25C11.25 3.55964 10.6904 3 10 3Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2 11H3.35352C3.67998 9.84575 4.74122 9 6 9C7.25878 9 8.32002 9.84575 8.64648 11H14C14.4142 11 14.75 11.3358 14.75 11.75C14.75 12.1642 14.4142 12.5 14 12.5H8.64648C8.32002 13.6543 7.25878 14.5 6 14.5C4.74122 14.5 3.67998 13.6543 3.35352 12.5H2C1.58579 12.5 1.25 12.1642 1.25 11.75C1.25 11.3358 1.58579 11 2 11ZM6 10.5C5.30964 10.5 4.75 11.0596 4.75 11.75C4.75 12.4404 5.30964 13 6 13C6.69036 13 7.25 12.4404 7.25 11.75C7.25 11.0596 6.69036 10.5 6 10.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const TelegramIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="17"
            height="16"
            viewBox="0 0 17 16"
            fill="none"
            color="white"
            className={className}
        >
            <g clipPath="url(#clip0_27114_2672)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M1.5452 7.42434C5.62648 5.64619 8.34798 4.47392 9.70968 3.90754C13.5976 2.29041 14.4055 2.0095 14.9321 2.00022C15.0479 1.99818 15.3068 2.02688 15.4746 2.16299C15.6162 2.27792 15.6552 2.43316 15.6738 2.54213C15.6925 2.65109 15.7157 2.89931 15.6972 3.09326C15.4866 5.30699 14.5749 10.6791 14.1111 13.1585C13.9149 14.2077 13.5284 14.5595 13.1543 14.5939C12.3413 14.6687 11.724 14.0566 10.9365 13.5404C9.70435 12.7327 9.00825 12.2299 7.8122 11.4417C6.42996 10.5308 7.32601 10.0302 8.11374 9.21204C8.3199 8.99792 11.902 5.7397 11.9714 5.44413C11.98 5.40717 11.9881 5.26937 11.9062 5.19661C11.8244 5.12385 11.7035 5.14874 11.6163 5.16852C11.4928 5.19657 9.52434 6.49763 5.71107 9.07169C5.15234 9.45536 4.64625 9.6423 4.19282 9.6325C3.69295 9.6217 2.7314 9.34987 2.01658 9.11751C1.13983 8.83251 0.443006 8.68183 0.503683 8.19782C0.535288 7.94571 0.882459 7.68789 1.5452 7.42434Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_27114_2672">
                    <rect
                        width="16"
                        height="16"
                        fill="white"
                        transform="translate(0.5 0.00012207)"
                    />
                </clipPath>
            </defs>
        </svg>
    );
};

export const StatsIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M13.25 2C12.8358 2 12.5 2.33579 12.5 2.75V13.25C12.5 13.6642 12.8358 14 13.25 14C13.6642 14 14 13.6642 14 13.25V2.75C14 2.33579 13.6642 2 13.25 2ZM2.75 4C2.33579 4 2 4.33579 2 4.75V13.25C2 13.6642 2.33579 14 2.75 14C3.16421 14 3.5 13.6642 3.5 13.25V4.75C3.5 4.33579 3.16421 4 2.75 4ZM6.25 8C5.83579 8 5.5 8.33579 5.5 8.75V13.25C5.5 13.6642 5.83579 14 6.25 14C6.66421 14 7 13.6642 7 13.25V8.75C7 8.33579 6.66421 8 6.25 8ZM9 6.75C9 6.33579 9.33579 6 9.75 6C10.1642 6 10.5 6.33579 10.5 6.75V13.25C10.5 13.6642 10.1642 14 9.75 14C9.33579 14 9 13.6642 9 13.25V6.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CoinsIcon: FC<{ className?: string; color?: string }> = ({ className, color }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M4.63569 3.21983C3.77767 3.66927 3.5 4.15946 3.5 4.5C3.5 4.84054 3.77767 5.33073 4.63569 5.78017C5.45819 6.211 6.64642 6.5 8 6.5C9.35358 6.5 10.5418 6.211 11.3643 5.78017C12.2223 5.33073 12.5 4.84054 12.5 4.5C12.5 4.15946 12.2223 3.66927 11.3643 3.21983C10.5418 2.789 9.35358 2.5 8 2.5C6.64642 2.5 5.45819 2.789 4.63569 3.21983ZM12.5 6.85168C12.358 6.94438 12.2106 7.03018 12.0603 7.10892C10.9827 7.67339 9.54592 8 8 8C6.45408 8 5.01731 7.67339 3.93968 7.10892C3.78936 7.03018 3.64205 6.94438 3.5 6.85168V8C3.5 8.34054 3.77767 8.83073 4.63569 9.28017C5.45819 9.711 6.64642 10 8 10C9.35358 10 10.5418 9.711 11.3643 9.28017C12.2223 8.83073 12.5 8.34054 12.5 8V6.85168ZM14 8V4.5C14 3.32176 13.1024 2.43695 12.0603 1.89108C10.9827 1.32661 9.54592 1 8 1C6.45408 1 5.01731 1.32661 3.93968 1.89108C2.89758 2.43695 2 3.32176 2 4.5V8V11.5C2 12.6782 2.89758 13.5631 3.93968 14.1089C5.01731 14.6734 6.45408 15 8 15C9.54592 15 10.9827 14.6734 12.0603 14.1089C13.1024 13.5631 14 12.6782 14 11.5V8ZM12.5 10.3517C12.358 10.4444 12.2106 10.5302 12.0603 10.6089C10.9827 11.1734 9.54592 11.5 8 11.5C6.45408 11.5 5.01731 11.1734 3.93968 10.6089C3.78936 10.5302 3.64205 10.4444 3.5 10.3517V11.5C3.5 11.8405 3.77767 12.3307 4.63569 12.7802C5.45819 13.211 6.64642 13.5 8 13.5C9.35358 13.5 10.5418 13.211 11.3643 12.7802C12.2223 12.3307 12.5 11.8405 12.5 11.5V10.3517Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SaleBadgeIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M9.04155 1.00003L8.95378 1.00002C8.28542 0.999818 7.82282 0.999681 7.37995 1.10866C6.98863 1.20495 6.61538 1.3637 6.27457 1.57879C5.88888 1.8222 5.5681 2.15552 5.10465 2.63708L5.10464 2.63709L5.04377 2.70032L3.33599 4.47379L3.31387 4.49676L3.31387 4.49676C2.74067 5.092 2.27912 5.57129 1.93659 5.99001C1.58403 6.421 1.31556 6.8366 1.1662 7.31781C0.928137 8.08483 0.935893 8.90709 1.18839 9.66948C1.3468 10.1478 1.62305 10.5583 1.98368 10.9825C2.33405 11.3947 2.80457 11.8652 3.3889 12.4495L3.41144 12.4721L3.52804 12.5887L3.55059 12.6112C4.1349 13.1955 4.60541 13.6661 5.01759 14.0164C5.44185 14.3771 5.85231 14.6533 6.33062 14.8117C7.09301 15.0642 7.91528 15.072 8.6823 14.8339C9.16351 14.6845 9.57911 14.4161 10.0101 14.0635C10.4288 13.721 10.9081 13.2594 11.5033 12.6862L11.5263 12.6641L13.2998 10.9563L13.363 10.8955L13.363 10.8955C13.8446 10.432 14.1779 10.1112 14.4213 9.72553C14.6364 9.38473 14.7951 9.01148 14.8914 8.62015C15.0004 8.17728 15.0003 7.71468 15.0001 7.04632L15.0001 6.95855V5.75003V5.7186V5.71856C15.0001 5.04521 15.0001 4.49452 14.9635 4.04696C14.9257 3.5837 14.8449 3.16531 14.6458 2.77456C14.3343 2.16304 13.8371 1.66585 13.2255 1.35426C12.8348 1.15517 12.4164 1.07444 11.9531 1.03659C11.5056 1.00002 10.9549 1.00002 10.2815 1.00003L10.2501 1.00003H9.04155ZM7.73837 2.56521C7.98267 2.50509 8.25203 2.50003 9.04155 2.50003H10.2501C10.9625 2.50003 11.4517 2.50062 11.831 2.5316C12.2015 2.56187 12.4005 2.61738 12.5446 2.69077C12.8738 2.85855 13.1416 3.12627 13.3093 3.45555C13.3827 3.59959 13.4382 3.79861 13.4685 4.16911C13.4995 4.54837 13.5001 5.03759 13.5001 5.75003V6.95855C13.5001 7.74808 13.495 8.01743 13.4349 8.26173C13.3771 8.49653 13.2819 8.72048 13.1528 8.92496C13.0185 9.13772 12.828 9.3282 12.2593 9.87585L10.4859 11.5836C9.8623 12.1841 9.42645 12.603 9.06033 12.9025C8.70111 13.1964 8.45838 13.3328 8.23765 13.4013C7.76947 13.5466 7.26757 13.5419 6.80221 13.3878C6.58282 13.3151 6.3427 13.1741 5.98909 12.8735C5.62868 12.5672 5.20082 12.1401 4.5887 11.528L4.4721 11.4114C3.85999 10.7993 3.43293 10.3714 3.12658 10.011C2.82599 9.6574 2.68499 9.41729 2.61233 9.19789C2.45821 8.73254 2.45347 8.23063 2.59879 7.76245C2.6673 7.54172 2.80375 7.29899 3.09761 6.93977C3.3971 6.57365 3.81601 6.1378 4.41647 5.51425L6.12425 3.74079C6.6719 3.17208 6.86238 2.98156 7.07514 2.84729C7.27963 2.71823 7.50357 2.62299 7.73837 2.56521ZM10.875 6.25004C11.4963 6.25004 12 5.74636 12 5.12504C12 4.50372 11.4963 4.00004 10.875 4.00004C10.2537 4.00004 9.75001 4.50372 9.75001 5.12504C9.75001 5.74636 10.2537 6.25004 10.875 6.25004Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ClockSmoothIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <g clipPath="url(#clip0_28080_3060)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M14 8C14 11.3137 11.3137 14 8 14C4.68629 14 2 11.3137 2 8C2 4.68629 4.68629 2 8 2C11.3137 2 14 4.68629 14 8ZM15.5 8C15.5 12.1421 12.1421 15.5 8 15.5C3.85786 15.5 0.5 12.1421 0.5 8C0.5 3.85786 3.85786 0.5 8 0.5C12.1421 0.5 15.5 3.85786 15.5 8ZM8.75 5C8.75 4.58579 8.41421 4.25 8 4.25C7.58579 4.25 7.25 4.58579 7.25 5V8C7.25 8.19891 7.32902 8.38968 7.46967 8.53033L8.96967 10.0303C9.26256 10.3232 9.73744 10.3232 10.0303 10.0303C10.3232 9.73744 10.3232 9.26256 10.0303 8.96967L8.75 7.68934V5Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_28080_3060">
                    <rect width="16" height="16" fill="white" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const SettingsSmoothIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8.00035 4.99998C6.3435 4.99998 5.00035 6.34313 5.00035 7.99998C5.00035 9.65684 6.3435 11 8.00035 11C9.65721 11 11.0004 9.65684 11.0004 7.99998C11.0004 6.34313 9.65721 4.99998 8.00035 4.99998ZM6.50035 7.99998C6.50035 7.17156 7.17192 6.49998 8.00035 6.49998C8.82878 6.49998 9.50035 7.17156 9.50035 7.99998C9.50035 8.82841 8.82878 9.49998 8.00035 9.49998C7.17192 9.49998 6.50035 8.82841 6.50035 7.99998Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M6.12318 0.99993C5.54131 0.999471 5.08414 0.999111 4.66252 1.13794C4.29185 1.25999 3.95141 1.45953 3.66381 1.7233C3.33668 2.02334 3.11362 2.4224 2.82971 2.93031L1.02354 6.1556C0.750737 6.64187 0.536328 7.02407 0.449711 7.44234C0.373534 7.8102 0.373534 8.18977 0.449711 8.55763C0.536329 8.9759 0.750738 9.35809 1.02354 9.84437L2.82971 13.0697C3.11362 13.5776 3.33668 13.9766 3.66381 14.2767C3.95141 14.5404 4.29185 14.74 4.66252 14.862C5.08414 15.0009 5.54131 15.0005 6.12319 15H9.8775C10.4594 15.0005 10.9166 15.0009 11.3382 14.862C11.7088 14.74 12.0493 14.5404 12.3369 14.2767C12.664 13.9766 12.8871 13.5776 13.171 13.0696L14.9771 9.84435C15.2499 9.35808 15.4643 8.97589 15.551 8.55762C15.6271 8.18977 15.6271 7.8102 15.551 7.44234C15.4643 7.02408 15.2499 6.64189 14.9771 6.15561L13.171 2.93032C12.8871 2.42241 12.664 2.02335 12.3369 1.72331C12.0493 1.45954 11.7088 1.25999 11.3382 1.13794C10.9166 0.999111 10.4594 0.999471 9.87751 0.99993H6.12318ZM5.13165 2.56269C5.29493 2.50893 5.49177 2.49998 6.22594 2.49998H9.77475C10.5089 2.49998 10.7058 2.50893 10.869 2.56269C11.0375 2.61817 11.1923 2.70887 11.323 2.82877C11.4497 2.94497 11.5537 3.11234 11.9124 3.75291L13.6204 6.80291C13.9638 7.4161 14.0485 7.58442 14.0821 7.74651C14.1167 7.91372 14.1167 8.08625 14.0821 8.25346C14.0485 8.41554 13.9638 8.58387 13.6204 9.19706L11.9124 12.2471C11.5537 12.8876 11.4497 13.055 11.323 13.1712C11.1923 13.2911 11.0375 13.3818 10.869 13.4373C10.7058 13.491 10.5089 13.5 9.77475 13.5H6.22594C5.49177 13.5 5.29493 13.491 5.13165 13.4373C4.96317 13.3818 4.80842 13.2911 4.67769 13.1712C4.551 13.055 4.44702 12.8876 4.0883 12.2471L2.38029 9.19707C2.0369 8.58387 1.95211 8.41555 1.91855 8.25346C1.88392 8.08625 1.88392 7.91372 1.91855 7.74651C1.95211 7.58442 2.0369 7.41609 2.38029 6.8029L4.0883 3.7529C4.44702 3.11233 4.551 2.94496 4.67769 2.82877C4.80842 2.70887 4.96317 2.61817 5.13165 2.56269Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const NotCoinIcon: FC<{ className?: string; size?: string }> = ({
    className,
    size = '16'
}) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width={size}
            height={size}
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
            style={{ transform: 'rotateX(180deg)' }}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.14138 15.4996C7.2095 15.5943 7.34508 15.767 7.576 15.8689C7.84607 15.9881 8.15383 15.9881 8.4239 15.8689C8.65482 15.767 8.7904 15.5943 8.85852 15.4996C8.92424 15.4081 8.99102 15.2945 9.04911 15.1957L14.3316 6.21548C14.6645 5.64971 14.9405 5.18052 15.1235 4.79199C15.3081 4.40015 15.4553 3.97793 15.4105 3.52755C15.3472 2.89047 15.0155 2.31052 14.4985 1.93295C14.133 1.66602 13.6944 1.57879 13.2631 1.53922C12.8355 1.49998 12.2912 1.49999 11.6348 1.5H4.36532C3.70891 1.49999 3.16447 1.49998 2.73678 1.53922C2.30546 1.57879 1.86692 1.66602 1.50141 1.93294C0.984382 2.31052 0.652684 2.89047 0.589388 3.52755C0.544641 3.97793 0.691804 4.40015 0.876386 4.79199C1.05941 5.18052 1.33542 5.6497 1.66824 6.21547L6.95075 15.1956C7.00885 15.2945 7.07565 15.4081 7.14138 15.4996ZM8.74995 3L8.74994 12.7458L8.74995 12.7563L13.4336 4.80929C13.7831 4.21637 13.9578 3.91991 13.9337 3.67629C13.9127 3.46383 13.8022 3.27038 13.6298 3.14442C13.4322 3 13.088 3 12.3996 3H8.74995ZM7.99995 14.0208L7.98772 14H8.01218L7.99995 14.0208ZM7.24995 12.7563V3H3.60028C2.91192 3 2.56774 3 2.37007 3.14442C2.19769 3.27038 2.08716 3.46383 2.06617 3.67629C2.04211 3.91991 2.21684 4.21637 2.56629 4.80929L7.24995 12.7563Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ArrowUpIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                d="M8.75 13.25C8.75 13.6642 8.41421 14 8 14C7.58579 14 7.25 13.6642 7.25 13.25V4.56066L4.53033 7.28033C4.23744 7.57322 3.76256 7.57322 3.46967 7.28033C3.17678 6.98744 3.17678 6.51256 3.46967 6.21967L7.46967 2.21967C7.76256 1.92678 8.23744 1.92678 8.53033 2.21967L12.5303 6.21967C12.8232 6.51256 12.8232 6.98744 12.5303 7.28033C12.2374 7.57322 11.7626 7.57322 11.4697 7.28033L8.75 4.56066V13.25Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ArrowDownIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                d="M8.75 2.75C8.75 2.33578 8.41421 2 8 2C7.58579 2 7.25 2.33578 7.25 2.75V11.4393L4.53033 8.71967C4.23744 8.42678 3.76256 8.42678 3.46967 8.71967C3.17678 9.01256 3.17678 9.48744 3.46967 9.78033L7.46967 13.7803C7.76256 14.0732 8.23744 14.0732 8.53033 13.7803L12.5303 9.78033C12.8232 9.48744 12.8232 9.01256 12.5303 8.71967C12.2374 8.42678 11.7626 8.42678 11.4697 8.71967L8.75 11.4393V2.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SwapIcon: FC<{ className?: string; color?: string }> = ({ className, color }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M8.71967 6.28033C8.42678 5.98744 8.42678 5.51256 8.71967 5.21967L11.7197 2.21967C12.0126 1.92678 12.4874 1.92678 12.7803 2.21967L15.7803 5.21967C16.0732 5.51256 16.0732 5.98744 15.7803 6.28033C15.4874 6.57322 15.0126 6.57322 14.7197 6.28033L13 4.56066V12.75C13 13.1642 12.6642 13.5 12.25 13.5C11.8358 13.5 11.5 13.1642 11.5 12.75V4.56066L9.78033 6.28033C9.48744 6.57322 9.01256 6.57322 8.71967 6.28033Z"
                fill="currentColor"
            />
            <path
                d="M7.28033 9.71967C7.57322 10.0126 7.57322 10.4874 7.28033 10.7803L4.28033 13.7803C4.13968 13.921 3.94891 14 3.75 14C3.55109 14 3.36032 13.921 3.21967 13.7803L0.21967 10.7803C-0.0732233 10.4874 -0.0732232 10.0126 0.21967 9.71967C0.512563 9.42678 0.987437 9.42678 1.28033 9.71967L3 11.4393V3.25C3 2.83579 3.33579 2.5 3.75 2.5C4.16421 2.5 4.5 2.83579 4.5 3.25V11.4393L6.21967 9.71967C6.51256 9.42678 6.98744 9.42678 7.28033 9.71967Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const GlobeIcon: FC<{ className?: string }> = ({ className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme.iconSecondary}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M15 8C15 11.866 11.866 15 8 15C4.13401 15 1 11.866 1 8C1 4.13401 4.13401 1 8 1C11.866 1 15 4.13401 15 8ZM9.73386 8.75C9.61748 11.4328 8.88555 13.5 8 13.5C7.11445 13.5 6.38252 11.4328 6.26613 8.75H9.73386ZM11.2352 8.75C11.1815 10.0969 10.9838 11.3478 10.6668 12.3439C10.605 12.5381 10.5358 12.732 10.4579 12.9216C12.0486 12.1257 13.1996 10.5812 13.4493 8.75H11.2352ZM13.4493 7.25H11.2352C11.1815 5.90306 10.9838 4.65225 10.6668 3.65611C10.605 3.46195 10.5358 3.26799 10.4579 3.0784C12.0486 3.87435 13.1996 5.41878 13.4493 7.25ZM9.73386 7.25H6.26613C6.38251 4.56763 7.11423 2.5006 7.99961 2.5C8.88516 2.5 9.61748 4.56724 9.73386 7.25ZM4.76485 7.25C4.81852 5.90306 5.01622 4.65225 5.33317 3.65611C5.39495 3.46195 5.46421 3.268 5.54213 3.0784C3.95143 3.87435 2.8004 5.41878 2.55071 7.25H4.76485ZM2.55071 8.75C2.8004 10.5812 3.95143 12.1257 5.54213 12.9216C5.46421 12.732 5.39495 12.5381 5.33317 12.3439C5.01622 11.3478 4.81852 10.0969 4.76485 8.75H2.55071Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const XMarkCircleIcon: FC<{ className?: string; color?: string }> = ({
    color,
    className
}) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8 15.0039C11.866 15.0039 15 11.8699 15 8.00391C15 4.13791 11.866 1.00391 8 1.00391C4.13401 1.00391 1 4.13791 1 8.00391C1 11.8699 4.13401 15.0039 8 15.0039ZM6.28033 5.22358C5.98744 4.93068 5.51256 4.93068 5.21967 5.22358C4.92678 5.51647 4.92678 5.99134 5.21967 6.28424L6.93934 8.00391L5.21967 9.72358C4.92678 10.0165 4.92678 10.4913 5.21967 10.7842C5.51256 11.0771 5.98744 11.0771 6.28033 10.7842L8 9.06457L9.71967 10.7842C10.0126 11.0771 10.4874 11.0771 10.7803 10.7842C11.0732 10.4913 11.0732 10.0165 10.7803 9.72358L9.06066 8.00391L10.7803 6.28424C11.0732 5.99134 11.0732 5.51647 10.7803 5.22358C10.4874 4.93068 10.0126 4.93068 9.71967 5.22358L8 6.94325L6.28033 5.22358Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SparkIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <g clipPath="url(#clip0_28669_25847)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M5.60172 2.49649C6.30547 0.103713 9.69479 0.103707 10.3985 2.49649L11.1042 4.89579L13.5035 5.60147C15.8963 6.30523 15.8963 9.69454 13.5035 10.3983L11.1042 11.104L10.3985 13.5033C9.69479 15.8961 6.30548 15.8961 5.60172 13.5033L4.89604 11.104L2.49673 10.3983C0.103957 9.69454 0.103951 6.30523 2.49673 5.60147L4.89604 4.89579L5.60172 2.49649ZM8.9595 2.91974C8.67799 1.96263 7.32227 1.96263 7.04076 2.91974L6.33509 5.31904C6.19118 5.80831 5.80855 6.19094 5.31929 6.33484L2.91998 7.04052C1.96287 7.32202 1.96287 8.67775 2.91998 8.95925L5.31929 9.66493C5.80855 9.80883 6.19118 10.1915 6.33509 10.6807L7.04076 13.08C7.32227 14.0371 8.67799 14.0371 8.9595 13.08L9.66517 10.6807C9.80908 10.1915 10.1917 9.80883 10.681 9.66493L13.0803 8.95925C14.0374 8.67775 14.0374 7.32202 13.0803 7.04052L10.681 6.33484C10.1917 6.19094 9.80908 5.80831 9.66517 5.31904L8.9595 2.91974Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_28669_25847">
                    <rect width="16" height="16" fill="currentColor" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const FireIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8 14.5C10.4853 14.5 12.5 12.4853 12.5 9.99999C12.5 9.94273 12.4992 9.88711 12.4975 9.83311C12.3548 9.92627 12.2027 10.0067 12.0418 10.0724C11.1953 10.4178 10.1771 10.2984 9.43934 9.56064C8.91718 9.03848 8.74591 8.40751 8.72246 7.86822C8.70115 7.37806 8.79857 6.89641 8.87989 6.53994C8.92182 6.35614 8.96331 6.1858 9.00251 6.02488C9.14047 5.45852 9.25 5.0089 9.25 4.49998V4.49994L9.25 4.49982L9.25 4.49935L9.24998 4.49755L9.24986 4.49087L9.24903 4.46705C9.24815 4.44683 9.24655 4.41963 9.24379 4.3861C9.23825 4.31887 9.22832 4.22984 9.21115 4.12427C9.17657 3.91166 9.1146 3.64417 9.00712 3.35805C8.84492 2.92628 8.58841 2.47328 8.1772 2.08619C7.61284 3.99533 6.39014 5.38258 5.44404 6.45601C5.34929 6.56351 5.25731 6.66786 5.16905 6.76927C4.08569 8.01407 3.5 8.85912 3.5 9.99999C3.5 12.4853 5.51471 14.5 8 14.5ZM8.47275 1.06322C8.47276 1.06328 8.4721 1.06475 8.47071 1.0675C8.47204 1.06454 8.47273 1.06317 8.47275 1.06322ZM12.0447 8.22994C11.6014 8.80781 10.933 8.93298 10.5 8.49998C10.0505 8.05047 10.226 7.33698 10.4409 6.4631C10.5864 5.87152 10.75 5.20644 10.75 4.49998C10.75 4.08726 10.5796 1.80066 8.55151 0.480721C8.01498 0.13154 7.74671 -0.04305 7.43664 0.0766556C7.12657 0.196361 7.0139 0.625047 6.78858 1.48242C6.35134 3.14615 5.3044 4.33961 4.3006 5.48388C3.12048 6.82915 2 8.10644 2 9.99999C2 13.3137 4.68629 16 8 16C11.3137 16 14 13.3137 14 9.99999C14 9.27112 13.8907 8.69529 13.7311 8.24298C13.7199 8.21118 13.7084 8.18 13.6967 8.14942C13.5732 7.82646 13.4243 7.57042 13.2741 7.36923C13.085 7.11605 12.9905 6.98945 12.7853 7.01828C12.5801 7.04711 12.4959 7.26942 12.3274 7.71405C12.2537 7.90863 12.1597 8.07782 12.0522 8.22008C12.0497 8.22339 12.0472 8.22667 12.0447 8.22994Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CodeIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M10.4722 1.95754C10.5839 1.55867 10.3511 1.14478 9.95222 1.03309C9.55335 0.921411 9.13946 1.15422 9.02778 1.5531L5.52778 14.0531C5.41609 14.452 5.6489 14.8659 6.04778 14.9775C6.44665 15.0892 6.86054 14.8564 6.97222 14.4575L10.4722 1.95754Z"
                fill="currentColor"
            />
            <path
                d="M3.66603 3.63128C4.01067 3.86104 4.1038 4.3267 3.87404 4.67134L1.65139 8.00532L3.87404 11.3393C4.1038 11.6839 4.01067 12.1496 3.66603 12.3794C3.32138 12.6091 2.85573 12.516 2.62596 12.1713L0.125962 8.42134C-0.0419874 8.16942 -0.0419874 7.84122 0.125962 7.58929L2.62596 3.83929C2.85573 3.49465 3.32138 3.40152 3.66603 3.63128Z"
                fill="currentColor"
            />
            <path
                d="M12.334 3.63128C11.9893 3.86104 11.8962 4.3267 12.126 4.67134L14.3486 8.00532L12.126 11.3393C11.8962 11.6839 11.9893 12.1496 12.334 12.3794C12.6786 12.6091 13.1443 12.516 13.374 12.1713L15.874 8.42134C16.042 8.16942 16.042 7.84122 15.874 7.58929L13.374 3.83929C13.1443 3.49465 12.6786 3.40152 12.334 3.63128Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const HammerIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8.26824 1.90805C8.41887 1.8591 8.58113 1.8591 8.73177 1.90805C8.77135 1.92091 8.84474 1.95434 9.00059 2.08664C9.16313 2.22462 9.36097 2.42164 9.66673 2.7274L12.2726 5.33329C12.5784 5.63905 12.7754 5.83689 12.9134 5.99943C13.0457 6.15528 13.0791 6.22867 13.092 6.26825C13.1409 6.41889 13.1409 6.58115 13.092 6.73178C13.0791 6.77136 13.0457 6.84476 12.9134 7.0006C12.7754 7.16314 12.5784 7.36098 12.2726 7.66674L11.6667 8.27263C11.361 8.57839 11.1631 8.77541 11.0006 8.91339C10.8447 9.0457 10.7713 9.07913 10.7318 9.09199C10.5811 9.14093 10.4189 9.14093 10.2682 9.09199C10.2287 9.07913 10.1553 9.0457 9.99942 8.91339C9.83688 8.77541 9.63904 8.57839 9.33328 8.27263L8.03033 6.96969L6.72739 5.66674C6.42163 5.36098 6.22461 5.16314 6.08662 5.0006C5.95432 4.84476 5.92089 4.77136 5.90803 4.73178C5.85909 4.58115 5.85909 4.41889 5.90803 4.26825C5.92089 4.22867 5.95432 4.15528 6.08662 3.99943C6.22461 3.8369 6.42163 3.63905 6.72739 3.33329L7.33328 2.7274C7.63904 2.42164 7.83688 2.22462 7.99942 2.08664C8.15526 1.95434 8.22866 1.92091 8.26824 1.90805ZM9.19529 0.481461C8.74339 0.33463 8.25661 0.33463 7.80471 0.481461C7.5018 0.579882 7.25545 0.750597 7.02865 0.94313C6.81378 1.12555 6.57261 1.36673 6.29306 1.6463L6.29305 1.64631L6.27262 1.66674L5.66673 2.27263L5.64629 2.29307L5.64628 2.29308C5.36671 2.57263 5.12553 2.81379 4.94311 3.02867C4.75058 3.25546 4.57987 3.50182 4.48144 3.80473C4.33461 4.25663 4.33461 4.74341 4.48144 5.19531C4.57987 5.49822 4.75058 5.74457 4.94311 5.97137C5.12553 6.18624 5.36671 6.4274 5.64627 6.70695L5.64628 6.70695L5.66673 6.72741L6.43934 7.50002L1.21967 12.7197C0.926777 13.0126 0.926777 13.4875 1.21967 13.7803C1.51256 14.0732 1.98744 14.0732 2.28033 13.7803L7.5 8.56068L8.27262 9.33329L8.29307 9.35374C8.57262 9.63331 8.81378 9.87449 9.02865 10.0569C9.25545 10.2494 9.5018 10.4202 9.80471 10.5186C10.2566 10.6654 10.7434 10.6654 11.1953 10.5186C11.4982 10.4202 11.7446 10.2494 11.9714 10.0569C12.1862 9.87449 12.4274 9.63331 12.7069 9.35373L12.7274 9.33329L13.3333 8.7274L13.3537 8.70695L13.3538 8.70689C13.6333 8.42737 13.8745 8.18623 14.0569 7.97137C14.2494 7.74457 14.4201 7.49822 14.5186 7.19531C14.6654 6.74341 14.6654 6.25663 14.5186 5.80473C14.4201 5.50182 14.2494 5.25546 14.0569 5.02867C13.8745 4.8138 13.6333 4.57265 13.3538 4.29311L13.3537 4.29307L13.3333 4.27263L10.7274 1.66674L10.7069 1.64629C10.4274 1.36672 10.1862 1.12554 9.97135 0.94313C9.74456 0.750597 9.4982 0.579882 9.19529 0.481461ZM7.75002 13.5C7.33581 13.5 7.00002 13.8358 7.00002 14.25C7.00002 14.6642 7.33581 15 7.75002 15H14.25C14.6642 15 15 14.6642 15 14.25C15 13.8358 14.6642 13.5 14.25 13.5H7.75002Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ExitIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M6.46858 1.5C5.79522 1.49999 5.24449 1.49999 4.79693 1.53655C4.33367 1.5744 3.91527 1.65514 3.52453 1.85423C2.91301 2.16582 2.41582 2.66301 2.10423 3.27453C1.90514 3.66527 1.8244 4.08367 1.78655 4.54693C1.74999 4.99449 1.74999 5.54519 1.75 6.21855V9.78142C1.74999 10.4548 1.74999 11.0055 1.78655 11.4531C1.8244 11.9163 1.90514 12.3347 2.10423 12.7255C2.41582 13.337 2.91301 13.8342 3.52453 14.1458C3.91527 14.3449 4.33367 14.4256 4.79693 14.4634C5.24449 14.5 5.79518 14.5 6.46853 14.5H8.25C8.66422 14.5 9 14.1642 9 13.75C9 13.3358 8.66422 13 8.25 13H6.5C5.78756 13 5.29834 12.9994 4.91908 12.9684C4.54858 12.9382 4.34956 12.8827 4.20552 12.8093C3.87624 12.6415 3.60852 12.3738 3.44074 12.0445C3.36735 11.9004 3.31184 11.7014 3.28157 11.3309C3.25059 10.9517 3.25 10.4624 3.25 9.75V6.25C3.25 5.53756 3.25059 5.04833 3.28157 4.66908C3.31184 4.29858 3.36735 4.09956 3.44074 3.95552C3.60852 3.62623 3.87624 3.35852 4.20552 3.19074C4.34956 3.11735 4.54858 3.06184 4.91908 3.03157C5.29834 3.00058 5.78756 3 6.5 3H8.25C8.66422 3 9 2.66421 9 2.25C9 1.83579 8.66422 1.5 8.25 1.5H6.46858Z"
                fill="currentColor"
            />
            <path
                d="M12.2803 4.46967C11.9874 4.17678 11.5126 4.17678 11.2197 4.46967C10.9268 4.76256 10.9268 5.23744 11.2197 5.53033L12.9393 7.25H6.75C6.33579 7.25 6 7.58579 6 8C6 8.41421 6.33579 8.75 6.75 8.75H12.9393L11.2197 10.4697C10.9268 10.7626 10.9268 11.2374 11.2197 11.5303C11.5126 11.8232 11.9874 11.8232 12.2803 11.5303L15.2803 8.53033C15.5732 8.23744 15.5732 7.76256 15.2803 7.46967L12.2803 4.46967Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const EmptyIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M6.93728 1.08026C7.28424 1.02736 7.63917 1 8 1C8.36083 1 8.71577 1.02736 9.06272 1.08026C9.4722 1.14268 9.75355 1.52524 9.69113 1.93472C9.6287 2.3442 9.24615 2.62555 8.83666 2.56312C8.56427 2.5216 8.2849 2.5 8 2.5C7.7151 2.5 7.43574 2.5216 7.16334 2.56312C6.75385 2.62555 6.3713 2.3442 6.30888 1.93472C6.24645 1.52523 6.5278 1.14268 6.93728 1.08026Z"
                fill="currentColor"
            />
            <path
                d="M4.90745 2.51603C5.15271 2.84983 5.08094 3.31925 4.74715 3.56451C4.29553 3.89635 3.89635 4.29553 3.56451 4.74715C3.31925 5.08094 2.84983 5.15271 2.51603 4.90745C2.18224 4.66218 2.11047 4.19276 2.35574 3.85897C2.77758 3.28486 3.28486 2.77757 3.85897 2.35574C4.19276 2.11047 4.66218 2.18224 4.90745 2.51603Z"
                fill="currentColor"
            />
            <path
                d="M11.0926 2.51603C11.3378 2.18224 11.8072 2.11047 12.141 2.35574C12.7151 2.77758 13.2224 3.28486 13.6443 3.85897C13.8895 4.19276 13.8178 4.66218 13.484 4.90745C13.1502 5.15271 12.6808 5.08094 12.4355 4.74715C12.1037 4.29553 11.7045 3.89635 11.2529 3.56451C10.9191 3.31925 10.8473 2.84983 11.0926 2.51603Z"
                fill="currentColor"
            />
            <path
                d="M1.93472 6.30887C2.3442 6.3713 2.62555 6.75385 2.56312 7.16334C2.5216 7.43573 2.5 7.7151 2.5 8C2.5 8.2849 2.5216 8.56426 2.56312 8.83666C2.62555 9.24615 2.3442 9.6287 1.93472 9.69112C1.52523 9.75355 1.14268 9.4722 1.08026 9.06272C1.02736 8.71576 1 8.36083 1 8C1 7.63917 1.02736 7.28423 1.08026 6.93728C1.14268 6.5278 1.52524 6.24645 1.93472 6.30887Z"
                fill="currentColor"
            />
            <path
                d="M14.0653 6.30888C14.4748 6.24645 14.8573 6.5278 14.9197 6.93728C14.9726 7.28424 15 7.63917 15 8C15 8.36083 14.9726 8.71577 14.9197 9.06272C14.8573 9.4722 14.4748 9.75355 14.0653 9.69113C13.6558 9.6287 13.3745 9.24615 13.4369 8.83666C13.4784 8.56427 13.5 8.2849 13.5 8C13.5 7.7151 13.4784 7.43574 13.4369 7.16334C13.3745 6.75385 13.6558 6.3713 14.0653 6.30888Z"
                fill="currentColor"
            />
            <path
                d="M2.51603 11.0926C2.84983 10.8473 3.31925 10.9191 3.56451 11.2529C3.89635 11.7045 4.29553 12.1037 4.74715 12.4355C5.08094 12.6808 5.15271 13.1502 4.90745 13.484C4.66218 13.8178 4.19276 13.8895 3.85897 13.6443C3.28486 13.2224 2.77757 12.7151 2.35574 12.141C2.11047 11.8072 2.18224 11.3378 2.51603 11.0926Z"
                fill="currentColor"
            />
            <path
                d="M13.484 11.0926C13.8178 11.3378 13.8895 11.8072 13.6443 12.141C13.2224 12.7151 12.7151 13.2224 12.141 13.6443C11.8072 13.8895 11.3378 13.8178 11.0926 13.484C10.8473 13.1502 10.9191 12.6808 11.2529 12.4355C11.7045 12.1037 12.1037 11.7045 12.4355 11.2529C12.6808 10.9191 13.1502 10.8473 13.484 11.0926Z"
                fill="currentColor"
            />
            <path
                d="M6.30887 14.0653C6.3713 13.6558 6.75385 13.3745 7.16334 13.4369C7.43573 13.4784 7.7151 13.5 8 13.5C8.2849 13.5 8.56426 13.4784 8.83666 13.4369C9.24615 13.3745 9.6287 13.6558 9.69112 14.0653C9.75355 14.4748 9.4722 14.8573 9.06272 14.9197C8.71576 14.9726 8.36083 15 8 15C7.63917 15 7.28423 14.9726 6.93728 14.9197C6.5278 14.8573 6.24645 14.4748 6.30887 14.0653Z"
                fill="currentColor"
            />
        </svg>
    );
};

const SpinnerRingContainer = styled.div<{ color?: string }>`
    color: ${props => props.color};
    position: relative;
    width: 16px;
    height: 16px;
    display: flex;
    align-items: center;
    justify-content: center;

    @keyframes rotate {
        100% {
            transform: rotate(360deg);
        }
    }

    > svg:last-child {
        position: absolute;
        left: 0;
        top: 0;

        animation: rotate 1s linear infinite;
    }
`;

export const SpinnerRing: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <SpinnerRingContainer className={className} color={theme[color || 'iconSecondary']}>
            <svg
                xmlns="http://www.w3.org/2000/svg"
                width="14"
                height="14"
                viewBox="0 0 14 14"
                fill="none"
            >
                <path
                    opacity="0.32"
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M7 2C4.23858 2 2 4.23858 2 7C2 9.76142 4.23858 12 7 12C9.76142 12 12 9.76142 12 7C12 4.23858 9.76142 2 7 2ZM0 7C0 3.13401 3.13401 0 7 0C10.866 0 14 3.13401 14 7C14 10.866 10.866 14 7 14C3.13401 14 0 10.866 0 7Z"
                    fill="currentColor"
                />
            </svg>

            <svg
                xmlns="http://www.w3.org/2000/svg"
                width="16"
                height="16"
                viewBox="0 0 16 16"
                fill="none"
            >
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M5.49968 3.66992C3.10821 5.05063 2.28884 8.10858 3.66955 10.5C3.94569 10.9783 3.78182 11.5899 3.30353 11.8661C2.82523 12.1422 2.21364 11.9783 1.9375 11.5C0.00450337 8.152 1.15163 3.87086 4.49968 1.93787C4.97797 1.66172 5.58956 1.8256 5.8657 2.30389C6.14185 2.78218 5.97797 3.39377 5.49968 3.66992Z"
                    fill="currentColor"
                />
            </svg>
        </SpinnerRingContainer>
    );
};

export const ArrowLeftIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M13.25 8.75C13.6642 8.75 14 8.41421 14 8C14 7.58579 13.6642 7.25 13.25 7.25L4.56066 7.25L7.28033 4.53033C7.57322 4.23744 7.57322 3.76256 7.28033 3.46967C6.98744 3.17678 6.51256 3.17678 6.21967 3.46967L2.21967 7.46967C1.92678 7.76256 1.92678 8.23744 2.21967 8.53033L6.21967 12.5303C6.51256 12.8232 6.98744 12.8232 7.28033 12.5303C7.57322 12.2374 7.57322 11.7626 7.28033 11.4697L4.56066 8.75H13.25Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const TonkeeperSkeletIcon: FC<{ className?: string; color?: string }> = ({
    color,
    className
}) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.55657 0.962013C7.85024 0.91403 8.14976 0.91403 8.44344 0.962013C8.77676 1.01647 9.0896 1.15077 9.48706 1.32139C9.50976 1.33113 9.53272 1.34099 9.55599 1.35096L15.2954 3.81073C15.4989 3.89795 15.6535 4.07062 15.7177 4.2825C15.782 4.49438 15.7493 4.72381 15.6285 4.90934L8.6285 15.6593C8.4901 15.8719 8.25364 16.0001 8 16.0001C7.74636 16.0001 7.50991 15.8719 7.3715 15.6593L0.371505 4.90934C0.250692 4.72381 0.218028 4.49438 0.282262 4.2825C0.346496 4.07062 0.501063 3.89795 0.704564 3.81073L6.44402 1.35096C6.46728 1.34099 6.49025 1.33113 6.51294 1.32139C6.91041 1.15077 7.22325 1.01647 7.55657 0.962013ZM8.20157 2.44238C8.06808 2.42057 7.93193 2.42057 7.79844 2.44238C7.6689 2.46355 7.53363 2.51594 7.0349 2.72968L2.90395 4.50009L7.0349 6.2705C7.53363 6.48424 7.6689 6.53663 7.79844 6.5578C7.93193 6.57961 8.06808 6.57961 8.20157 6.5578C8.3311 6.53663 8.46637 6.48424 8.96511 6.2705L13.0961 4.50009L8.96511 2.72968C8.46637 2.51594 8.3311 2.46355 8.20157 2.44238ZM13.0216 6.16397L9.55599 7.64922C9.53272 7.65919 9.50975 7.66905 9.48706 7.67879C9.21255 7.79663 8.97841 7.89714 8.75 7.96599V12.7239L13.0216 6.16397ZM7.25 12.7239V7.96599C7.0216 7.89714 6.78746 7.79663 6.51295 7.67879C6.49026 7.66905 6.46729 7.65919 6.44402 7.64922L2.97845 6.16397L7.25 12.7239Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const AppearanceIcon: FC<{ className?: string; color?: string }> = ({
    color,
    className
}) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M15 8C15 11.866 11.866 15 8 15C4.13401 15 1 11.866 1 8C1 4.13401 4.13401 1 8 1C11.866 1 15 4.13401 15 8ZM13.5 8C13.5 11.0376 11.0376 13.5 8 13.5V2.5C11.0376 2.5 13.5 4.96243 13.5 8Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PlaceIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M13 6.5C13 7.80774 12.5304 9.16058 11.7813 10.4709C11.036 11.7747 10.0621 12.9522 9.17446 13.8822C8.51294 14.5753 7.48706 14.5753 6.82553 13.8822C5.93793 12.9522 4.964 11.7747 4.21872 10.4709C3.46964 9.16058 3 7.80774 3 6.5C3 3.73858 5.23858 1.5 8 1.5C10.7614 1.5 13 3.73858 13 6.5ZM14.5 6.5C14.5 9.81674 12.1362 12.9516 10.2595 14.9179C9.0071 16.2301 6.9929 16.2301 5.74045 14.9179C3.8638 12.9516 1.5 9.81674 1.5 6.5C1.5 2.91015 4.41015 0 8 0C11.5899 0 14.5 2.91015 14.5 6.5ZM9.5 6.5C9.5 7.32843 8.82843 8 8 8C7.17157 8 6.5 7.32843 6.5 6.5C6.5 5.67157 7.17157 5 8 5C8.82843 5 9.5 5.67157 9.5 6.5ZM11 6.5C11 8.15685 9.65685 9.5 8 9.5C6.34315 9.5 5 8.15685 5 6.5C5 4.84315 6.34315 3.5 8 3.5C9.65685 3.5 11 4.84315 11 6.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const EnvelopeIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M6.05 1.25H6.0179H6.01788C5.20505 1.24999 4.54944 1.24999 4.01853 1.29336C3.4719 1.33803 2.99175 1.43239 2.54754 1.65873C1.84193 2.01825 1.26825 2.59193 0.908726 3.29754C0.682386 3.74175 0.588026 4.2219 0.543364 4.76853C0.499987 5.29944 0.499993 5.95505 0.5 6.76788V6.7679V6.8V8.7V8.73211V8.73212C0.499993 9.54495 0.499987 10.2006 0.543364 10.7315C0.588026 11.2781 0.682386 11.7582 0.908726 12.2025C1.26825 12.9081 1.84193 13.4817 2.54754 13.8413C2.99175 14.0676 3.4719 14.162 4.01853 14.2066C4.54944 14.25 5.20505 14.25 6.01788 14.25H6.05H9.95H9.98212C10.795 14.25 11.4506 14.25 11.9815 14.2066C12.5281 14.162 13.0082 14.0676 13.4525 13.8413C14.1581 13.4817 14.7317 12.9081 15.0913 12.2025C15.3176 11.7582 15.412 11.2781 15.4566 10.7315C15.5 10.2006 15.5 9.54496 15.5 8.73212V8.7V6.8V6.76788C15.5 5.95505 15.5 5.29944 15.4566 4.76853C15.412 4.2219 15.3176 3.74175 15.0913 3.29754C14.7317 2.59193 14.1581 2.01825 13.4525 1.65873C13.0082 1.43239 12.5281 1.33803 11.9815 1.29336C11.4506 1.24999 10.7949 1.24999 9.98212 1.25H9.9821H9.95H6.05ZM3.22852 2.99524C3.42604 2.8946 3.68681 2.82547 4.14068 2.78838C4.60331 2.75058 5.19755 2.75 6.05 2.75H9.95C10.8025 2.75 11.3967 2.75058 11.8593 2.78838C12.3132 2.82547 12.574 2.8946 12.7715 2.99524C12.8057 3.01267 12.8394 3.03094 12.8725 3.05002L9.31686 6.09774C8.68385 6.64032 7.75598 6.661 7.09943 6.14718L3.13583 3.04523C3.16628 3.02788 3.19718 3.0112 3.22852 2.99524ZM2.16008 4.18635C2.10449 4.35802 2.06387 4.57874 2.03838 4.89068C2.00058 5.35331 2 5.94755 2 6.8V8.7C2 9.55245 2.00058 10.1467 2.03838 10.6093C2.07547 11.0632 2.1446 11.324 2.24524 11.5215C2.46095 11.9448 2.80516 12.289 3.22852 12.5048C3.42604 12.6054 3.68681 12.6745 4.14068 12.7116C4.60331 12.7494 5.19755 12.75 6.05 12.75H9.95C10.8025 12.75 11.3967 12.7494 11.8593 12.7116C12.3132 12.6745 12.574 12.6054 12.7715 12.5048C13.1948 12.289 13.539 11.9448 13.7548 11.5215C13.8554 11.324 13.9245 11.0632 13.9616 10.6093C13.9994 10.1467 14 9.55245 14 8.7V6.8C14 5.94755 13.9994 5.35331 13.9616 4.89068C13.9365 4.58352 13.8967 4.3648 13.8425 4.19427L10.293 7.23663C9.11747 8.24427 7.39428 8.28268 6.17497 7.32844L2.16008 4.18635Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const DocIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M8.49904 0.250092L6.7679 0.250092C5.95506 0.250085 5.29944 0.250079 4.76853 0.293456C4.2219 0.338118 3.74176 0.432477 3.29754 0.658818C2.59193 1.01834 2.01825 1.59202 1.65873 2.29763C1.43239 2.74185 1.33803 3.22199 1.29336 3.76862C1.24999 4.29953 1.24999 4.95513 1.25 5.76796V10.2322C1.24999 11.045 1.24999 11.7007 1.29336 12.2316C1.33803 12.7782 1.43239 13.2583 1.65873 13.7026C2.01825 14.4082 2.59193 14.9818 3.29754 15.3414C3.74176 15.5677 4.2219 15.6621 4.76853 15.7067C5.29944 15.7501 5.95505 15.7501 6.76788 15.7501H9.23212C10.045 15.7501 10.7006 15.7501 11.2315 15.7067C11.7781 15.6621 12.2582 15.5677 12.7025 15.3414C13.4081 14.9818 13.9818 14.4082 14.3413 13.7026C14.5676 13.2583 14.662 12.7782 14.7066 12.2316C14.75 11.7007 14.75 11.045 14.75 10.2322L14.75 6.27252C14.7504 6.02283 14.7507 5.78386 14.6947 5.55097C14.6457 5.3469 14.5649 5.15181 14.4553 4.97286C14.3301 4.76864 14.161 4.59988 13.9842 4.42354L10.5766 1.01592C10.4002 0.839144 10.2315 0.669958 10.0272 0.544812C9.84829 0.435154 9.6532 0.354346 9.44912 0.305352C9.21623 0.249439 8.97726 0.249736 8.72758 0.250046L8.50096 0.250092C8.50064 0.250092 8.50128 0.250092 8.50096 0.250092C8.50064 0.250092 8.49936 0.250092 8.49904 0.250092ZM7.75 1.75009V4.08111C7.74999 4.48479 7.74998 4.83214 7.77334 5.11807C7.79797 5.41958 7.85228 5.7182 7.99851 6.00519C8.2171 6.4342 8.56589 6.783 8.9949 7.00159C9.2819 7.14782 9.58051 7.20212 9.88203 7.22676C10.168 7.25012 10.5153 7.25011 10.919 7.25009L13.25 7.25009V10.6501C13.25 11.9102 13.25 12.5403 13.0048 13.0216C12.7891 13.4449 12.4448 13.7891 12.0215 14.0049C11.5402 14.2501 10.9101 14.2501 9.65 14.2501H6.35C5.08988 14.2501 4.45982 14.2501 3.97852 14.0049C3.55516 13.7891 3.21095 13.4449 2.99524 13.0216C2.75 12.5403 2.75 11.9102 2.75 10.6501V5.35009C2.75 4.08997 2.75 3.45991 2.99524 2.97861C3.21095 2.55525 3.55516 2.21104 3.97852 1.99533C4.45982 1.75009 5.08988 1.75009 6.35 1.75009H7.75ZM13.1722 5.75009C13.1513 5.718 13.1096 5.67038 12.8839 5.44464L9.55546 2.11621C9.32971 1.89046 9.28209 1.8488 9.25 1.82789V4.55009C9.25 4.97013 9.25 5.18015 9.33175 5.34058C9.40365 5.48171 9.51839 5.59644 9.65951 5.66835C9.81994 5.75009 10.03 5.75009 10.45 5.75009H13.1722Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const KeyIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <rect x="4.75" y="4.75024" width="2" height="2" rx="1" fill="currentColor" />
            <path
                d="M10.9302 7.1804L10.1915 7.05073C10.1492 7.29154 10.2269 7.53785 10.3998 7.71073L10.9302 7.1804ZM7.18016 10.9304L7.71049 10.4001C7.5376 10.2272 7.2913 10.1494 7.05048 10.1917L7.18016 10.9304ZM8.75 12.5002L8.21967 13.0306C8.36032 13.1712 8.55109 13.2502 8.75 13.2502V12.5002ZM9.75 12.5002H10.5C10.5 12.086 10.1642 11.7502 9.75 11.7502V12.5002ZM9.75 13.5002H9C9 13.6992 9.07902 13.8899 9.21967 14.0306L9.75 13.5002ZM10.8376 14.5292L10.4457 15.1687L10.4457 15.1687L10.8376 14.5292ZM11.2712 14.7088L11.0961 15.4381L11.2712 14.7088ZM14.0675 14.614L13.727 13.9457H13.727L14.0675 14.614ZM14.6138 14.0677L13.9455 13.7272V13.7272L14.6138 14.0677ZM14.7086 11.2714L15.4378 11.0963L14.7086 11.2714ZM14.529 10.8378L15.1684 10.4459L15.1684 10.4459L14.529 10.8378ZM10.25 6.37524C10.25 6.60622 10.2299 6.83188 10.1915 7.05073L11.6689 7.31008C11.7222 7.00597 11.75 6.69354 11.75 6.37524H10.25ZM6.375 2.50024C8.5151 2.50024 10.25 4.23514 10.25 6.37524H11.75C11.75 3.40671 9.34353 1.00024 6.375 1.00024V2.50024ZM2.5 6.37524C2.5 4.23514 4.2349 2.50024 6.375 2.50024V1.00024C3.40647 1.00024 1 3.40671 1 6.37524H2.5ZM6.375 10.2502C4.2349 10.2502 2.5 8.51535 2.5 6.37524H1C1 9.34377 3.40647 11.7502 6.375 11.7502V10.2502ZM7.05048 10.1917C6.83163 10.2301 6.60597 10.2502 6.375 10.2502V11.7502C6.6933 11.7502 7.00573 11.7225 7.30984 11.6691L7.05048 10.1917ZM6.64983 11.4607L8.21967 13.0306L9.28033 11.9699L7.71049 10.4001L6.64983 11.4607ZM8.75 13.2502H9.75V11.7502H8.75V13.2502ZM9 12.5002V13.5002H10.5V12.5002H9ZM9.21967 14.0306L9.76673 14.5776L10.8274 13.517L10.2803 12.9699L9.21967 14.0306ZM9.76673 14.5776C10.0009 14.8118 10.201 15.0188 10.4457 15.1687L11.2294 13.8897C11.1714 13.8541 11.1121 13.8016 10.8274 13.517L9.76673 14.5776ZM10.4457 15.1687C10.647 15.292 10.8665 15.383 11.0961 15.4381L11.4462 13.9795C11.3697 13.9611 11.2966 13.9308 11.2294 13.8897L10.4457 15.1687ZM11.0961 15.4381C11.3751 15.5051 11.6629 15.5002 11.9941 15.5002V14.0002C11.5915 14.0002 11.5125 13.9954 11.4462 13.9795L11.0961 15.4381ZM11.9941 15.5002H12.75V14.0002H11.9941V15.5002ZM12.75 15.5002C13.0877 15.5002 13.386 15.5008 13.6321 15.4807C13.8869 15.4599 14.1509 15.4132 14.408 15.2823L13.727 13.9457C13.7167 13.951 13.672 13.9725 13.5099 13.9857C13.3391 13.9997 13.1124 14.0002 12.75 14.0002V15.5002ZM14.408 15.2823C14.7843 15.0905 15.0903 14.7845 15.282 14.4082L13.9455 13.7272C13.8976 13.8213 13.8211 13.8978 13.727 13.9457L14.408 15.2823ZM15.282 14.4082C15.413 14.1512 15.4597 13.8871 15.4805 13.6323C15.5006 13.3863 15.5 13.0879 15.5 12.7502H14C14 13.1127 13.9994 13.3394 13.9855 13.5102C13.9722 13.6722 13.9508 13.7169 13.9455 13.7272L15.282 14.4082ZM15.5 12.7502V11.9944H14V12.7502H15.5ZM15.5 11.9944C15.5 11.6632 15.5048 11.3753 15.4378 11.0963L13.9793 11.4465C13.9952 11.5127 14 11.5918 14 11.9944H15.5ZM15.4378 11.0963C15.3827 10.8667 15.2918 10.6473 15.1684 10.4459L13.8895 11.2297C13.9306 11.2968 13.9609 11.37 13.9793 11.4465L15.4378 11.0963ZM15.1684 10.4459C15.0185 10.2013 14.8116 10.0012 14.5774 9.76697L13.5167 10.8276C13.8014 11.1123 13.8539 11.1716 13.8895 11.2297L15.1684 10.4459ZM14.5774 9.76697L11.4605 6.65007L10.3998 7.71073L13.5167 10.8276L14.5774 9.76697Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const SwitchIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2.64626 5.69515C2.89197 6.02861 3.36148 6.09975 3.69495 5.85404L8.00005 2.68186L12.3052 5.85404C12.6386 6.09975 13.1081 6.02861 13.3538 5.69515C13.5996 5.36168 13.5284 4.89216 13.1949 4.64645L8.44495 1.14645C8.18038 0.951508 7.81972 0.951508 7.55515 1.14645L2.80515 4.64645C2.47168 4.89216 2.40055 5.36168 2.64626 5.69515ZM2.64626 10.3054C2.89197 9.97198 3.36148 9.90084 3.69495 10.1466L8.00005 13.3187L12.3052 10.1466C12.6386 9.90084 13.1081 9.97198 13.3538 10.3054C13.5996 10.6389 13.5284 11.1084 13.1949 11.3541L8.44495 14.8541C8.18038 15.0491 7.81972 15.0491 7.55515 14.8541L2.80515 11.3541C2.47168 11.1084 2.40055 10.6389 2.64626 10.3054Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const BankIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M9.21847 0.571156C8.46118 0.148482 7.53893 0.148482 6.78164 0.571156L1.6167 3.45391C-0.117392 4.42178 0.308017 6.88187 2.00005 7.40236V13.5H1.75005C1.33584 13.5 1.00005 13.8358 1.00005 14.25C1.00005 14.6643 1.33584 15 1.75005 15H14.2501C14.6643 15 15.0001 14.6643 15.0001 14.25C15.0001 13.8358 14.6643 13.5 14.2501 13.5H14.0001V7.40236C15.6921 6.88187 16.1175 4.42178 14.3834 3.45391L9.21847 0.571156ZM12.5001 7.50002H10.5001V13.5H12.5001V7.50002ZM9.00005 7.50002H7.00005V13.5H9.00005V7.50002ZM5.50005 7.50002H3.50005V13.5H5.50005V7.50002ZM7.51269 1.88095C7.8156 1.71188 8.1845 1.71188 8.48742 1.88095L13.6524 4.76371C14.2464 5.09529 14.011 6.00002 13.3307 6.00002H2.66941C1.98907 6.00002 1.75367 5.09529 2.34775 4.76371L7.51269 1.88095Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ErrorIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="84"
            height="84"
            viewBox="0 0 84 84"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M42 75C60.2254 75 75 60.2254 75 42C75 23.7746 60.2254 9 42 9C23.7746 9 9 23.7746 9 42C9 60.2254 23.7746 75 42 75ZM38.1817 28.9959C38.0827 26.819 39.8209 25 42 25C44.1792 25 45.9174 26.819 45.8184 28.9959L45.1362 44.0031C45.06 45.6798 43.6784 47 42 47C40.3216 47 38.94 45.6798 38.8638 44.0031L38.1817 28.9959ZM46 55C46 57.2091 44.2091 59 42 59C39.7909 59 38 57.2091 38 55C38 52.7909 39.7909 51 42 51C44.2091 51 46 52.7909 46 55Z"
                fill="#8994A3"
            />
            <path
                opacity="0.32"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M42 25C39.8209 25 38.0827 26.819 38.1817 28.9959L38.8638 44.0031C38.94 45.6798 40.3216 47 42 47C43.6784 47 45.06 45.6798 45.1362 44.0031L45.8184 28.9959C45.9174 26.819 44.1792 25 42 25ZM42 59C44.2091 59 46 57.2091 46 55C46 52.7909 44.2091 51 42 51C39.7909 51 38 52.7909 38 55C38 57.2091 39.7909 59 42 59Z"
                fill="#8994A3"
            />
        </svg>
    );
};

export const DotIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <circle cx="8" cy="8" r="3" fill="currentColor" />
        </svg>
    );
};

export const ResponsiveSpinner: FC<{ className?: string }> = ({ className }) => {
    const { displayType } = useTheme();

    if (displayType === 'full-width') {
        return <SpinnerRing className={className} />;
    }

    return <SpinnerIcon className={className} />;
};

export const LockIcon = () => {
    return (
        <svg
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M10 4V6H6V4C6 2.89543 6.89543 2 8 2C9.10457 2 10 2.89543 10 4ZM4.5 6.0143V6V4C4.5 2.067 6.067 0.5 8 0.5C9.933 0.5 11.5 2.067 11.5 4V6V6.0143C12.1748 6.03943 12.6236 6.10871 12.9985 6.29973C13.5159 6.56338 13.9366 6.98408 14.2003 7.50153C14.5 8.08978 14.5 8.85986 14.5 10.4V11.1C14.5 12.6401 14.5 13.4102 14.2003 13.9985C13.9366 14.5159 13.5159 14.9366 12.9985 15.2003C12.4102 15.5 11.6401 15.5 10.1 15.5H5.9C4.35986 15.5 3.58978 15.5 3.00153 15.2003C2.48408 14.9366 2.06338 14.5159 1.79973 13.9985C1.5 13.4102 1.5 12.6401 1.5 11.1V10.4C1.5 8.85986 1.5 8.08978 1.79973 7.50153C2.06338 6.98408 2.48408 6.56338 3.00153 6.29973C3.37642 6.10871 3.82516 6.03943 4.5 6.0143ZM7.25 9.25C7.25 8.83579 7.58579 8.5 8 8.5C8.41421 8.5 8.75 8.83579 8.75 9.25V10.75C8.75 11.1642 8.41421 11.5 8 11.5C7.58579 11.5 7.25 11.1642 7.25 10.75V9.25Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const RefreshIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2.5 8C2.5 4.96243 4.96243 2.5 8 2.5C9.82096 2.5 11.4363 3.38501 12.4377 4.75H10.5C10.0858 4.75 9.75 5.08579 9.75 5.5C9.75 5.91421 10.0858 6.25 10.5 6.25H13.7135C13.7247 6.25025 13.7359 6.25025 13.7471 6.25H14.25C14.6642 6.25 15 5.91421 15 5.5V1.75C15 1.33579 14.6642 1 14.25 1C13.8358 1 13.5 1.33579 13.5 1.75V3.66963C12.2187 2.04438 10.2318 1 8 1C4.13401 1 1 4.13401 1 8C1 11.866 4.13401 15 8 15C10.7611 15 13.1473 13.4014 14.2863 11.0826C14.4689 10.7108 14.3156 10.2613 13.9438 10.0787C13.572 9.89611 13.1226 10.0495 12.9399 10.4212C12.0435 12.2464 10.1674 13.5 8 13.5C4.96243 13.5 2.5 11.0376 2.5 8Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const RefreshIconAnimated: FC<{ className?: string; color?: string }> = ({
    color,
    className
}) => {
    const theme = useTheme();
    const clipId = useId().replaceAll(':', '_');
    const arrowId = useId().replaceAll(':', '_');

    return (
        <svg
            height="16px"
            width="16px"
            viewBox="0 0 16 16"
            xmlns="http://www.w3.org/2000/svg"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                d="M2.5 8C2.5 4.96243 4.96243 2.5 8 2.5C9.82096 2.5 11.4363 3.38501 12.4377 4.75H10.5C10.0858 4.75 9.75 5.08579 9.75 5.5C9.75 5.91421 10.0858 6.25 10.5 6.25H13.7135C13.7247 6.25025 13.7359 6.25025 13.7471 6.25H14.25C14.6642 6.25 15 5.91421 15 5.5V1.75C15 1.33579 14.6642 1 14.25 1C13.8358 1 13.5 1.33579 13.5 1.75V3.66963C12.2187 2.04438 10.2318 1 8 1C4.13401 1 1 4.13401 1 8C1 11.866 4.13401 15 8 15C10.7611 15 13.1473 13.4014 14.2863 11.0826C14.4689 10.7108 14.3156 10.2613 13.9438 10.0787C13.572 9.89611 13.1226 10.0495 12.9399 10.4212C12.0435 12.2464 10.1674 13.5 8 13.5C4.96243 13.5 2.5 11.0376 2.5 8Z"
                stroke="none"
                fill={theme.iconPrimary}
            />
            <defs>
                <path
                    d="M2.5 8C2.5 4.96243 4.96243 2.5 8 2.5C9.82096 2.5 11.4363 3.38501 12.4377 4.75H10.5C10.0858 4.75 9.75 5.08579 9.75 5.5C9.75 5.91421 10.0858 6.25 10.5 6.25H13.7135C13.7247 6.25025 13.7359 6.25025 13.7471 6.25H14.25C14.6642 6.25 15 5.91421 15 5.5V1.75C15 1.33579 14.6642 1 14.25 1C13.8358 1 13.5 1.33579 13.5 1.75V3.66963C12.2187 2.04438 10.2318 1 8 1C4.13401 1 1 4.13401 1 8C1 11.866 4.13401 15 8 15C10.7611 15 13.1473 13.4014 14.2863 11.0826C14.4689 10.7108 14.3156 10.2613 13.9438 10.0787C13.572 9.89611 13.1226 10.0495 12.9399 10.4212C12.0435 12.2464 10.1674 13.5 8 13.5C4.96243 13.5 2.5 11.0376 2.5 8Z"
                    id={arrowId}
                    stroke="none"
                    fill="none"
                />
                <clipPath id={clipId}>
                    <use href={`#${arrowId}`}></use>
                </clipPath>
            </defs>
            <g clipPath={`url(#${clipId})`}>
                <circle
                    cx="8"
                    cy="8"
                    r="5"
                    transform="rotate(360,12,12)"
                    fill="none"
                    stroke="currentColor"
                    strokeWidth="10"
                    strokeDasharray="30"
                    strokeDashoffset="0"
                >
                    <animate
                        attributeName="stroke-dashoffset"
                        values="60;30"
                        begin="0"
                        repeatCount="indefinite"
                        dur="15s"
                    ></animate>
                </circle>
            </g>
            <use href={`#${arrowId}`}></use>
        </svg>
    );
};

export const InfoCircleIcon: FC<{ className?: string; color?: string }> = ({
    color,
    className
}) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="12"
            height="12"
            viewBox="0 0 12 12"
            fill="none"
            color={theme[color || 'iconTertiary']}
            className={className}
        >
            <g clipPath="url(#clip0_33740_221833)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M0 6C0 2.68629 2.68629 0 6 0C9.31371 0 12 2.68629 12 6C12 9.31371 9.31371 12 6 12C2.68629 12 0 9.31371 0 6ZM5.1001 3.4C5.1001 3.89706 5.50304 4.3 6.0001 4.3C6.49715 4.3 6.9001 3.89706 6.9001 3.4C6.9001 2.90295 6.49715 2.5 6.0001 2.5C5.50304 2.5 5.1001 2.90295 5.1001 3.4ZM6 5.3C5.58579 5.3 5.25 5.63579 5.25 6.05V8.75C5.25 9.16421 5.58579 9.5 6 9.5C6.41421 9.5 6.75 9.16421 6.75 8.75V6.05C6.75 5.63579 6.41421 5.3 6 5.3Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_33740_221833">
                    <rect width="12" height="12" fill="white" transform="matrix(1 0 0 -1 0 12)" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const LinkOutIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            color={theme[color || 'iconTertiary']}
            className={className}
        >
            <path
                d="M8.49991 2.75098C8.49991 3.16519 8.8357 3.50098 9.24991 3.50098H11.4393L6.71967 8.22065C6.42678 8.51354 6.42678 8.98841 6.71967 9.28131C7.01256 9.5742 7.48744 9.5742 7.78033 9.28131L12.4999 4.56173V6.75098C12.4999 7.16519 12.8357 7.50098 13.2499 7.50098C13.6641 7.50098 13.9999 7.16519 13.9999 6.75098V2.76261C14.0029 2.56681 13.9297 2.37005 13.7803 2.22065C13.6986 2.13889 13.6026 2.07996 13.5004 2.04384C13.4221 2.01608 13.3378 2.00098 13.2499 2.00098H9.24991C8.8357 2.00098 8.49991 2.33676 8.49991 2.75098Z"
                fill="currentColor"
            />
            <path
                d="M6.24325 2.00101L6.22138 2.00101C5.66277 2.001 5.20171 2.00099 4.82525 2.0295C4.43457 2.05908 4.07438 2.1225 3.73215 2.28208C3.09292 2.58016 2.57915 3.09393 2.28108 3.73315C2.12149 4.07538 2.05808 4.43557 2.02849 4.82625C1.99999 5.2027 1.99999 5.66376 2 6.22235V9.68198C1.99999 10.2857 1.99998 10.784 2.03315 11.1899C2.06759 11.6114 2.14151 11.999 2.32698 12.363C2.6146 12.9275 3.07354 13.3864 3.63803 13.674C4.00203 13.8595 4.38956 13.9334 4.81113 13.9679C5.21702 14.001 5.71527 14.001 6.31897 14.001H9.77862C10.3372 14.001 10.7983 14.001 11.1748 13.9725C11.5654 13.9429 11.9256 13.8795 12.2679 13.7199C12.9071 13.4219 13.4208 12.9081 13.7189 12.2689C13.8785 11.9266 13.9419 11.5664 13.9715 11.1758C14 10.7993 14 10.3382 14 9.77963V9.75101C14 9.33679 13.6642 9.00101 13.25 9.00101C12.8358 9.00101 12.5 9.33679 12.5 9.75101C12.5 10.3454 12.4995 10.7494 12.4758 11.0625C12.4527 11.3678 12.4107 11.5251 12.3595 11.6349C12.2104 11.9545 11.9535 12.2114 11.6339 12.3605C11.5241 12.4117 11.3668 12.4537 11.0615 12.4768C10.7484 12.5005 10.3444 12.501 9.75 12.501H6.35C5.70757 12.501 5.27085 12.5004 4.93328 12.4728C4.60447 12.446 4.43632 12.3973 4.31902 12.3375C4.03677 12.1937 3.8073 11.9642 3.66349 11.682C3.60372 11.5647 3.55503 11.3965 3.52816 11.0677C3.50058 10.7302 3.5 10.2934 3.5 9.65101V6.25101C3.5 5.65664 3.5005 5.25261 3.52421 4.93952C3.54733 4.63424 3.58932 4.47692 3.64054 4.36708C3.78958 4.04747 4.04646 3.79058 4.36607 3.64154C4.47592 3.59032 4.63324 3.54833 4.93851 3.52522C5.2516 3.50151 5.65563 3.50101 6.25 3.50101C6.66422 3.50101 7 3.16519 7 2.75098C7 2.33676 6.66421 2.00098 6.25 2.00098C6.24775 2.00098 6.2455 2.00099 6.24325 2.00101Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const AppsIcon: FC<{ className?: string; color?: string }> = ({ color, className }) => {
    const theme = useTheme();
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            color={theme[color || 'iconSecondary']}
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M1 4.125C1 3.07559 1 2.55088 1.19202 2.14489C1.38986 1.7266 1.7266 1.38986 2.14489 1.19202C2.55088 1 3.07559 1 4.125 1C5.17441 1 5.69912 1 6.10511 1.19202C6.5234 1.38986 6.86014 1.7266 7.05798 2.14489C7.25 2.55088 7.25 3.07559 7.25 4.125C7.25 5.17441 7.25 5.69912 7.05798 6.10511C6.86014 6.5234 6.5234 6.86014 6.10511 7.05798C5.69912 7.25 5.17441 7.25 4.125 7.25C3.07559 7.25 2.55088 7.25 2.14489 7.05798C1.7266 6.86014 1.38986 6.5234 1.19202 6.10511C1 5.69912 1 5.17441 1 4.125ZM2.5 3.3C2.5 3.01997 2.5 2.87996 2.5545 2.773C2.60243 2.67892 2.67892 2.60243 2.773 2.5545C2.87996 2.5 3.01997 2.5 3.3 2.5H4.95C5.23003 2.5 5.37004 2.5 5.477 2.5545C5.57108 2.60243 5.64757 2.67892 5.6955 2.773C5.75 2.87996 5.75 3.01997 5.75 3.3V4.95C5.75 5.23003 5.75 5.37004 5.6955 5.477C5.64757 5.57108 5.57108 5.64757 5.477 5.6955C5.37004 5.75 5.23003 5.75 4.95 5.75H3.3C3.01997 5.75 2.87996 5.75 2.773 5.6955C2.67892 5.64757 2.60243 5.57108 2.5545 5.477C2.5 5.37004 2.5 5.23003 2.5 4.95V3.3ZM1 11.875C1 10.8256 1 10.3009 1.19202 9.89489C1.38986 9.4766 1.7266 9.13986 2.14489 8.94202C2.55088 8.75 3.07559 8.75 4.125 8.75C5.17441 8.75 5.69912 8.75 6.10511 8.94202C6.5234 9.13986 6.86014 9.4766 7.05798 9.89489C7.25 10.3009 7.25 10.8256 7.25 11.875C7.25 12.9244 7.25 13.4491 7.05798 13.8551C6.86014 14.2734 6.5234 14.6101 6.10511 14.808C5.69912 15 5.17441 15 4.125 15C3.07559 15 2.55088 15 2.14489 14.808C1.7266 14.6101 1.38986 14.2734 1.19202 13.8551C1 13.4491 1 12.9244 1 11.875ZM2.5 11.05C2.5 10.77 2.5 10.63 2.5545 10.523C2.60243 10.4289 2.67892 10.3524 2.773 10.3045C2.87996 10.25 3.01997 10.25 3.3 10.25H4.95C5.23003 10.25 5.37004 10.25 5.477 10.3045C5.57108 10.3524 5.64757 10.4289 5.6955 10.523C5.75 10.63 5.75 10.77 5.75 11.05V12.7C5.75 12.98 5.75 13.12 5.6955 13.227C5.64757 13.3211 5.57108 13.3976 5.477 13.4455C5.37004 13.5 5.23003 13.5 4.95 13.5H3.3C3.01997 13.5 2.87996 13.5 2.773 13.4455C2.67892 13.3976 2.60243 13.3211 2.5545 13.227C2.5 13.12 2.5 12.98 2.5 12.7V11.05ZM8.94202 2.14489C8.75 2.55088 8.75 3.07559 8.75 4.125C8.75 5.17441 8.75 5.69912 8.94202 6.10511C9.13986 6.5234 9.4766 6.86014 9.89489 7.05798C10.3009 7.25 10.8256 7.25 11.875 7.25C12.9244 7.25 13.4491 7.25 13.8551 7.05798C14.2734 6.86014 14.6101 6.5234 14.808 6.10511C15 5.69912 15 5.17441 15 4.125C15 3.07559 15 2.55088 14.808 2.14489C14.6101 1.7266 14.2734 1.38986 13.8551 1.19202C13.4491 1 12.9244 1 11.875 1C10.8256 1 10.3009 1 9.89489 1.19202C9.4766 1.38986 9.13986 1.7266 8.94202 2.14489ZM10.3045 2.773C10.25 2.87996 10.25 3.01997 10.25 3.3V4.95C10.25 5.23003 10.25 5.37004 10.3045 5.477C10.3524 5.57108 10.4289 5.64757 10.523 5.6955C10.63 5.75 10.77 5.75 11.05 5.75H12.7C12.98 5.75 13.12 5.75 13.227 5.6955C13.3211 5.64757 13.3976 5.57108 13.4455 5.477C13.5 5.37004 13.5 5.23003 13.5 4.95V3.3C13.5 3.01997 13.5 2.87996 13.4455 2.773C13.3976 2.67892 13.3211 2.60243 13.227 2.5545C13.12 2.5 12.98 2.5 12.7 2.5H11.05C10.77 2.5 10.63 2.5 10.523 2.5545C10.4289 2.60243 10.3524 2.67892 10.3045 2.773ZM8.75 11.875C8.75 10.8256 8.75 10.3009 8.94202 9.89489C9.13986 9.4766 9.4766 9.13986 9.89489 8.94202C10.3009 8.75 10.8256 8.75 11.875 8.75C12.9244 8.75 13.4491 8.75 13.8551 8.94202C14.2734 9.13986 14.6101 9.4766 14.808 9.89489C15 10.3009 15 10.8256 15 11.875C15 12.9244 15 13.4491 14.808 13.8551C14.6101 14.2734 14.2734 14.6101 13.8551 14.808C13.4491 15 12.9244 15 11.875 15C10.8256 15 10.3009 15 9.89489 14.808C9.4766 14.6101 9.13986 14.2734 8.94202 13.8551C8.75 13.4491 8.75 12.9244 8.75 11.875ZM10.25 11.05C10.25 10.77 10.25 10.63 10.3045 10.523C10.3524 10.4289 10.4289 10.3524 10.523 10.3045C10.63 10.25 10.77 10.25 11.05 10.25H12.7C12.98 10.25 13.12 10.25 13.227 10.3045C13.3211 10.3524 13.3976 10.4289 13.4455 10.523C13.5 10.63 13.5 10.77 13.5 11.05V12.7C13.5 12.98 13.5 13.12 13.4455 13.227C13.3976 13.3211 13.3211 13.3976 13.227 13.4455C13.12 13.5 12.98 13.5 12.7 13.5H11.05C10.77 13.5 10.63 13.5 10.523 13.4455C10.4289 13.3976 10.3524 13.3211 10.3045 13.227C10.25 13.12 10.25 12.98 10.25 12.7V11.05Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const VisibleIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M5.45228 14.6892C5.30025 14.4155 5.18194 14.1813 5.09639 14.0029C5.18194 13.8246 5.30025 13.5904 5.45228 13.3167C5.81867 12.6572 6.37444 11.7805 7.13069 10.9079C8.64778 9.15737 10.8954 7.50293 14 7.50293C17.1046 7.50293 19.3522 9.15737 20.8693 10.9079C21.6256 11.7805 22.1813 12.6572 22.5477 13.3167C22.6998 13.5904 22.8181 13.8246 22.9036 14.0029C22.8181 14.1813 22.6998 14.4155 22.5477 14.6892C22.1813 15.3487 21.6256 16.2254 20.8693 17.098C19.3522 18.8485 17.1046 20.5029 14 20.5029C10.8954 20.5029 8.64778 18.8485 7.13069 17.098C6.37444 16.2254 5.81867 15.3487 5.45228 14.6892ZM24.9282 13.6307C24.9283 13.6311 24.9285 13.6315 24 14.0029C24.9285 14.3743 24.9283 14.3747 24.9282 14.3751L24.9278 14.3761L24.9267 14.3786L24.9239 14.3857L24.9148 14.4078C24.9072 14.4261 24.8966 14.4513 24.8829 14.483C24.8555 14.5465 24.8159 14.6361 24.7636 14.7481C24.6592 14.9719 24.5041 15.286 24.296 15.6604C23.8812 16.4072 23.2494 17.4055 22.3807 18.4079C20.6478 20.4074 17.8954 22.5029 14 22.5029C10.1046 22.5029 7.35222 20.4074 5.61931 18.4079C4.75056 17.4055 4.11883 16.4072 3.70397 15.6604C3.49594 15.286 3.34082 14.9719 3.2364 14.7481C3.18414 14.6361 3.14447 14.5465 3.1171 14.483C3.10342 14.4513 3.0928 14.4261 3.08521 14.4078L3.07611 14.3857L3.07325 14.3786L3.07225 14.3761L3.07185 14.3751C3.07168 14.3747 3.07152 14.3743 4 14.0029C3.07152 13.6315 3.07168 13.6311 3.07185 13.6307L3.07225 13.6297L3.07325 13.6272L3.07611 13.6202L3.08521 13.5981C3.0928 13.5798 3.10342 13.5546 3.1171 13.5228C3.14447 13.4593 3.18414 13.3697 3.2364 13.2578C3.34082 13.034 3.49594 12.7199 3.70397 12.3454C4.11883 11.5987 4.75056 10.6004 5.61931 9.598C7.35222 7.59849 10.1046 5.50293 14 5.50293C17.8954 5.50293 20.6478 7.59849 22.3807 9.598C23.2494 10.6004 23.8812 11.5987 24.296 12.3454C24.5041 12.7199 24.6592 13.034 24.7636 13.2578C24.8159 13.3697 24.8555 13.4593 24.8829 13.5228C24.8966 13.5546 24.9072 13.5798 24.9148 13.5981L24.9239 13.6202L24.9267 13.6272L24.9278 13.6297L24.9282 13.6307ZM24 14.0029L24.9285 13.6315C25.0238 13.8699 25.0238 14.1359 24.9285 14.3743L24 14.0029ZM3.07152 13.6315L4 14.0029L3.07152 14.3743C2.97616 14.1359 2.97616 13.8699 3.07152 13.6315ZM14 17.7529C16.0711 17.7529 17.75 16.074 17.75 14.0029C17.75 11.9319 16.0711 10.2529 14 10.2529C11.9289 10.2529 10.25 11.9319 10.25 14.0029C10.25 16.074 11.9289 17.7529 14 17.7529Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const InvisibleIcon = () => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M23.7071 5.70808C24.0976 5.31756 24.0976 4.68439 23.7071 4.29387C23.3165 3.90335 22.6834 3.90335 22.2929 4.29387L19.4961 7.09065C18.0032 6.15658 16.174 5.50098 14.0002 5.50098C10.1048 5.50098 7.35242 7.59653 5.61951 9.59605C4.75077 10.5984 4.11903 11.5967 3.70417 12.3435C3.49614 12.7179 3.34103 13.032 3.2366 13.2558C3.18435 13.3678 3.14467 13.4574 3.11731 13.5209C3.10362 13.5526 3.093 13.5778 3.08542 13.5961L3.07631 13.6182L3.07346 13.6253L3.07245 13.6278L3.07205 13.6288C3.07205 13.6288 3.07173 13.6296 4.0002 14.001L3.07205 13.6288C2.97669 13.8672 2.97636 14.134 3.07173 14.3724L4.0002 14.001C3.07173 14.3724 3.07173 14.3724 3.07173 14.3724L3.07215 14.3734L3.07269 14.3748L3.0741 14.3783L3.07829 14.3886L3.09211 14.4219C3.10376 14.4497 3.12029 14.4884 3.14176 14.5372C3.1847 14.6348 3.24751 14.7727 3.33075 14.9434C3.49705 15.2843 3.74593 15.7582 4.08204 16.3038C4.69847 17.3043 5.62298 18.5699 6.89004 19.6967L4.29285 22.2939C3.90233 22.6844 3.90233 23.3176 4.29285 23.7081C4.68337 24.0986 5.31654 24.0986 5.70707 23.7081L23.7071 5.70808ZM8.30712 18.2796L18.0363 8.5504C16.9005 7.91812 15.5636 7.50098 14.0002 7.50098C10.8956 7.50098 8.64798 9.15542 7.13089 10.9059C6.37464 11.7785 5.81887 12.6552 5.45248 13.3147C5.30045 13.5884 5.18214 13.8226 5.09659 14.001L5.12834 14.0666C5.27048 14.358 5.48822 14.7733 5.78482 15.2547C6.34966 16.1715 7.18495 17.303 8.30712 18.2796Z"
                fill="currentColor"
            />
            <path
                d="M21.8243 10.4336C22.2714 10.1093 22.8967 10.2088 23.221 10.6558C23.7967 11.4494 24.2191 12.1825 24.4987 12.7196C24.6388 12.9887 24.7438 13.21 24.815 13.3669C24.8506 13.4454 24.8777 13.5079 24.8966 13.5523C24.9061 13.5746 24.9135 13.5923 24.9188 13.6052L24.9253 13.6211L24.9274 13.6263L24.9282 13.6282L24.9286 13.6293C24.9286 13.6293 24.9287 13.6296 24.0003 14.001C24.9287 14.3724 24.9284 14.3732 24.9284 14.3732L24.927 14.3767L24.9241 14.3837L24.915 14.4059C24.9075 14.4241 24.8968 14.4494 24.8832 14.4811C24.8558 14.5446 24.8161 14.6342 24.7639 14.7461C24.6594 14.9699 24.5043 15.2841 24.2963 15.6585C23.8814 16.4053 23.2497 17.4035 22.3809 18.4059C20.648 20.4054 17.8956 22.501 14.0003 22.501C13.3463 22.501 12.7215 22.4416 12.1259 22.3321C11.5827 22.2323 11.2232 21.7111 11.3231 21.1679C11.4229 20.6247 11.9442 20.2653 12.4873 20.3651C12.9642 20.4527 13.4681 20.501 14.0003 20.501C17.1049 20.501 19.3525 18.8466 20.8696 17.0961C21.6258 16.2235 22.1816 15.3467 22.548 14.6872C22.7 14.4136 22.8183 14.1793 22.9039 14.001C22.855 13.8991 22.7953 13.7788 22.7248 13.6433C22.4806 13.1744 22.1086 12.5285 21.6021 11.8303C21.2778 11.3832 21.3773 10.7579 21.8243 10.4336Z"
                fill="currentColor"
            />
            <path
                d="M24.0003 14.001L24.9286 13.6293C25.024 13.8677 25.0238 14.1348 24.9284 14.3732L24.0003 14.001Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PinIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="28"
            height="28"
            viewBox="0 0 28 28"
            fill="none"
            className={className}
        >
            <g clipPath="url(#clip0_35112_65606)">
                <path
                    d="M16.8891 6.87327C18.3033 5.45905 17.2426 4.39839 17.9497 3.69129C18.4801 3.16096 19.7752 3.39546 22.1924 5.81261C24.6095 8.22975 24.844 9.52492 24.3137 10.0552C23.6066 10.7624 22.5459 9.70169 21.1317 11.1159L19.0104 13.2372C16.5355 15.7121 20.0711 19.2476 17.9497 21.369C16.8891 22.4296 14.4142 20.6618 11.5891 17.8368L8.76052 20.665C6.99295 22.433 4.51786 24.9076 3.81077 24.2005C3.10368 23.4935 5.57856 21.0186 7.34635 19.2508L10.1749 16.4226C7.34314 13.5908 5.57538 11.1159 6.63604 10.0552C8.75736 7.93393 12.2929 11.4695 14.7678 8.99459L16.8891 6.87327Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_35112_65606">
                    <rect width="28" height="28" fill="white" transform="translate(0 0.00488281)" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const ExportIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                d="M5.71858 1C5.04521 0.999992 4.49449 0.999985 4.04693 1.03655C3.58367 1.0744 3.16527 1.15514 2.77453 1.35423C2.163 1.66582 1.66582 2.163 1.35423 2.77453C1.15514 3.16527 1.0744 3.58367 1.03655 4.04693C0.999985 4.49449 0.999992 5.0452 1 5.71857V10.2814C0.999992 10.9548 0.999985 11.5055 1.03655 11.9531C1.0744 12.4163 1.15514 12.8347 1.35423 13.2255C1.66582 13.837 2.163 14.3342 2.77453 14.6458C3.16527 14.8449 3.58367 14.9256 4.04693 14.9634C4.49448 15 5.04518 15 5.71853 15H10.2814C10.9548 15 11.5055 15 11.9531 14.9634C12.4163 14.9256 12.8347 14.8449 13.2255 14.6458C13.837 14.3342 14.3342 13.837 14.6458 13.2255C14.8449 12.8347 14.9256 12.4163 14.9634 11.9531C15 11.5055 15 10.9548 15 10.2815V9.5C15 9.08579 14.6642 8.75 14.25 8.75C13.8358 8.75 13.5 9.08579 13.5 9.5V10.25C13.5 10.9624 13.4994 11.4517 13.4684 11.8309C13.4382 12.2014 13.3827 12.4004 13.3093 12.5445C13.1415 12.8738 12.8738 13.1415 12.5445 13.3093C12.4004 13.3827 12.2014 13.4382 11.8309 13.4684C11.4517 13.4994 10.9624 13.5 10.25 13.5H5.75C5.03756 13.5 4.54833 13.4994 4.16908 13.4684C3.79858 13.4382 3.59956 13.3827 3.45552 13.3093C3.12623 13.1415 2.85852 12.8738 2.69074 12.5445C2.61735 12.4004 2.56184 12.2014 2.53157 11.8309C2.50058 11.4517 2.5 10.9624 2.5 10.25V5.75C2.5 5.03756 2.50058 4.54833 2.53157 4.16908C2.56184 3.79858 2.61735 3.59956 2.69074 3.45552C2.85852 3.12623 3.12623 2.85852 3.45552 2.69074C3.59956 2.61735 3.79858 2.56184 4.16908 2.53157C4.54833 2.50058 5.03756 2.5 5.75 2.5H6.5C6.91421 2.5 7.25 2.16421 7.25 1.75C7.25 1.33579 6.91421 1 6.5 1H5.71858Z"
                fill="currentColor"
            />
            <path
                d="M9.5 1C9.08579 1 8.75 1.33579 8.75 1.75C8.75 2.16421 9.08579 2.5 9.5 2.5H12.4393L7.46967 7.46967C7.17678 7.76256 7.17678 8.23744 7.46967 8.53033C7.76256 8.82322 8.23744 8.82322 8.53033 8.53033L13.5 3.56066V6.5C13.5 6.91421 13.8358 7.25 14.25 7.25C14.6642 7.25 15 6.91421 15 6.5V1.75C15 1.33579 14.6642 1 14.25 1H9.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const EyeDisableIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M2.78033 0.71967C2.48744 0.426777 2.01256 0.426777 1.71967 0.71967C1.42678 1.01256 1.42678 1.48744 1.71967 1.78033L3.39477 3.45543C2.62494 3.99287 1.98721 4.63453 1.50894 5.30146C0.893591 6.15954 0.5 7.1207 0.5 8.00001C0.5 9.2926 1.34793 10.7923 2.62326 11.9338C3.92842 13.1021 5.78895 14 8 14C9.77112 14 11.3179 13.4236 12.5335 12.5941L14.2197 14.2803C14.5126 14.5732 14.9874 14.5732 15.2803 14.2803C15.5732 13.9874 15.5732 13.5126 15.2803 13.2197L2.78033 0.71967ZM5.84863 5.90929L4.47589 4.53655C3.7448 5.00472 3.15157 5.58482 2.7279 6.1756C2.22264 6.88017 2 7.53943 2 8.00001C2 8.70742 2.52504 9.83276 3.62369 10.8162C4.69251 11.7729 6.20698 12.5 8 12.5C9.32442 12.5 10.4964 12.1036 11.4498 11.5104L10.0907 10.1514C9.55107 10.6758 8.81331 11 8 11C6.34315 11 5 9.65686 5 8.00001C5 7.18653 5.3241 6.44905 5.84863 5.90929ZM6.90938 6.97004C6.65496 7.23924 6.5 7.60167 6.5 8.00001C6.5 8.82844 7.17157 9.50001 8 9.50001C8.3985 9.50001 8.76068 9.34514 9.02994 9.0906L6.90938 6.97004Z"
                fill="currentColor"
            />
            <path
                d="M8.00004 3.5C7.89358 3.5 7.78818 3.50256 7.68388 3.50756C7.27014 3.5274 6.91865 3.20809 6.89881 2.79435C6.87897 2.38061 7.19828 2.02912 7.61202 2.00928C7.7402 2.00313 7.86955 2 8.00004 2C10.2111 2 12.0716 2.8979 13.3768 4.06618C14.6521 5.20775 15.5 6.70741 15.5 8C15.5 8.59014 15.3224 9.21098 15.0319 9.8016C14.849 10.1733 14.3995 10.3264 14.0278 10.1435C13.6561 9.96067 13.5031 9.51114 13.6859 9.13947C13.906 8.69207 14 8.29955 14 8C14 7.29259 13.475 6.16725 12.3764 5.18383C11.3075 4.2271 9.79306 3.5 8.00004 3.5Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const BlockIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M15 8.00098C15 11.867 11.866 15.001 8 15.001C4.13401 15.001 1 11.867 1 8.00098C1 4.13498 4.13401 1.00098 8 1.00098C11.866 1.00098 15 4.13498 15 8.00098ZM11.3226 12.3843C10.3996 13.0851 9.24835 13.501 8 13.501C4.96243 13.501 2.5 11.0385 2.5 8.00098C2.5 6.75262 2.9159 5.60141 3.6167 4.67833L11.3226 12.3843ZM12.3833 11.3236L4.67736 3.61767C5.60043 2.91688 6.75165 2.50098 8 2.50098C11.0376 2.50098 13.5 4.96341 13.5 8.00098C13.5 9.24933 13.0841 10.4005 12.3833 11.3236Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const GearIconEmpty: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.99998 5.0001C6.34313 5.0001 4.99998 6.34325 4.99998 8.0001C4.99998 9.65696 6.34313 11.0001 7.99998 11.0001C9.65684 11.0001 11 9.65696 11 8.0001C11 6.34325 9.65684 5.0001 7.99998 5.0001ZM6.49998 8.0001C6.49998 7.17168 7.17156 6.5001 7.99998 6.5001C8.82841 6.5001 9.49998 7.17168 9.49998 8.0001C9.49998 8.82853 8.82841 9.50011 7.99998 9.50011C7.17156 9.50011 6.49998 8.82853 6.49998 8.0001Z"
                fill="currentColor"
            />
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M6.12282 1.00005C5.54094 0.999593 5.08378 0.999233 4.66215 1.13806C4.29149 1.26011 3.95104 1.45965 3.66345 1.72343C3.33631 2.02346 3.11325 2.42252 2.82934 2.93043L1.02317 6.15572C0.750371 6.642 0.535962 7.02419 0.449345 7.44246C0.373167 7.81032 0.373167 8.18989 0.449345 8.55775C0.535962 8.97602 0.750372 9.35822 1.02317 9.84449L2.82934 13.0698C3.11325 13.5777 3.33631 13.9767 3.66345 14.2768C3.95104 14.5406 4.29149 14.7401 4.66215 14.8621C5.08378 15.001 5.54094 15.0006 6.12282 15.0002H9.87714C10.459 15.0006 10.9162 15.001 11.3378 14.8621C11.7085 14.7401 12.0489 14.5406 12.3365 14.2768C12.6637 13.9767 12.8867 13.5777 13.1706 13.0698L14.9768 9.84447C15.2496 9.3582 15.464 8.97601 15.5506 8.55774C15.6268 8.18989 15.6268 7.81032 15.5506 7.44247C15.464 7.0242 15.2496 6.64201 14.9768 6.15574L13.1706 2.93045C12.8867 2.42253 12.6637 2.02347 12.3365 1.72343C12.0489 1.45966 11.7085 1.26011 11.3378 1.13806C10.9162 0.999233 10.459 0.999593 9.87714 1.00005H6.12282ZM5.13129 2.56281C5.29457 2.50905 5.49141 2.50011 6.22557 2.50011H9.77438C10.5086 2.50011 10.7054 2.50905 10.8687 2.56281C11.0372 2.61829 11.1919 2.70899 11.3226 2.82889C11.4493 2.94509 11.5533 3.11246 11.912 3.75303L13.62 6.80304C13.9634 7.41622 14.0482 7.58455 14.0817 7.74663C14.1164 7.91384 14.1164 8.08637 14.0817 8.25358C14.0482 8.41566 13.9634 8.58399 13.62 9.19718L11.912 12.2472C11.5533 12.8877 11.4493 13.0551 11.3226 13.1713C11.1919 13.2912 11.0372 13.3819 10.8687 13.4374C10.7054 13.4912 10.5086 13.5001 9.77438 13.5001H6.22557C5.4914 13.5001 5.29457 13.4912 5.13129 13.4374C4.9628 13.3819 4.80805 13.2912 4.67733 13.1713C4.55064 13.0551 4.44666 12.8878 4.08794 12.2472L2.37993 9.19719C2.03654 8.58399 1.95175 8.41567 1.91818 8.25358C1.88356 8.08637 1.88356 7.91384 1.91818 7.74663C1.95175 7.58454 2.03654 7.41622 2.37993 6.80302L4.08794 3.75302C4.44666 3.11245 4.55064 2.94508 4.67733 2.82889C4.80805 2.70899 4.9628 2.61829 5.13129 2.56281Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PencilIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <g clipPath="url(#clip0_39971_30802)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M1.06976 10.1295L9.7 1.49927C10.454 0.745269 10.831 0.36827 11.2529 0.196822C11.796 -0.0239409 12.404 -0.0239407 12.9471 0.196823C13.369 0.368271 13.746 0.745269 14.5 1.49927C15.254 2.25327 15.631 2.63026 15.8024 3.05212C16.0232 3.59531 16.0232 4.20322 15.8024 4.74642C15.631 5.16827 15.254 5.54527 14.5 6.29927L5.86975 14.9295C5.53778 15.2615 5.3718 15.4275 5.1851 15.5564C4.89371 15.7576 4.56288 15.8947 4.21454 15.9584C3.99135 15.9993 3.75662 15.9993 3.28714 15.9993C2.32119 15.9993 1.83821 15.9993 1.45311 15.8534C0.850189 15.6251 0.374185 15.1491 0.145845 14.5462C0 14.1611 0 13.6781 0 12.7121C0 12.2426 0 12.0079 0.0408471 11.7847C0.104601 11.4364 0.241632 11.1055 0.442867 10.8141C0.571799 10.6274 0.737787 10.4615 1.06976 10.1295ZM1.52763 11.98C1.5 12.0951 1.5 12.2174 1.5 12.462V13.2992C1.5 13.7193 1.5 13.9293 1.58175 14.0897C1.65365 14.2308 1.76839 14.3456 1.90951 14.4175C2.06994 14.4992 2.27996 14.4992 2.7 14.4992H3.53726C3.78185 14.4992 3.90415 14.4992 4.01923 14.4716C4.12127 14.4471 4.21881 14.4067 4.30829 14.3519C4.4092 14.29 4.49568 14.2035 4.66863 14.0306L12.0697 6.62959L9.36967 3.92959L1.96863 11.3306C1.79569 11.5036 1.7092 11.59 1.64736 11.691C1.59253 11.7804 1.55213 11.878 1.52763 11.98ZM13.9515 4.74778L13.1303 5.56893L10.4303 2.86894L11.2515 2.04779C11.5485 1.75078 11.697 1.60227 11.8682 1.54663C12.0189 1.49769 12.1811 1.49769 12.3318 1.54663C12.503 1.60227 12.6515 1.75078 12.9485 2.04779L13.9515 3.05072C14.2485 3.34772 14.397 3.49625 14.4526 3.66749C14.5016 3.81812 14.5016 3.98038 14.4526 4.13102C14.397 4.30226 14.2485 4.45077 13.9515 4.74778Z"
                    fill="currentColor"
                />
                <path
                    d="M9.25 14.4993C8.83579 14.4993 8.5 14.8351 8.5 15.2493C8.5 15.6635 8.83579 15.9993 9.25 15.9993H15.25C15.6642 15.9993 16 15.6635 16 15.2493C16 14.8351 15.6642 14.4993 15.25 14.4993H9.25Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_39971_30802">
                    <rect width="16" height="16" fill="white" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const TextIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M1 3.75C1 3.33579 1.33579 3 1.75 3H14.25C14.6642 3 15 3.33579 15 3.75C15 4.16421 14.6642 4.5 14.25 4.5H1.75C1.33579 4.5 1 4.16421 1 3.75ZM1 8.25C1 7.83579 1.33579 7.5 1.75 7.5H14.25C14.6642 7.5 15 7.83579 15 8.25C15 8.66421 14.6642 9 14.25 9H1.75C1.33579 9 1 8.66421 1 8.25ZM1.75 12C1.33579 12 1 12.3358 1 12.75C1 13.1642 1.33579 13.5 1.75 13.5H10.25C10.6642 13.5 11 13.1642 11 12.75C11 12.3358 10.6642 12 10.25 12H1.75Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const ListIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M3 3.75C3 4.30228 2.55228 4.75 2 4.75C1.44772 4.75 1 4.30228 1 3.75C1 3.19772 1.44772 2.75 2 2.75C2.55228 2.75 3 3.19772 3 3.75ZM5.25 3C4.83579 3 4.5 3.33579 4.5 3.75C4.5 4.16421 4.83579 4.5 5.25 4.5H14.25C14.6642 4.5 15 4.16421 15 3.75C15 3.33579 14.6642 3 14.25 3H5.25ZM5.25 7.25C4.83579 7.25 4.5 7.58579 4.5 8C4.5 8.41421 4.83579 8.75 5.25 8.75H14.25C14.6642 8.75 15 8.41421 15 8C15 7.58579 14.6642 7.25 14.25 7.25H5.25ZM4.5 12.25C4.5 11.8358 4.83579 11.5 5.25 11.5H14.25C14.6642 11.5 15 11.8358 15 12.25C15 12.6642 14.6642 13 14.25 13H5.25C4.83579 13 4.5 12.6642 4.5 12.25ZM2 9C2.55228 9 3 8.55228 3 8C3 7.44772 2.55228 7 2 7C1.44772 7 1 7.44772 1 8C1 8.55228 1.44772 9 2 9ZM2 13.25C2.55228 13.25 3 12.8023 3 12.25C3 11.6977 2.55228 11.25 2 11.25C1.44772 11.25 1 11.6977 1 12.25C1 12.8023 1.44772 13.25 2 13.25Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const PinIconOutline: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <g clipPath="url(#clip0_41815_30580)">
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M11.199 1.13852C9.87508 -0.18542 7.60764 0.592557 7.3754 2.45043L7.28773 3.15181L5.73177 4.70778L3.72315 4.43996C1.35341 4.124 -0.0655134 6.99531 1.62498 8.6858L3.93962 11.0004L0.219708 14.7198C-0.0732061 15.0127 -0.0732405 15.4876 0.219631 15.7805C0.512503 16.0734 0.987377 16.0734 1.28029 15.7806L5.00028 12.0611L7.3144 14.3752C9.00489 16.0657 11.8762 14.6468 11.5602 12.2771L11.2924 10.2684L12.8484 8.71247L13.5498 8.6248C15.4076 8.39257 16.1856 6.12513 14.8617 4.80119L11.199 1.13852ZM8.86382 2.63648C8.94123 2.01719 9.69705 1.75787 10.1384 2.19918L13.801 5.86185C14.2423 6.30316 13.983 7.05897 13.3637 7.13638L12.407 7.25597C12.2415 7.27666 12.0876 7.35192 11.9697 7.46985L9.96969 9.46985C9.80451 9.63503 9.72573 9.86775 9.7566 10.0993L10.0734 12.4753C10.1998 13.4232 9.05126 13.9908 8.37506 13.3146L2.68564 7.62514C2.00944 6.94894 2.57701 5.80042 3.52491 5.9268L5.9009 6.2436C6.13245 6.27448 6.36517 6.19569 6.53035 6.03051L8.53035 4.03051C8.64828 3.91258 8.72354 3.7587 8.74423 3.59321L8.86382 2.63648Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_41815_30580">
                    <rect
                        width="16"
                        height="16"
                        fill="white"
                        transform="translate(0 0.000244141)"
                    />
                </clipPath>
            </defs>
        </svg>
    );
};

export const UnpinIconOutline: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <g clipPath="url(#clip0_41815_29055)">
                <path
                    d="M11.199 1.13828C9.87506 -0.185662 7.60762 0.592309 7.37538 2.45019L7.28235 3.19441C7.23098 3.60543 7.52252 3.98027 7.93354 4.03165C8.34455 4.08302 8.71939 3.79148 8.77077 3.38046L8.8638 2.63624C8.94121 2.01695 9.69702 1.75762 10.1383 2.19894L13.801 5.8616C14.2423 6.30292 13.983 7.05873 13.3637 7.13614L12.6204 7.22905C12.2094 7.28042 11.9179 7.65527 11.9693 8.06628C12.0206 8.4773 12.3955 8.76884 12.8065 8.71746L13.5498 8.62456C15.4076 8.39232 16.1856 6.12489 14.8617 4.80094L11.199 1.13828Z"
                    fill="currentColor"
                />
                <path
                    fillRule="evenodd"
                    clipRule="evenodd"
                    d="M11.0372 9.47645C11.0329 9.4721 11.0286 9.4678 11.0243 9.46356L6.53638 4.97566C6.53215 4.97132 6.52786 4.96704 6.52352 4.9628L2.78033 1.21961C2.48744 0.926715 2.01256 0.926715 1.71967 1.21961C1.42678 1.5125 1.42678 1.98738 1.71967 2.28027L3.90311 4.46371L3.72315 4.43972C1.35341 4.12375 -0.0655134 6.99506 1.62498 8.68555L3.93962 11.0002L0.219708 14.7196C-0.0732061 15.0124 -0.0732405 15.4873 0.219631 15.7802C0.512503 16.0731 0.987377 16.0732 1.28029 15.7803L5.00028 12.0609L7.3144 14.375C9.00489 16.0655 11.8762 14.6466 11.5602 12.2768L11.5362 12.0968L13.7197 14.2803C14.0126 14.5732 14.4874 14.5732 14.7803 14.2803C15.0732 13.9874 15.0732 13.5125 14.7803 13.2196L11.0372 9.47645ZM2.68564 7.62489C2.00944 6.9487 2.57701 5.80017 3.52491 5.92656L5.6492 6.2098L9.79016 10.3508L10.0734 12.4751C10.1998 13.4229 9.05126 13.9905 8.37506 13.3143L2.68564 7.62489Z"
                    fill="currentColor"
                />
            </g>
            <defs>
                <clipPath id="clip0_41815_29055">
                    <rect width="16" height="16" fill="white" />
                </clipPath>
            </defs>
        </svg>
    );
};

export const InboxIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M5.11124 0.999944L5.00991 0.999848C4.52949 0.99917 4.1074 0.998574 3.72549 1.14004C3.39128 1.26384 3.09125 1.46526 2.85013 1.72772C2.57459 2.02763 2.41533 2.41851 2.23405 2.86343L2.23405 2.86343L2.19577 2.95725L0.55584 6.96597C0.518967 7.0561 0.5 7.15256 0.5 7.24994V9.44994V9.48205V9.48206C0.499993 10.2949 0.499987 10.9505 0.543364 11.4814C0.588026 12.028 0.682386 12.5082 0.908726 12.9524C1.26825 13.658 1.84193 14.2317 2.54754 14.5912C2.99175 14.8176 3.4719 14.9119 4.01853 14.9566C4.54944 15 5.20505 15 6.01788 14.9999H6.05H9.95H9.98212C10.795 15 11.4506 15 11.9815 14.9566C12.5281 14.9119 13.0082 14.8176 13.4525 14.5912C14.1581 14.2317 14.7317 13.658 15.0913 12.9524C15.3176 12.5082 15.412 12.028 15.4566 11.4814C15.5 10.9505 15.5 10.2949 15.5 9.48207V9.44994V7.24994C15.5 7.15256 15.481 7.0561 15.4442 6.96597L13.8042 2.95725L13.766 2.86343C13.5847 2.41852 13.4254 2.02763 13.1499 1.72772C12.9087 1.46526 12.6087 1.26384 12.2745 1.14004C11.8926 0.998574 11.4705 0.99917 10.9901 0.999848L10.8888 0.999944H5.11124ZM4.24652 2.54664C4.34226 2.51117 4.46982 2.49994 5.11124 2.49994H10.8888C11.5302 2.49994 11.6577 2.51117 11.7535 2.54664C11.8649 2.58791 11.9649 2.65505 12.0453 2.74253C12.1144 2.81773 12.173 2.93154 12.4159 3.5252L13.6328 6.49994H10.75C10.3617 6.49994 10.0423 6.79507 10.0039 7.17326C10.0013 7.19847 10 7.22406 10 7.24994C10 7.24996 10 7.24998 10 7.25C10 8.35457 9.10457 9.25 8 9.25C6.89566 9.25 6.00037 8.35494 6 7.25069C6 7.25044 6 7.25019 6 7.24994C6 6.83573 5.66421 6.49994 5.25 6.49994H2.36715L3.58409 3.5252C3.82695 2.93154 3.88565 2.81773 3.95473 2.74253C4.0351 2.65505 4.13511 2.58791 4.24652 2.54664ZM4.58053 7.99994H2V9.44994C2 10.3024 2.00058 10.8966 2.03838 11.3593C2.07547 11.8131 2.1446 12.0739 2.24524 12.2714C2.46095 12.6948 2.80516 13.039 3.22852 13.2547C3.42604 13.3553 3.68681 13.4245 4.14068 13.4616C4.60331 13.4994 5.19755 13.4999 6.05 13.4999H9.95C10.8025 13.4999 11.3967 13.4994 11.8593 13.4616C12.3132 13.4245 12.574 13.3553 12.7715 13.2547C13.1948 13.039 13.539 12.6948 13.7548 12.2714C13.8554 12.0739 13.9245 11.8131 13.9616 11.3593C13.9994 10.8966 14 10.3024 14 9.44994V7.99994H11.4195C11.0762 9.57256 9.67558 10.75 8 10.75C6.32442 10.75 4.92385 9.57256 4.58053 7.99994Z"
                fill="#8D8D93"
            />
        </svg>
    );
};

export const FolderIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                opacity="0.48"
                fillRule="evenodd"
                clipRule="evenodd"
                d="M7.09168 1.55491C6.92289 1.50841 6.75005 1.50934 6.6452 1.5099L6.6155 1.51001H2.95L2.92608 1.51001C2.73679 1.50999 2.55245 1.50997 2.39633 1.52272C2.22489 1.53673 2.02258 1.56978 1.81902 1.6735C1.53677 1.81731 1.3073 2.04678 1.16349 2.32903C1.05977 2.53259 1.02672 2.7349 1.01271 2.90634C0.999956 3.06246 0.999978 3.2468 1 3.43609V3.46001V11.36V11.3889C0.999989 11.7843 0.999977 12.1254 1.02293 12.4063C1.04716 12.7028 1.10064 12.9977 1.24524 13.2815C1.46095 13.7049 1.80516 14.0491 2.22852 14.2648C2.51231 14.4094 2.80722 14.4629 3.10373 14.4871C3.38466 14.51 3.72572 14.51 4.12108 14.51H4.15H11.85H11.8789C12.2743 14.51 12.6153 14.51 12.8963 14.4871C13.1928 14.4629 13.4877 14.4094 13.7715 14.2648C14.1948 14.0491 14.5391 13.7049 14.7548 13.2815C14.8994 12.9977 14.9529 12.7028 14.9771 12.4063C15 12.1254 15 11.7843 15 11.3889V11.36V7.16001V7.13109C15 6.73573 15 6.39467 14.9771 6.11374C14.9529 5.81723 14.8994 5.52232 14.7548 5.23853C14.5391 4.81517 14.1948 4.47096 13.7715 4.25525C13.4877 4.11065 13.1928 4.05717 12.8963 4.03294C12.6153 4.00999 12.2743 4.01 11.8789 4.01001H11.8789L11.85 4.01001H9.3845L9.36047 4.01001L9.34546 3.99125L7.82585 2.09174L7.80739 2.06848C7.74232 1.98625 7.63508 1.8507 7.49333 1.74795C7.37209 1.66006 7.23604 1.59468 7.09168 1.55491Z"
                fill="currentColor"
            />
            <path
                d="M1 6.00977C1 5.3097 1 4.95967 1.13624 4.69228C1.25608 4.45707 1.44731 4.26585 1.68251 4.14601C1.9499 4.00977 2.29993 4.00977 3 4.00977H11.8C12.9201 4.00977 13.4802 4.00977 13.908 4.22775C14.2843 4.4195 14.5903 4.72546 14.782 5.10178C15 5.52961 15 6.08966 15 7.20977V11.3098C15 12.4299 15 12.9899 14.782 13.4177C14.5903 13.7941 14.2843 14.1 13.908 14.2918C13.4802 14.5098 12.9201 14.5098 11.8 14.5098H4.2C3.0799 14.5098 2.51984 14.5098 2.09202 14.2918C1.71569 14.1 1.40973 13.7941 1.21799 13.4177C1 12.9899 1 12.4299 1 11.3098V6.00977Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const CoinsHorizontalIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="24"
            height="24"
            viewBox="0 0 24 24"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M12 16.899C10.7295 18.1955 8.95869 19 7 19C3.13401 19 0 15.866 0 12C0 8.13401 3.13401 5 7 5C8.95869 5 10.7295 5.80447 12 7.10102C13.2705 5.80447 15.0413 5 17 5C20.866 5 24 8.13401 24 12C24 15.866 20.866 19 17 19C15.0413 19 13.2705 18.1955 12 16.899ZM11.0619 15.7083C10.0561 16.8093 8.60872 17.5 7 17.5C3.96243 17.5 1.5 15.0376 1.5 12C1.5 8.96243 3.96243 6.5 7 6.5C8.60872 6.5 10.0561 7.19068 11.0619 8.29168C10.389 9.36692 10 10.638 10 12C10 13.362 10.389 14.6331 11.0619 15.7083ZM22.5 12C22.5 15.0376 20.0376 17.5 17 17.5C13.9624 17.5 11.5 15.0376 11.5 12C11.5 8.96243 13.9624 6.5 17 6.5C20.0376 6.5 22.5 8.96243 22.5 12Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const TrashBinIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M5.01027 3C5.13731 1.46012 6.42733 0.25 8 0.25C9.57267 0.25 10.8627 1.46012 10.9897 3H15C15.4142 3 15.75 3.33579 15.75 3.75C15.75 4.16421 15.4142 4.5 15 4.5H14V10.2814C14 10.9548 14 11.5055 13.9634 11.9531C13.9256 12.4163 13.8449 12.8347 13.6458 13.2255C13.3342 13.837 12.837 14.3342 12.2255 14.6458C11.8347 14.8449 11.4163 14.9256 10.9531 14.9634C10.5055 15 9.95483 15 9.28147 15H6.71856C6.04521 15 5.49448 15 5.04693 14.9634C4.58367 14.9256 4.16527 14.8449 3.77453 14.6458C3.163 14.3342 2.66582 13.837 2.35423 13.2255C2.15514 12.8347 2.0744 12.4163 2.03655 11.9531C1.99998 11.5055 1.99999 10.9548 2 10.2814L2 4.5H1C0.585786 4.5 0.25 4.16421 0.25 3.75C0.25 3.33579 0.585786 3 1 3H5.01027ZM6.52074 3C6.63976 2.29057 7.25675 1.75 8 1.75C8.74325 1.75 9.36024 2.29057 9.47926 3H6.52074ZM3.5 10.25V4.5H12.5V10.25C12.5 10.9624 12.4994 11.4517 12.4684 11.8309C12.4382 12.2014 12.3827 12.4004 12.3093 12.5445C12.1415 12.8738 11.8738 13.1415 11.5445 13.3093C11.4004 13.3827 11.2014 13.4382 10.8309 13.4684C10.4517 13.4994 9.96244 13.5 9.25 13.5H6.75C6.03756 13.5 5.54833 13.4994 5.16908 13.4684C4.79858 13.4382 4.59956 13.3827 4.45552 13.3093C4.12623 13.1415 3.85852 12.8738 3.69074 12.5445C3.61735 12.4004 3.56184 12.2014 3.53157 11.8309C3.50058 11.4517 3.5 10.9624 3.5 10.25Z"
                fill="currentColor"
            />
        </svg>
    );
};

export const BatteryIcon: FC<{ className?: string }> = ({ className }) => {
    return (
        <svg
            xmlns="http://www.w3.org/2000/svg"
            width="16"
            height="16"
            viewBox="0 0 16 16"
            fill="none"
            className={className}
        >
            <path
                fillRule="evenodd"
                clipRule="evenodd"
                d="M6.25 2.5H5.85C5.28995 2.5 5.00992 2.5 4.79601 2.60899C4.60785 2.70487 4.45487 2.85785 4.35899 3.04601C4.25 3.25992 4.25 3.53995 4.25 4.1V7H11.75V4.1C11.75 3.53995 11.75 3.25992 11.641 3.04601C11.5451 2.85785 11.3922 2.70487 11.204 2.60899C10.9901 2.5 10.7101 2.5 10.15 2.5H9.75H6.25ZM9.58166 1.00015C10.7642 1.00184 11.3942 1.02242 11.885 1.27248C12.3554 1.51217 12.7378 1.89462 12.9775 2.36502C13.25 2.8998 13.25 3.59987 13.25 5V12C13.25 13.4001 13.25 14.1002 12.9775 14.635C12.7378 15.1054 12.3554 15.4878 11.885 15.7275C11.3502 16 10.6501 16 9.25 16H6.75C5.34987 16 4.6498 16 4.11502 15.7275C3.64462 15.4878 3.26217 15.1054 3.02248 14.635C2.75 14.1002 2.75 13.4001 2.75 12V5C2.75 3.59987 2.75 2.8998 3.02248 2.36502C3.26217 1.89462 3.64462 1.51217 4.11502 1.27248C4.6058 1.02242 5.23579 1.00184 6.41834 1.00015C6.69917 0.408833 7.30184 0 8 0C8.69816 0 9.30083 0.408833 9.58166 1.00015Z"
                fill="currentColor"
            />
        </svg>
    );
};
