/**
 * Generic TigrisError
 */
export class TigrisError extends Error {
	readonly errMsg = this.name + ": " + this.message;

	constructor(message: string) {
		super(message);
	}

	override get name(): string {
		return "TigrisError";
	}
}

/**
 * An error thrown when the user attempts to consume a cursor that has already been
 * used
 *
 * @public
 */
export class CursorInUseError extends TigrisError {
	constructor(message = "Cursor is already in use or used. Please reset()") {
		super(message);
	}
}

export class ReflectionNotEnabled extends TigrisError {
	constructor(object: Object, propertyName: string) {
		super(
			`Cannot infer property "type" for ${object.constructor.name}#${propertyName} using Reflection.
			Ensure that "experimentalDecorators" and "emitDecoratorMetadata" options are set to true in
			"tsconfig.json" and "reflect-metadata" npm package is added to dependencies in "package.json".
			Alternatively, specify the property's "field type" manually.`
		);
	}

	override get name(): string {
		return "ReflectionNotEnabled";
	}
}

export class MissingArgumentError extends TigrisError {
	constructor(propertyName: string) {
		super(`'${propertyName}' is required and cannot be 'undefined'`);
	}

	override get name(): string {
		return "MissingArgumentError";
	}
}

export class CannotInferFieldTypeError extends TigrisError {
	constructor(object: Object, propertyName: string) {
		super(`Field type for '${object.constructor.name}#${propertyName}' cannot be determined`);
	}

	override get name(): string {
		return "CannotInferFieldTypeError";
	}
}

export class IncompleteArrayTypeDefError extends TigrisError {
	constructor(object: Object, propertyName: string) {
		super(
			`Missing "EmbeddedFieldOptions". Array's item type for '${object.constructor.name}#${propertyName}' cannot be determined`
		);
	}
	override get name(): string {
		return "IncompleteArrayTypeDefError";
	}
}

export class IncorrectVectorDefError extends TigrisError {
	constructor(object: Object, propertyName: string) {
		super(`'${propertyName}' in '${object.constructor.name}' defines "dimensions" field option identifying it as a Vector data type.
		The primitive data type for Vector can only be a 'number[]'`);
	}
	override get name(): string {
		return "IncorrectVectorDefError";
	}
}
export class MissingPrimaryKeyOrderInSchemaDefinitionError extends TigrisError {
	constructor(propertyName: string) {
		super(`Missing 'order' value for '${propertyName}' primary key`);
	}

	override get name(): string {
		return "MissingPrimaryKeyOrderInSchemaDefinitionError";
	}
}

export class DuplicatePrimaryKeyOrderError extends TigrisError {
	constructor(order: string, propertyName: string) {
		super(`Primary Key order '${order}' already exists for '${propertyName}'`);
	}

	override get name(): string {
		return "DuplicatePrimaryKeyOrderError";
	}
}

export class IncompletePrimaryKeyOrderError extends TigrisError {
	constructor(name: string, collectionName: string) {
		super(
			`Missing 'order' value in "PrimaryKeyOptions" for variable '${name}' in ${collectionName} collection`
		);
	}

	override get name(): string {
		return "IncompletePrimaryKeyOrderError";
	}
}

export class CollectionNotFoundError extends TigrisError {
	constructor(name: string) {
		super(`Collection not found : '${name}'`);
	}

	override get name(): string {
		return "CollectionNotFoundError";
	}
}

export class BranchNameRequiredError extends TigrisError {
	constructor() {
		super(`Database branch name is required. Include a branch name in client config or specify one in
		environment file as 'TIGRIS_DB_BRANCH=your_branch_name'`);
	}

	override get name(): string {
		return "BranchNameRequiredError";
	}
}
