<?php

declare(strict_types=1);

namespace Tempest\View\Renderers;

use Stringable;
use Tempest\Support\HtmlString;
use Tempest\View\Exceptions\ViewCompilationError;
use Tempest\View\GenericView;
use Tempest\View\View;
use Tempest\View\ViewCache;
use Tempest\View\ViewRenderer;
use Throwable;
use function Tempest\Support\arr;
use function Tempest\Support\str;

final class TempestViewRenderer implements ViewRenderer
{
    private ?View $currentView = null;

    public function __construct(
        private readonly TempestViewCompiler $compiler,
        private readonly ViewCache $viewCache,
    ) {
    }

    public function __get(string $name): mixed
    {
        return $this->currentView?->get($name);
    }

    public function __call(string $name, array $arguments)
    {
        return $this->currentView?->{$name}(...$arguments);
    }

    public function render(string|View $view): string
    {
        $view = is_string($view) ? new GenericView($view) : $view;

        $path = $this->viewCache->getCachedViewPath(
            path: $view->getPath(),
            compiledView: fn () => $this->cleanupCompiled($this->compiler->compile($view->getPath())),
        );

        return $this->renderCompiled($view, $path);
    }

    private function cleanupCompiled(string $compiled): string
    {
        // Remove strict type declarations
        $compiled = str($compiled)->replace('declare(strict_types=1);', '');

        // Cleanup and bundle imports
        $imports = arr();
        $compiled = $compiled
            ->replaceRegex('/use .*;/', function (array $matches) use (&$imports) {
                $imports[$matches[0]] = $matches[0];

                return '';
            })
            ->prepend(
                sprintf(
                    '<?php
%s
?>',
                    $imports->implode(PHP_EOL),
                ),
            );

        // Remove empty PHP blocks
        $compiled = $compiled->replaceRegex('/<\?php\s*\?>/', '');

        return $compiled->toString();
    }

    private function renderCompiled(View $_view, string $_path): string
    {
        $this->currentView = $_view;

        ob_start();

        // Extract data from view into local variables so that they can be accessed directly
        $_data = $_view->getData();

        extract($_data, flags: EXTR_SKIP);

        try {
            include $_path;
        } catch (Throwable $throwable) {
            throw new ViewCompilationError(content: file_get_contents($_path), previous: $throwable);
        }

        $this->currentView = null;

        return trim(ob_get_clean());
    }

    public function escape(null|string|HtmlString|Stringable $value): string
    {
        if ($value instanceof HtmlString) {
            return (string) $value;
        }

        return htmlentities((string) $value);
    }
}
