/*
    * WinDBG Anti-RootKit extension
    * Copyright  2013-2018  Vyacheslav Rusakoff
    * 
    * This program is free software: you can redistribute it and/or modify
    * it under the terms of the GNU General Public License as published by
    * the Free Software Foundation, either version 3 of the License, or
    * (at your option) any later version.
    * 
    * This program is distributed in the hope that it will be useful,
    * but WITHOUT ANY WARRANTY; without even the implied warranty of
    * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    * GNU General Public License for more details.
    * 
    * You should have received a copy of the GNU General Public License
    * along with this program.  If not, see <http://www.gnu.org/licenses/>.

    * This work is licensed under the terms of the GNU GPL, version 3.  See
    * the COPYING file in the top-level directory.
*/

#include <string>
#include <memory>

#include "wdbgark.hpp"
#include "analyze.hpp"
#include "manipulators.hpp"
#include "whitelist.hpp"

namespace wa {

EXT_COMMAND(wa_objtype,
            "Output kernel-mode object type(s)",
            "{type;s,o;type;Object type name}") {
    std::string type = "*";

    RequireKernelMode();

    if ( !Init() ) {
        throw ExtStatusException(S_OK, "global init failed");
    }

    if ( HasArg("type") ) {     // object type was provided
        type.assign(GetArgStr("type"));
    }

    out << wa::showplus << "Displaying \\ObjectTypes\\" << type << endlout;

    auto object_types_directory_offset = m_obj_helper->FindObjectByName("ObjectTypes");

    if ( !object_types_directory_offset ) {
        err << wa::showminus << __FUNCTION__ << ": failed to get \"ObjectTypes\" directory" << endlerr;
        return;
    }

    auto display = WDbgArkAnalyzeBase::Create(m_sym_cache, WDbgArkAnalyzeBase::AnalyzeType::AnalyzeTypeObjType);

    if ( !display->AddRangeWhiteList("nt") ) {
        warn << wa::showqmark << __FUNCTION__ ": AddRangeWhiteList failed" << endlwarn;
    }

    display->SetWhiteListEntries(GetObjectTypesWhiteList());
    display->PrintHeader();

    try {
        if ( type == "*" ) {
            WalkDirectoryObject(object_types_directory_offset,
                                reinterpret_cast<void*>(display.get()),
                                DirectoryObjectTypeCallback);
        } else {
            auto offset = m_obj_helper->FindObjectByName(type, object_types_directory_offset, "\\ObjectTypes\\", false);

            const std::string obj_type("nt!_OBJECT_TYPE");
            ExtRemoteTyped object_type(obj_type.c_str(),
                                       offset,
                                       false,
                                       m_sym_cache->GetCookieCache(obj_type),
                                       nullptr);

            if ( FAILED(DirectoryObjectTypeCallback(this, object_type, reinterpret_cast<void*>(display.get()))) )
                err << wa::showminus << __FUNCTION__ << ": DirectoryObjectTypeCallback failed" << endlerr;
        }
    }
    catch ( const ExtRemoteException &Ex ) {
        err << wa::showminus << __FUNCTION__ << ": " << Ex.GetMessage() << endlerr;
    }
    catch( const ExtInterruptException& ) {
        throw;
    }

    display->PrintFooter();
}

HRESULT WDbgArk::DirectoryObjectTypeCallback(WDbgArk* wdbg_ark_class, const ExtRemoteTyped &object, void* context) {
    WDbgArkAnalyzeBase* display = reinterpret_cast<WDbgArkAnalyzeBase*>(context);

    try {
        const std::string obj_type("nt!_OBJECT_TYPE");
        ExtRemoteTyped object_type(obj_type.c_str(),
                                   object.m_Offset,
                                   false,
                                   wdbg_ark_class->m_sym_cache->GetCookieCache(obj_type),
                                   nullptr);

        auto typeinfo = object_type.Field("TypeInfo");
        display->Analyze(typeinfo, object);
    }
    catch ( const ExtRemoteException &Ex ) {
        err << wa::showminus << __FUNCTION__ << ": " << Ex.GetMessage() << endlerr;
        return Ex.GetStatus();
    }

    return S_OK;
}

}   // namespace wa
