// Copyright 2014 The Closure Library Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview Unit tests for goog.html.SafeStyleSheet and its builders.
 */

goog.provide('goog.html.safeStyleSheetTest');

goog.require('goog.html.SafeStyleSheet');
goog.require('goog.string');
goog.require('goog.string.Const');
goog.require('goog.testing.jsunit');

goog.setTestOnly('goog.html.safeStyleSheetTest');


function testSafeStyleSheet() {
  var styleSheet = 'P.special { color:red ; }';
  var safeStyleSheet =
      goog.html.SafeStyleSheet.fromConstant(goog.string.Const.from(styleSheet));
  var extracted = goog.html.SafeStyleSheet.unwrap(safeStyleSheet);
  assertEquals(styleSheet, extracted);
  assertEquals(styleSheet, safeStyleSheet.getTypedStringValue());
  assertEquals('SafeStyleSheet{' + styleSheet + '}', String(safeStyleSheet));

  // Interface marker is present.
  assertTrue(safeStyleSheet.implementsGoogStringTypedString);
}


/** @suppress {checkTypes} */
function testUnwrap() {
  var evil = {};
  evil.safeStyleSheetValueWithSecurityContract__googHtmlSecurityPrivate_ =
      'P.special { color:expression(evil) ; }';
  evil.SAFE_STYLE_TYPE_MARKER__GOOG_HTML_SECURITY_PRIVATE_ = {};

  var exception = assertThrows(function() {
    goog.html.SafeStyleSheet.unwrap(evil);
  });
  assertTrue(goog.string.contains(
      exception.message,
      'expected object of type SafeStyleSheet'));
}


function testFromConstant_allowsEmptyString() {
  assertEquals(
      goog.html.SafeStyleSheet.EMPTY,
      goog.html.SafeStyleSheet.fromConstant(goog.string.Const.from('')));
}


function testFromConstant_throwsOnLessThanCharacter() {
  assertThrows(function() {
    goog.html.SafeStyleSheet.fromConstant(goog.string.Const.from('x<x'));
  });
}


function testEmpty() {
  assertEquals(
      '', goog.html.SafeStyleSheet.unwrap(goog.html.SafeStyleSheet.EMPTY));
}
