import React from 'react';
import clsx from 'clsx';
import Layout from '@theme/Layout';
import PropTypes from 'prop-types';
import Link from '@docusaurus/Link';
import useDocusaurusContext from '@docusaurus/useDocusaurusContext';
import useBaseUrl from '@docusaurus/useBaseUrl';
import { VisuallyHidden } from '../components/VisuallyHidden';
import { Row, Column } from '../components/Column';
import { Stack } from '../components/Stack';
import { ImageLink } from '../components/ImageLink';
import styles from './index.module.css';

const features = [
	{
		title: 'Development environment',
		subtitle:
			'Focus on one component at a time, see all its variants and work faster with hot reload',
		paragraphs: [
			'Supports JavaScript, TypeScript and Flow',
			'Works with Create React App out of the box',
		],
		figure: (
			<ImageLink href="https://react-styleguidist.js.org/examples/basic/">
				<img src="/img/workbench.jpg" alt="React Styleguidist example style guide" />
			</ImageLink>
		),
	},
	{
		title: 'Style guide',
		subtitle: 'Share components with your team, including designers and developers',
		paragraphs: [
			'All components in one place',
			'Autogenerated usage documentation',
			'Editable live examples',
		],
		figure: (
			<Row>
				<Column size={6}>
					<ImageLink href="https://cdds.netlify.app/styleguide/">
						<img src="/img/styleguide1.jpg" alt="Component-driven.io style guide" />
					</ImageLink>
				</Column>
				<Column size={6}>
					<ImageLink href="https://tamiadev.github.io/tamia/">
						<img src="/img/styleguide2.jpg" alt="React Styleguidist example style guide" />
					</ImageLink>
				</Column>
			</Row>
		),
	},
	{
		title: 'Interactive playground',
		subtitle: 'See how components react to different props and data right in the browser',
		paragraphs: ['Find the right combination of props and copy the code'],
		figure: (
			<ImageLink href="https://react-kawaii.now.sh/" className={styles.featureFigureKawaii}>
				<img src="/img/playground1.png" alt="React Kawaii docs" />
			</ImageLink>
		),
	},
];

const examples = [
	{
		title: 'Example style guide',
		href: 'https://react-styleguidist.js.org/examples/basic/',
		image: '/img/example1.png',
	},
	{
		title: 'Dialog components',
		href: 'https://dialogs.github.io/dialog-web-components/',
		image: '/img/example2.png',
	},
	{
		title: 'Everydayhero Constructicon',
		href: 'https://everydayhero.github.io/constructicon/',
		image: '/img/example3.png',
	},
	{
		title: 'Re-bulma',
		href: 'https://bokuweb.github.io/re-bulma/',
		image: '/img/example4.png',
	},
];

function Feature({ title, subtitle, paragraphs, figure, flipped }) {
	return (
		<Row className={styles.feature}>
			<Column size={4} order={flipped ? 2 : undefined}>
				<Stack gap="m">
					<h3>{title}</h3>
					<p>
						<b>{subtitle}</b>
					</p>
					<Stack gap="s">
						{paragraphs.map((para) => (
							<p key={para}>{para}</p>
						))}
					</Stack>
				</Stack>
			</Column>
			<Column size={8}>
				<div className={styles.featureFigure}>{figure}</div>
			</Column>
		</Row>
	);
}

Feature.propTypes = {
	title: PropTypes.node.isRequired,
	subtitle: PropTypes.node.isRequired,
	paragraphs: PropTypes.arrayOf(PropTypes.node).isRequired,
	figure: PropTypes.node.isRequired,
	flipped: PropTypes.bool.isRequired,
};

function Resource({ href, title, cover }) {
	return (
		<Link to={href} className="col col--6">
			<img src={useBaseUrl(cover)} alt={title} />
		</Link>
	);
}

Resource.propTypes = {
	href: PropTypes.string.isRequired,
	title: PropTypes.string.isRequired,
	cover: PropTypes.string.isRequired,
};

function Home() {
	const context = useDocusaurusContext();
	const { siteConfig = {} } = context;
	return (
		<Layout title={`${siteConfig.title}: ${siteConfig.tagline}`} description={siteConfig.tagline}>
			<header className={clsx('hero', styles.heroBanner)}>
				<Stack gap="l" className="container">
					<Stack gap="m" className="container">
						<img src={useBaseUrl('img/logo.svg')} alt="" width={300} />
						<h1 className="hero__title">{siteConfig.title}</h1>
						<p className="hero__subtitle">{siteConfig.tagline}</p>
					</Stack>
					<div>
						<Link className="button button--lg" to="/docs/getting-started">
							Get Started
						</Link>
					</div>
				</Stack>
			</header>
			<main className={styles.main}>
				<div className="container">
					<section className={styles.section}>
						<VisuallyHidden as="h2">React Styleguidist features</VisuallyHidden>
						<Stack gap="xl">
							{features.map((feature, index) => (
								<Feature key={feature.title} flipped={index % 2} {...feature} />
							))}
						</Stack>
					</section>
					<section className={styles.section}>
						<h2>See it in action</h2>
						<Row>
							{examples.map((example) => (
								<Column key={example.href} size={3}>
									<ImageLink href={example.href}>
										<Stack gap="xs">
											<img src={example.image} alt="" />
											<div>{example.title}</div>
										</Stack>
									</ImageLink>
								</Column>
							))}
						</Row>
					</section>
				</div>
			</main>
		</Layout>
	);
}

export default Home;
