/**
 *     Posjsonhelper library is an open-source project that adds support of
 *     Hibernate query for https://www.postgresql.org/docs/10/functions-json.html)
 *
 *     Copyright (C) 2023  Szymon Tarnowski
 *
 *     This library is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU Lesser General Public
 *     License as published by the Free Software Foundation; either
 *     version 2.1 of the License, or (at your option) any later version.
 *
 *     This library is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *     Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public
 *     License along with this library; if not, write to the Free Software
 *     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *     USA
 */
package com.github.starnowski.posjsonhelper.hibernate5;

import com.github.starnowski.posjsonhelper.core.Context;
import com.github.starnowski.posjsonhelper.core.CoreContextPropertiesSupplier;
import com.github.starnowski.posjsonhelper.core.HibernateContext;
import com.github.starnowski.posjsonhelper.core.HibernateContextPropertiesSupplier;
import com.github.starnowski.posjsonhelper.hibernate5.functions.JsonArrayFunction;
import org.hibernate.dialect.PostgreSQL81Dialect;
import org.hibernate.dialect.function.StandardSQLFunction;
import org.hibernate.type.StandardBasicTypes;

/**
 * The component that enriches the {@link PostgreSQL81Dialect} object with Hibernate and SQL definitions used by the library.
 */
public class PostgreSQLDialectEnricher {

    /**
     * Supplier for {@link Context} object based on system properties
     */
    private final CoreContextPropertiesSupplier coreContextPropertiesSupplier;
    /**
     * Supplier for {@link HibernateContext} object based on system properties
     */
    private final HibernateContextPropertiesSupplier hibernateContextPropertiesSupplier;

    public PostgreSQLDialectEnricher() {
        this(new CoreContextPropertiesSupplier(), new HibernateContextPropertiesSupplier());
    }

    public PostgreSQLDialectEnricher(CoreContextPropertiesSupplier coreContextPropertiesSupplier, HibernateContextPropertiesSupplier hibernateContextPropertiesSupplier) {
        this.coreContextPropertiesSupplier = coreContextPropertiesSupplier;
        this.hibernateContextPropertiesSupplier = hibernateContextPropertiesSupplier;
    }

    /**
     * Invokes {@link #enrich(PostgreSQL81Dialect, Context, HibernateContext)} and pass core context and hibernate context.
     * Generated by {@link #coreContextPropertiesSupplier} and {@link #hibernateContextPropertiesSupplier}
     * @param postgreSQL81Dialect postgres dialect
     */
    public void enrich(PostgreSQL81Dialect postgreSQL81Dialect) {
        enrich(postgreSQL81Dialect, coreContextPropertiesSupplier.get(), hibernateContextPropertiesSupplier.get());
    }

    /**
     * Register function definitions based on {@link HibernateContext} and {@link Context} in {@link PostgreSQL81Dialect} object.
     *
     * @param postgreSQL81Dialect postgres dialect
     * @param context core context based on which enricher component is going to register used functions.
     * @param hibernateContext hibernate context based on which enricher component is going to register used functions.
     */
    public void enrich(PostgreSQL81Dialect postgreSQL81Dialect, Context context, HibernateContext hibernateContext) {
        postgreSQL81Dialect.getFunctions().put(hibernateContext.getJsonFunctionJsonArrayOperator(), new JsonArrayFunction());
        postgreSQL81Dialect.getFunctions().put(hibernateContext.getJsonbAllArrayStringsExistOperator(), new StandardSQLFunction(context.getJsonbAllArrayStringsExistFunctionReference(), StandardBasicTypes.BOOLEAN));
        postgreSQL81Dialect.getFunctions().put(hibernateContext.getJsonbAnyArrayStringsExistOperator(), new StandardSQLFunction(context.getJsonbAnyArrayStringsExistFunctionReference(), StandardBasicTypes.BOOLEAN));
    }

    CoreContextPropertiesSupplier getCoreContextPropertiesSupplier() {
        return coreContextPropertiesSupplier;
    }

    HibernateContextPropertiesSupplier getHibernateContextPropertiesSupplier() {
        return hibernateContextPropertiesSupplier;
    }
}
