import { Observable } from 'observable-fns'
import type { AuthStatus, ModelsData, ResolvedConfiguration, UserProductSubscription } from '../..'
import type { SerializedPromptEditorState } from '../..'
import type { ChatMessage, UserLocalHistory } from '../../chat/transcript/messages'
import type { ContextItem, DefaultContext } from '../../codebase-context/messages'
import type { CodyCommand } from '../../commands/types'
import type { FeatureFlag } from '../../experimentation/FeatureFlagProvider'
import type { ContextMentionProviderMetadata } from '../../mentions/api'
import type { MentionQuery } from '../../mentions/query'
import type { Model } from '../../models/model'
import type { FetchHighlightFileParameters, Prompt } from '../../sourcegraph-api/graphql/client'
import { type createMessageAPIForWebview, proxyExtensionAPI } from './rpc'

export interface WebviewToExtensionAPI {
    /**
     * Get the data to display in the @-mention menu for the given query.
     */
    mentionMenuData(query: MentionQuery): Observable<MentionMenuData>

    /**
     * Get the evaluated value of a feature flag.
     */
    evaluatedFeatureFlag(flag: FeatureFlag): Observable<boolean | undefined>

    /**
     * Observe the results of querying prompts in the Prompt Library. For backcompat, it also
     * includes matching builtin commands and custom commands (which are both deprecated in favor of
     * the Prompt Library).
     */
    prompts(input: PromptsInput): Observable<PromptsResult>

    /**
     * Stream with actions from cody agent service, serves as transport for any client
     * based actions/effects.
     */
    clientActionBroadcast(): Observable<ClientActionBroadcast>

    /** The commands to prompts library migration information. */
    promptsMigrationStatus(): Observable<PromptsMigrationStatus>

    startPromptsMigration(): Observable<void>

    /**
     * The models data, including all available models, site defaults, and user preferences.
     */
    models(): Observable<ModelsData | null>

    /**
     * Observe the list of available chat models.
     */
    chatModels(): Observable<Model[]>

    highlights(query: FetchHighlightFileParameters): Observable<string[][]>

    hydratePromptMessage(
        promptText: string,
        initialContext?: ContextItem[]
    ): Observable<SerializedPromptEditorState>

    /**
     * Set the chat model.
     */
    setChatModel(model: Model['id']): Observable<void>

    /**
     * Observe the default context that should be populated in the chat message input field and suggestions.
     */
    defaultContext(): Observable<DefaultContext>

    detectIntent(
        text: string
    ): Observable<
        { intent: ChatMessage['intent']; allScores: { intent: string; score: number }[] } | undefined
    >

    /**
     * Observe the current resolved configuration (same as the global {@link resolvedConfig}
     * observable).
     */
    resolvedConfig(): Observable<ResolvedConfiguration>

    /**
     * Observe the current auth status (same as the global {@link authStatus} observable).
     */
    authStatus(): Observable<AuthStatus>

    /**
     * Observe the current transcript.
     */
    transcript(): Observable<readonly ChatMessage[]>

    /**
     * The current user's chat history.
     */
    userHistory(): Observable<UserLocalHistory | null>

    /**
     * The current user's product subscription information (Cody Free/Pro).
     */
    userProductSubscription(): Observable<UserProductSubscription | null>
}

export function createExtensionAPI(
    messageAPI: ReturnType<typeof createMessageAPIForWebview>,

    // As a workaround for Cody Web, support providing static initial context.
    staticDefaultContext?: DefaultContext
): WebviewToExtensionAPI {
    const hydratePromptMessage = proxyExtensionAPI(messageAPI, 'hydratePromptMessage')

    return {
        mentionMenuData: proxyExtensionAPI(messageAPI, 'mentionMenuData'),
        evaluatedFeatureFlag: proxyExtensionAPI(messageAPI, 'evaluatedFeatureFlag'),
        prompts: proxyExtensionAPI(messageAPI, 'prompts'),
        clientActionBroadcast: proxyExtensionAPI(messageAPI, 'clientActionBroadcast'),
        models: proxyExtensionAPI(messageAPI, 'models'),
        chatModels: proxyExtensionAPI(messageAPI, 'chatModels'),
        highlights: proxyExtensionAPI(messageAPI, 'highlights'),
        hydratePromptMessage: promptText =>
            hydratePromptMessage(promptText, staticDefaultContext?.initialContext),
        setChatModel: proxyExtensionAPI(messageAPI, 'setChatModel'),
        defaultContext: staticDefaultContext
            ? () => Observable.of(staticDefaultContext)
            : proxyExtensionAPI(messageAPI, 'defaultContext'),
        detectIntent: proxyExtensionAPI(messageAPI, 'detectIntent'),
        promptsMigrationStatus: proxyExtensionAPI(messageAPI, 'promptsMigrationStatus'),
        startPromptsMigration: proxyExtensionAPI(messageAPI, 'startPromptsMigration'),
        resolvedConfig: proxyExtensionAPI(messageAPI, 'resolvedConfig'),
        authStatus: proxyExtensionAPI(messageAPI, 'authStatus'),
        transcript: proxyExtensionAPI(messageAPI, 'transcript'),
        userHistory: proxyExtensionAPI(messageAPI, 'userHistory'),
        userProductSubscription: proxyExtensionAPI(messageAPI, 'userProductSubscription'),
    }
}

export interface MentionMenuData {
    providers: ContextMentionProviderMetadata[]
    items: (ContextItem & { icon?: string })[] | undefined

    /**
     * If an error is present, the client should display the error *and* still display the other
     * data that is present.
     */
    error?: string
}

export interface PromptAction extends Prompt {
    actionType: 'prompt'
}

export interface CommandAction extends CodyCommand {
    actionType: 'command'
}

export interface PromptsInput {
    query: string
    first?: number
    recommendedOnly: boolean
}

export type Action = PromptAction | CommandAction

export interface PromptsResult {
    arePromptsSupported: boolean

    /** List of all available actions (prompts and/or commands) */
    actions: Action[]

    /** The original query used to fetch this result. */
    query: string
}

export type PromptsMigrationStatus =
    | InitialPromptsMigrationStatus
    | InProgressPromptsMigrationStatus
    | SuccessfulPromptsMigrationStatus
    | FailedPromptsMigrationStatus
    | PromptsMigrationSkipStatus
    | NoPromptsMigrationNeeded

interface InitialPromptsMigrationStatus {
    type: 'initial_migration'
}

interface InProgressPromptsMigrationStatus {
    type: 'migrating'

    /**
     * Current number of commands that we've migrated during the current session
     * (current migration run).
     */
    commandsMigrated: number

    /**
     * undefined value means that we're still scanning existing prompts to calculate
     * total commands to migrate (scan first to avoid duplications after migration).
     */
    allCommandsToMigrate: number | undefined
}

interface SuccessfulPromptsMigrationStatus {
    type: 'migration_success'
}

interface FailedPromptsMigrationStatus {
    type: 'migration_failed'
    errorMessage: string
}

interface PromptsMigrationSkipStatus {
    type: 'migration_skip'
}

interface NoPromptsMigrationNeeded {
    type: 'no_migration_needed'
}

export interface ClientActionBroadcast {
    type: 'open-recently-prompts'
}
