/******************************************************************************
*       SOFA, Simulation Open-Framework Architecture, version 1.0 RC 1        *
*                (c) 2006-2011 INRIA, USTL, UJF, CNRS, MGH                    *
*                                                                             *
* This program is free software; you can redistribute it and/or modify it     *
* under the terms of the GNU General Public License as published by the Free  *
* Software Foundation; either version 2 of the License, or (at your option)   *
* any later version.                                                          *
*                                                                             *
* This program is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for    *
* more details.                                                               *
*                                                                             *
* You should have received a copy of the GNU General Public License along     *
* with this program; if not, write to the Free Software Foundation, Inc., 51  *
* Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.                   *
*******************************************************************************
*                            SOFA :: Applications                             *
*                                                                             *
* Authors: The SOFA Team and external contributors (see Authors.txt)          *
*                                                                             *
* Contact information: contact@sofa-framework.org                             *
******************************************************************************/
/** \example glutAnt.cpp
 * Glut application with picking and AntTweak
 */

/**
  A simple glut application featuring a Sofa simulation.
  Contrary to other projects, this does not use a sofa::gui.

  @author Francois Faure, 2014
  */

#include <iostream>
#include <fstream>
#include <ctime>
#include <GL/glew.h>
#include <GL/glut.h>
using std::endl;
using std::cout;

#include <AntTweakBar.h>

#include <sofa/helper/ArgumentParser.h>
#include <SofaSimpleGUI/SofaGLScene.h>
using namespace sofa::newgui;

// ---------------------------------------------------------------------
// Sofa interface
// ---------------------------------------------------------------------
sofa::newgui::SofaGLScene sofaScene;     ///< The interface of the application with Sofa
sofa::newgui::SpringInteractor* drag = NULL; ///< Mouse interactor


// ---------------------------------------------------------------------
// Various shared variables for glut
GLfloat light_flow[] = { -25.0, 4.0, -25.0 }; // opposite of light direction
GLfloat light_ambient[] = { 0.2, 0.2, 0.2, 0.0 };
GLfloat light_diffuse[] = { 1.0, 1.0, 1.0, 0.0 };
GLfloat light_specular[] = { 1.0, 1.0, 1.0, 0.0 };

GLfloat camera_position[] = { 0.0, 0.0, 25.0, 0.0 };
GLfloat znear = camera_position[2]-10;
GLfloat zfar = camera_position[2]+10;

bool isControlPressed(){ return glutGetModifiers()&GLUT_ACTIVE_CTRL; }
bool isShiftPressed(){ return glutGetModifiers()&GLUT_ACTIVE_SHIFT; }
bool isAltPressed(){ return glutGetModifiers()&GLUT_ACTIVE_ALT; }

bool animating = true;
bool interacting = false;



void initGL(void)
{
    glClearColor (0.0, 0.0, 0.0, 0.0);

    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_DEPTH_TEST);
}


void display(void)
{
    glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity ();

    float v[4];
    for(int i=0; i<3; i++) v[i] = -light_flow[i]; v[3] = 0;
    glLightfv(GL_LIGHT0, GL_POSITION, v);
    gluLookAt ( camera_position[0],camera_position[1],camera_position[2], 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);

    sofaScene.glDraw();

    // display a box, for debug
    glColor3f (1.0, 0.0, 0.0);
    glutWireCube (1.0);

    TwDraw();  // draw the tweak bar(s)

    glutSwapBuffers();
    // Due to some bug, the first display displays nothing. Hence this poor fix:
    static int first = true;
    if( first ) { glutPostRedisplay(); first = false; }
}

void reshape (int width, int height)
{
    glViewport (0, 0, (GLsizei) width, (GLsizei) height);
    glMatrixMode (GL_PROJECTION);
    glLoadIdentity ();
    gluPerspective (55.0, (GLfloat) width/(GLfloat) height, znear, zfar );
    glMatrixMode (GL_MODELVIEW);

    // Send the new window size to AntTweakBar
    TwWindowSize(width, height);
}

void keyboard(unsigned char key, int charX, int charY)
{
    if ( TwEventKeyboardGLUT(key, charX, charY) ) return;

    switch (key) {
    case 27:
        exit(0);
        break;
    }

    if( key == ' ' ){
        animating = !animating;
    }

}

void idle()
{
    if( !animating )
        return;

    sofaScene.animate();
    glutPostRedisplay();
}


void mouseButton(int button, int state, int mouseX, int mouseY)
{
//    cout<<"mousebutton, modifiers = " << glutGetModifiers() << endl;
    if( TwEventMouseButtonGLUT(button, state, mouseX, mouseY) ) return;

    switch (button) {
    case GLUT_LEFT_BUTTON:
        if (state == GLUT_DOWN)
        {
            PickedPoint glpicked = sofaScene.pick(camera_position[0],camera_position[1],camera_position[2], mouseX,mouseY);
            if( glpicked )
            {
                interacting = true;

                drag = new SpringInteractor(glpicked);
                sofaScene.attach(drag);
//                cout << "Particle glpicked: " << glpicked << endl;
                sofaScene.printGraph();
            }
            else {
                cout << "no particle glpicked" << endl;
            }
        }
        else
        {       
            if( interacting )
            {
                sofaScene.detach(drag);
                delete drag;

                interacting = false;
            }
        }
        break;
    case GLUT_RIGHT_BUTTON:
        //        if (state == GLUT_DOWN)
        //            exit(0);
        break;
    default:
        break;
    }

    glutPostRedisplay();
}

void mouseMotion(int mouseX, int mouseY)
{
    if( TwEventMouseMotionGLUT(mouseX, mouseY) ) return;

    if( interacting )
    {
        sofaScene.move(drag, mouseX,mouseY);
    }
    else{
    }
}


int main(int argc, char** argv)
{
    glutInit(&argc,argv);
    glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH );
    int myWindowWidth = 800, myWindowHeight = 600;
    glutInitWindowSize ( myWindowWidth, myWindowHeight );
    glutInitWindowPosition (100, 100);
    glutCreateWindow (argv[0]);

    // AntTweak interface
    TwInit(TW_OPENGL, NULL);
    TwWindowSize(myWindowWidth, myWindowHeight);
    TwBar *myBar = TwNewBar("SofaTweakBar");
    TwDefine(" GLOBAL help='This example shows how to integrate AntTweakBar with GLUT and OpenGL.' "); // Message added to the help bar.
    TwDefine(" SofaTweakBar size='200 100' color='96 216 224' "); // change default tweak bar size and color
    TwAddVarRW(myBar, "LightDir", TW_TYPE_DIR3F, &light_flow, " label='Light direction' opened=true help='Change the light direction.' ");

    initGL();
    glutDisplayFunc(display);
    glutReshapeFunc(reshape);
    glutKeyboardFunc(keyboard);
    glutIdleFunc(idle);
    glutMotionFunc(mouseMotion);
    glutPassiveMotionFunc((GLUTmousemotionfun)TwEventMouseMotionGLUT); // send it to AntTweakBar
    glutMouseFunc(mouseButton);

    // --- Parameter initialisation ---
    std::string fileName ;

    sofa::helper::parse("Simple glut application featuring a Sofa scene.")
            .option(&sofaScene.plugins,'l',"load","load given plugins")
            .parameter(&fileName,'f',"file","scene file to load")
            (argc,argv);

    // --- Init sofa ---
    sofaScene.init(fileName);

    glutMainLoop();


    return 0;
}

