using Microsoft.AspNetCore.Identity;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Logging.Abstractions;
using Moq;
using NUnit.Framework;
using SecureWebApp.Interfaces;
using SecureWebApp.Models;
using SecureWebApp.Services;
using System.Threading.Tasks;

namespace SecureWebApp.Tests
{
    public class EmailSenderTests
    {
        private IMailService _mailService;

        private ILogger<IMailService> _emailLogger;
        private Mock<IIpAddressService> _ipAddressServiceMock;
        private Mock<IConfiguration> _configMock;

        [SetUp]
        public void Setup()
        {
            _emailLogger = new Logger<IMailService>(new NullLoggerFactory());
            _ipAddressServiceMock = new Mock<IIpAddressService>();
            _configMock = new Mock<IConfiguration>();

            _mailService = new EmailSenderService(_emailLogger,
                _ipAddressServiceMock.Object, _configMock.Object);
        }

        [Test]
        public async Task Test_SendSimpleMail()
        {
            // Arrange
            var msg = new EmailMessage
            {
                To = "example@example.org",
                Subject = "Test Message",
                HtmlMessage = "Email message generated by an unit test."
            };

            // Act
            await _mailService.SendEmailAsync(msg);

            // Assert
            Assert.Pass();
        }

        [Test]
        public async Task Test_SendUnlockEmail()
        {
            // Arrange
            const string userid = "FF7DAB18-AD57-4989-981A-9263F5305FDA";
            const string username = "Test User";
            const string email = "testuser@example.com";

            const string ip = "12.34.56.78";
            const string token = "aGVsbG86d29ybGQ=";
            
            var user = new IdentityUser
            {
                Id = userid,
                UserName = username,
                NormalizedUserName = username.ToUpper(),
                Email = email,
                NormalizedEmail = email.ToUpper(),
            };

            _ipAddressServiceMock.Setup(_ => _.GetRequestIp(It.IsAny<bool>())).Returns(ip);
            _configMock.Setup(_ => _["ServerName"]).Returns("example.com");

            // Act
            var msg = await _mailService.GetEmailTemplateAsync(EmailSenderService.Templates.AccountUnlock, username, ip, userid, token);
            var res = await _mailService.SendAccountUnlockEmailAsync(user, token);

            // Assert
            Assert.AreEqual(email, res.To);
            Assert.AreEqual(msg, res.HtmlMessage);
        }
    }
}