import template from './sw-order-detail.html.twig';
import './sw-order-detail.scss';
import swOrderDetailState from '../../state/order-detail.store';

/**
 * @package checkout
 */

const { State, Mixin, Utils } = Shopware;
const { Criteria } = Shopware.Data;
const { array } = Utils;
const { mapState } = Shopware.Component.getComponentHelper();
const ApiService = Shopware.Classes.ApiService;

// eslint-disable-next-line sw-deprecation-rules/private-feature-declarations
export default {
    template,

    compatConfig: Shopware.compatConfig,

    inject: [
        'repositoryFactory',
        'acl',
        'orderService',
        'feature',
    ],

    provide() {
        return {
            swOrderDetailOnIdentifierChange: this.updateIdentifier,
            swOrderDetailOnCreatedByIdChange: this.updateCreatedById,
            swOrderDetailOnLoadingChange: this.onUpdateLoading,
            swOrderDetailOnEditingChange: this.onUpdateEditing,
            swOrderDetailOnSaveAndRecalculate: this.onSaveAndRecalculate,
            swOrderDetailOnRecalculateAndReload: this.onRecalculateAndReload,
            swOrderDetailOnReloadEntityData: this.reloadEntityData,
            swOrderDetailOnSaveAndReload: this.onSaveAndReload,
            swOrderDetailOnSaveEdits: this.onSaveEdits,
            swOrderDetailOnError: this.onError,
        };
    },

    mixins: [
        Mixin.getByName('notification'),
    ],

    props: {
        orderId: {
            type: String,
            required: false,
            default: null,
        },
    },

    data() {
        return {
            /*
             * @deprecated tag:v6.7.0 - identifier will be removed
             */
            identifier: '',
            isEditing: false,
            isLoading: true,
            isSaveSuccessful: false,
            createdById: '',
            isDisplayingLeavePageWarning: false,
            nextRoute: null,
            hasNewVersionId: false,
            hasOrderDeepEdit: false,
            missingProductLineItems: [],
            promotionsToDelete: [],
            deliveryDiscountsToDelete: [],
        };
    },

    metaInfo() {
        return {
            title: this.$createTitle(this.orderIdentifier),
        };
    },

    computed: {
        ...mapState('swOrderDetail', [
            'order',
            'versionContext',
            'orderAddressIds',
            'editing',
            'loading',
        ]),

        orderIdentifier() {
            return this.order !== null ? this.order.orderNumber : '';
        },

        orderChanges() {
            if (!this.order) {
                return false;
            }

            return this.orderRepository.hasChanges(this.order);
        },

        showTabs() {
            return this.$route.meta.$module.routes.detail.children.length > 1;
        },

        showWarningTabStyle() {
            return this.isOrderEditing && this.$route.name === 'sw.order.detail.documents';
        },

        isOrderEditing() {
            return this.orderChanges || this.hasOrderDeepEdit || this.orderAddressIds?.length > 0;
        },

        orderRepository() {
            return this.repositoryFactory.create('order');
        },

        automaticPromotions() {
            return this.order.lineItems.filter((item) => item.type === 'promotion' && item.referencedId === null);
        },

        deliveryDiscounts() {
            return array.slice(this.order.deliveries, 1) || [];
        },

        orderCriteria() {
            const criteria = new Criteria(1, 25);

            criteria.addAssociation('currency').addAssociation('orderCustomer.salutation').addAssociation('language');

            criteria
                .getAssociation('lineItems')
                .addFilter(Criteria.equals('parentId', null))
                .addSorting(Criteria.sort('position', 'ASC'));

            criteria.getAssociation('lineItems.children').addSorting(Criteria.sort('position', 'ASC'));

            criteria.addAssociation('salesChannel');

            criteria
                .addAssociation('addresses.country')
                .addAssociation('addresses.countryState')
                .addAssociation('deliveries.shippingMethod')
                .addAssociation('deliveries.shippingOrderAddress')
                .addAssociation('transactions.paymentMethod')
                .addAssociation('documents.documentType')
                .addAssociation('tags');

            criteria.addAssociation('stateMachineState');

            criteria
                .getAssociation('deliveries')
                .addAssociation('stateMachineState')
                .addSorting(Criteria.sort('shippingCosts.unitPrice', 'DESC'));

            criteria
                .getAssociation('transactions')
                .addAssociation('stateMachineState')
                .addSorting(Criteria.sort('createdAt'));

            criteria.addAssociation('billingAddress');

            return criteria;
        },

        convertedProductLineItems() {
            return (
                this.order?.lineItems?.filter((lineItem) => {
                    return (
                        lineItem.payload?.isConvertedProductLineItem &&
                        lineItem.type === 'custom' &&
                        !this.missingProductLineItems.includes(lineItem)
                    );
                }) || []
            );
        },
    },

    watch: {
        orderId() {
            this.createdComponent();
        },

        isOrderEditing(value) {
            this.updateEditing(value);
        },
    },

    beforeCreate() {
        State.registerModule('swOrderDetail', swOrderDetailState);
    },

    beforeUnmount() {
        this.beforeDestroyComponent();

        State.unregisterModule('swOrderDetail');
    },

    beforeRouteLeave(to, from, next) {
        if (this.isOrderEditing) {
            this.nextRoute = next;
            this.isDisplayingLeavePageWarning = true;
        } else {
            next();
        }
    },

    created() {
        this.createdComponent();
    },

    methods: {
        createdComponent() {
            Shopware.ExtensionAPI.publishData({
                id: 'sw-order-detail-base__order',
                path: 'order',
                scope: this,
            });

            window.addEventListener('beforeunload', this.beforeDestroyComponent);

            Shopware.State.commit('shopwareApps/setSelectedIds', this.orderId ? [this.orderId] : []);

            this.createNewVersionId();
        },

        async beforeDestroyComponent() {
            if (this.hasNewVersionId) {
                const oldVersionContext = this.versionContext;
                State.commit('swOrderDetail/setVersionContext', Shopware.Context.api);
                this.hasNewVersionId = false;

                // clean up recently created version
                await this.orderRepository.deleteVersion(this.orderId, oldVersionContext.versionId, oldVersionContext);
            }
        },

        updateIdentifier(identifier) {
            this.identifier = identifier;
        },

        updateCreatedById(createdById) {
            this.createdById = createdById;
        },

        onChangeLanguage() {
            this.$root.$emit('language-change');
        },

        saveEditsFinish() {
            this.isSaveSuccessful = false;
            this.isEditing = false;
        },

        onStartEditing() {
            this.$root.$emit('order-edit-start');
        },

        async onSaveEdits() {
            this.isLoading = true;

            await this.handleOrderAddressUpdate(this.orderAddressIds);

            if (this.promotionsToDelete.length > 0) {
                this.order.lineItems = this.order.lineItems.filter(
                    (lineItem) => !this.promotionsToDelete.includes(lineItem.id),
                );
            }

            if (this.order.lineItems.length === 0) {
                this.createNotificationError({
                    message: this.$tc('sw-order.detail.messageEmptyLineItems'),
                });

                this.createNewVersionId().then(() => {
                    State.commit('swOrderDetail/setLoading', [
                        'order',
                        false,
                    ]);
                });

                return;
            }

            if (this.deliveryDiscountsToDelete.length > 0) {
                this.order.deliveries = this.order.deliveries.filter(
                    (delivery) => !this.deliveryDiscountsToDelete.includes(delivery.id),
                );
            }

            await this.orderRepository
                .save(this.order, this.versionContext)
                .then(() => {
                    this.hasOrderDeepEdit = false;
                    this.promotionsToDelete = [];
                    this.deliveryDiscountsToDelete = [];
                    return this.orderRepository.mergeVersion(this.versionContext.versionId, this.versionContext);
                })
                .then(() => this.createNewVersionId())
                .then(() => {
                    State.commit('swOrderDetail/setSavedSuccessful', true);
                })
                .catch((error) => {
                    this.onError('error', error);
                    this.isLoading = false;
                });

            this.$root.$emit('order-edit-save');
        },

        async handleOrderAddressUpdate(addressMappings) {
            const mappings = [];

            addressMappings.forEach((addressMapping) => {
                // If they are the same means that the address has not changed, so skip it
                if (addressMapping.customerAddressId === addressMapping.orderAddressId) {
                    return;
                }

                const mapping = {
                    customerAddressId: addressMapping.customerAddressId,
                    type: addressMapping.type,
                };

                if (addressMapping.type === 'shipping') {
                    mapping.deliveryId = this.order.deliveries[0].id;
                }

                mappings.push(mapping);
            });

            if (mappings.length === 0) {
                State.commit('swOrderDetail/setOrderAddressIds', false);

                return;
            }

            await this.updateOrderAddresses(mappings)
                .then(() => {
                    State.commit('swOrderDetail/setOrderAddressIds', false);
                })
                .catch((error) => {
                    this.createNotificationError({
                        message: error,
                    });
                });
        },

        onCancelEditing() {
            this.isLoading = true;
            State.commit('swOrderDetail/setLoading', [
                'order',
                true,
            ]);

            const oldVersionContext = this.versionContext;
            State.commit('swOrderDetail/setVersionContext', Shopware.Context.api);
            this.hasNewVersionId = false;

            return this.orderRepository
                .deleteVersion(this.orderId, oldVersionContext.versionId, oldVersionContext)
                .then(() => {
                    this.hasOrderDeepEdit = false;
                    State.commit('swOrderDetail/setOrderAddressIds', false);
                })
                .catch((error) => {
                    this.onError('error', error);
                })
                .finally(() => {
                    this.missingProductLineItems = [];

                    return this.createNewVersionId().then(() => {
                        State.commit('swOrderDetail/setLoading', [
                            'order',
                            false,
                        ]);
                    });
                });
        },

        async onSaveAndRecalculate() {
            State.commit('swOrderDetail/setLoading', [
                'order',
                true,
            ]);
            this.isLoading = true;

            this.order.lineItems = this.order.lineItems.filter((lineItem) => !this.automaticPromotions.includes(lineItem));
            this.order.deliveries = this.order.deliveries.filter((delivery) => !this.deliveryDiscounts.includes(delivery));

            try {
                await this.orderRepository.save(this.order, this.versionContext);
                await this.orderService.recalculateOrder(this.orderId, this.versionContext.versionId, {}, {});
                await this.orderService.toggleAutomaticPromotions(this.orderId, this.versionContext.versionId, false);
                await this.reloadEntityData();
            } catch (error) {
                this.onError('error', error);
            } finally {
                this.isLoading = false;
                Shopware.State.commit('swOrderDetail/setLoading', [
                    'order',
                    false,
                ]);
            }
        },

        async onRecalculateAndReload() {
            State.commit('swOrderDetail/setLoading', [
                'order',
                true,
            ]);
            try {
                this.promotionsToDelete = this.automaticPromotions.map((promotion) => promotion.id);
                this.deliveryDiscountsToDelete = this.deliveryDiscounts.map((discount) => discount.id);
                await this.orderService.recalculateOrder(this.orderId, this.versionContext.versionId, {}, {});
                await this.orderService.toggleAutomaticPromotions(this.orderId, this.versionContext.versionId, false);
                await this.reloadEntityData();
                this.order.lineItems = this.order.lineItems.filter(
                    (lineItem) => !this.promotionsToDelete.includes(lineItem.id),
                );
                this.order.deliveries = this.order.deliveries.filter(
                    (delivery) => !this.deliveryDiscountsToDelete.includes(delivery.id),
                );
            } catch (error) {
                this.onError('error', error);
                this.promotionsToDelete = [];
                this.deliveryDiscountsToDelete = [];
            } finally {
                Shopware.State.commit('swOrderDetail/setLoading', [
                    'order',
                    false,
                ]);
            }
        },

        onSaveAndReload() {
            State.commit('swOrderDetail/setLoading', [
                'order',
                true,
            ]);

            return this.orderRepository
                .save(this.order, this.versionContext)
                .then(() => this.reloadEntityData())
                .catch((error) => {
                    this.onError('error', error);
                })
                .finally(() => {
                    Shopware.State.commit('swOrderDetail/setLoading', [
                        'order',
                        false,
                    ]);
                });
        },

        onUpdateLoading(loadingValue) {
            this.isLoading = loadingValue;
        },

        onUpdateEditing(editingValue) {
            this.isEditing = editingValue;
        },

        onError(error) {
            let errorDetails = null;

            try {
                errorDetails = error.response.data.errors[0].detail;
            } catch (e) {
                errorDetails = '';
            }

            this.createNotificationError({
                message: this.$tc('sw-order.detail.messageRecalculationError') + errorDetails,
            });
        },

        onLeaveModalClose() {
            this.nextRoute(false);
            this.nextRoute = null;
            this.isDisplayingLeavePageWarning = false;
        },

        onLeaveModalConfirm() {
            this.isDisplayingLeavePageWarning = false;

            this.$nextTick(() => {
                this.nextRoute();
            });
        },

        reloadEntityData(isSaved = true) {
            State.commit('swOrderDetail/setLoading', [
                'order',
                true,
            ]);

            return this.orderRepository
                .get(this.orderId, this.versionContext, this.orderCriteria)
                .then((response) => {
                    if (this.$route.name !== 'sw.order.detail.documents' && isSaved) {
                        this.hasOrderDeepEdit = true;
                    }

                    State.commit('swOrderDetail/setOrder', response);
                })
                .finally(() => {
                    Shopware.State.commit('swOrderDetail/setLoading', [
                        'order',
                        false,
                    ]);
                    this.isLoading = false;
                });
        },

        createNewVersionId() {
            // Reset the current version context
            State.commit('swOrderDetail/setVersionContext', Shopware.Context.api);
            this.hasNewVersionId = false;

            return this.orderRepository
                .createVersion(this.orderId, this.versionContext)
                .then((newContext) => {
                    this.hasNewVersionId = true;

                    State.commit('swOrderDetail/setVersionContext', newContext);

                    return this.reloadEntityData(false);
                })
                .then(() => this.convertMissingProductLineItems());
        },

        updateOrderAddresses(mappings) {
            return this.orderService.updateOrderAddresses(
                this.orderId,
                mappings,
                {},
                ApiService.getVersionHeader(this.order.versionId),
            );
        },

        updateEditing(value) {
            State.commit('swOrderDetail/setEditing', value);
        },

        convertMissingProductLineItems() {
            this.missingProductLineItems =
                this.order?.lineItems?.filter((lineItem) => {
                    return lineItem.productId === null && lineItem.type === 'product';
                }) || [];

            if (this.missingProductLineItems.length === 0) {
                return Promise.resolve();
            }

            this.missingProductLineItems.forEach((lineItem) => {
                lineItem.type = 'custom';
                lineItem.productId = null;
                lineItem.referencedId = null;
                lineItem.payload.isConvertedProductLineItem = true;
            });

            return this.orderRepository.save(this.order, this.versionContext);
        },
    },
};
