# frozen_string_literal: true

require 'spec_helper'

describe Shoes::Swt::TextBlock::TextSegment do
  let(:layout) do
    double("layout", text: "the text",
                     :alignment= => nil, :justify= => nil, :spacing= => nil,
                     :text= => nil, :width= => nil,
                     disposed?: false, dispose: nil,
                     set_style: nil, bounds: bounds)
  end

  let(:bounds) { Java::OrgEclipseSwtGraphics::Rectangle.new(0, 0, 0, 0) }
  let(:element_left) { 0 }
  let(:element_top)  { 0 }
  let(:segment_width)  { 10 }
  let(:segment_height) { 10 }
  let(:left_offset)   { 5 }
  let(:top_offset)    { 5 }

  let(:font_factory) { double("font factory", create_font: font, dispose: nil) }
  let(:style_factory) { double("style factory", create_style: style, dispose: nil) }
  let(:font)   { double("font") }
  let(:style)  { double("style") }

  let(:style_hash) do
    {
      bg: double("bg"),
      fg: double("fg"),
      font_detail: {
        name: "Comic Sans",
        size: 18,
        styles: nil
      }
    }
  end

  let(:dsl) { double("dsl", font: "", size: 16, style: {}) }

  before(:each) do
    allow(::Swt::TextLayout).to receive(:new)            { layout }
    allow(Shoes::Swt::TextFontFactory).to receive(:new)  { font_factory }
    allow(Shoes::Swt::TextStyleFactory).to receive(:new) { style_factory }
  end

  subject do
    segment = Shoes::Swt::TextBlock::TextSegment.new(dsl, "text", segment_width)
    segment.position_at(element_left, element_top)
  end

  describe "disposal" do
    it "disposes of underlying layout" do
      allow(layout).to receive(:disposed?) { false }
      expect(layout).to receive(:dispose)
      subject.dispose
    end

    it "doesn't overdispose" do
      allow(layout).to receive(:disposed?) { true }
      expect(layout).to_not receive(:dispose)
      subject.dispose
    end
  end

  describe "setting style" do
    it "on full range" do
      subject.set_style(style_hash)
      expect(layout).to have_received(:set_style)
        .with(style, 0, layout.text.length - 1)
        .at_least(1).times
    end

    it "with a range" do
      subject.set_style(style_hash, 1..2)
      expect(layout).to have_received(:set_style).with(style, 1, 2)
    end

    it "ignores empty ranges" do
      subject.set_style(style_hash, 0...0)
      expect(layout).to_not have_received(:set_style).with(style, 0, 0)
    end
  end

  describe "shrinking on initialization" do
    it "happens when too long for container" do
      bounds.width = segment_width + 10
      expect(subject.layout).to have_received(:width=).with(segment_width)
    end

    it "happens even when enough containing width" do
      expect(subject.layout).to have_received(:width=).with(segment_width)
    end
  end

  describe "bounds checking" do
    before(:each) do
      set_bounds(0, 0, segment_width, segment_height)
    end

    it "checks boundaries" do
      expect(subject.in_bounds?(1, 1)).to be_truthy
    end

    describe "offsets left" do
      let(:element_left) { left_offset }

      it "checks boundaries" do
        expect(subject.in_bounds?(segment_width + left_offset - 1, 0)).to be_truthy
      end
    end

    describe "offsets top" do
      let(:element_top) { top_offset }

      it "checks boundaries" do
        expect(subject.in_bounds?(0, segment_height + top_offset - 1)).to be_truthy
      end
    end

    def set_bounds(x, y, width, height)
      bounds.x = x
      bounds.y = y
      bounds.width = width
      bounds.height = height
    end
  end

  describe "dispose" do
    it "should dispose its Swt fonts" do
      subject.dispose
      expect(font_factory).to have_received(:dispose)
    end

    it "should dispose its Swt colors" do
      subject.dispose
      expect(style_factory).to have_received(:dispose)
    end
  end

  describe "height" do
    let(:line_height)  { 17 }
    let(:line_spacing) { 4 }
    let(:line_bounds)  { ::Swt::Rectangle.new(-1, -1, -1, line_height) }

    before do
      allow(layout).to receive(:spacing).and_return(line_spacing)
      allow(layout).to receive(:get_line_bounds).and_return(line_bounds)
    end

    it "has a single line" do
      with_line_count(1)
      expect(subject.height).to eq(17)
    end

    it "has two lines" do
      with_line_count(2)
      expect(subject.height).to eq(38)
    end

    it "has three lines" do
      with_line_count(3)
      expect(subject.height).to eq(59)
    end

    def with_line_count(count)
      allow(layout).to receive(:line_count).and_return(count)
    end
  end
end
