---
head.title: 'Laravel Task Scheduler with Docker - Docker PHP - Server Side Up'
description: 'Learn how to configure a Laravel Task Scheduler with Docker.'
layout: docs
---

# Laravel Task Scheduler with Docker
Running a Laravel task scheduler with Docker can be a little different from the traditional methods.

## Important concepts
1. If you're using `fpm-apache` or `fpm-nginx`, might need to set the stop signal to `SIGTERM` for a graceful shutdown (see this [PR](https://github.com/serversideup/docker-php/pull/437) for more details why)
1. Be sure to set the health check
1. We will **not** use `cron` to run the scheduler
1. By default `schedule:work` checks every minute, so we will use that to run the system process
1. The actual time trigger itself is set within Laravel

## More detail
We need to run the [schedule:work](https://laravel.com/docs/11.x/scheduling#running-the-scheduler-locally) command from Laravel. Although the docs say "Running the scheduler locally", this is what we want in production. It will run the scheduler in the foreground and execute it every minute. You can configure your Laravel app for the exact time that a command should run through a [scheduled task](https://laravel.com/docs/11.x/scheduling#scheduling-artisan-commands).


## Examples
Here is a simplified example of how you can achieve this with Docker Compose:

::note
Notice we're calling the artisan command explicitly with the full path (`/var/www/html/artisan`). This is because we need to run the command from the context of the container.
::

::code-panel
---
label: Example & Simplified Docker Compose File
---
```yaml
services:
  php:
    image: my/laravel-app

  task:
    image: my/laravel-app
    command: ["php", "/var/www/html/artisan", "schedule:work"]
    stop_signal: SIGTERM # Set this for graceful shutdown if you're using fpm-apache or fpm-nginx
    healthcheck:
      # This is our native healthcheck script for the scheduler
      test: ["CMD", "healthcheck-schedule"]
      start_period: 10s
```
::


This is an example how we would set the actual execution time within Laravel itself:
::code-panel
---
label: Example in Laravel (version <= 10) using `Kernel.php`
---
```php
<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * The Artisan commands provided by your application.
     *
     * @var array
     */
    protected $commands = [
        //
    ];

    /**
     * Define the application's command schedule.
     *
     * @param  \Illuminate\Console\Scheduling\Schedule  $schedule
     * @return void
     */
    protected function schedule(Schedule $schedule)
    {
        $schedule->command('process:invoices')->daily()->at('02:00')->timezone('America/Chicago');
        $schedule->command('process:latefees')->daily()->at('04:00')->timezone('America/Chicago');
    }

    /**
     * Register the commands for the application.
     *
     * @return void
     */
    protected function commands()
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
```
::

::code-panel
---
label: Example in Laravel (version >= 11) using `routes/console.php`
---
```php
<?php
 
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schedule;
 
Schedule::call(function () {
    DB::table('recent_users')->delete();
})->daily()->at('04:00')->timezone('America/Chicago');
```
::

## Get Up and Running The Easy Way
We do all the heavy lifting for you with Spin Pro. It's as easy as selecting it in a menu and we'll configure everything else for you. Learn how easy it is to get up and running with the Task Scheduler on Spin Pro.

[Learn more about Laravel Task Scheduler + Spin Pro →](https://getspin.pro/docs/services/laravel-scheduler)
