---
head.title: 'Migrating from v2 to v3 - Server Side Up'
description: 'Learn how to migrate from serversideup/php v2 images to v3.'
layout: docs
---

# Migrating from v2 to v3
::lead-p
If you're moving from v2 to v3, there are a number of changes you should be aware of. We've tried to keep these to a minimum, but some of these changes were necessary to make the project more maintainable and easier to use.
::

## 👉 Preparing for the migration
If you're an existing user of our v2 images, be sure that your current configurations are NOT set to use the latest images. To do this, you can lock your images into the `v2.2.1` tag. This will ensure that you're not automatically upgraded to the v3 images.

For example, if you are using `8.2-fpm-nginx`, you would change your `docker-compose.yml` file to use the [`v2.2.1`](https://hub.docker.com/r/serversideup/php/tags?page=1&name=2.2.1) tag:

::code-panel
---
label: Original `docker-compose.yml` file
---
```yaml
services:
  php:
    image: serversideup/php:8.2-fpm-nginx
    ports:
      - 80:80
    volumes:
      - .:/var/www/html
```
::

::code-panel
---
label: Updated `docker-compose.yml` file
---
```yaml
services:
  php:
    image: serversideup/php:8.2-fpm-nginx-v2.2.1
    ports:
      - 80:80
    volumes:
      - .:/var/www/html
```
::

All you need to do is add `-v2.2.1` to the end of the image tag. This will ensure that you're not automatically upgraded to the v3 images.


## 🚀 New Features
We've been busy overhauling our PHP Docker Images to make them more production-ready and easier to use. Here are some of the new features we've added:
- **Based on official PHP Images** - We're now building an improved developer experience on top of the official PHP Docker images.
- **Unprivileged by default** - We're now running our images as an unprivileged user by default. This is a huge step forward in security and compatibility.
- **PHP 8.4 support** - We're now shipping the latest and greatest.
- **Pin to the exact minor version** - Pin your app to the exact minor version of PHP that you want to use. This means you can pin to `8.2.12` instead of `8.2`.
- **Easier start up script customization** - We now have a folder called `/etc/entrypoint.d` that allows you to easily customize your container with scripts. Just put them in numerical order and we'll execute any shell script you want. No S6 Overlay knowledge required.
- **Expanded Laravel Automations** - We added automations to run `config:cache`, `route:cache`, `view:cache`, `event:cache`, `migrate --force --isolated`, and `storage:link`
- **NGINX Unit Support** - We're offering NGINX Unit as a variation as an alternative to PHP-FPM. This allows you to run PHP applications without the need for a webserver like NGINX or Apache to run with PHP-FPM.
- **Available on GitHub Packages** - We're now publishing our images to GitHub Packages. This means you can use our images without needing to authenticate with Docker Hub.

## ⚠️ Breaking changes
The following changes are considered to be "breaking changes" and will require you to make changes to your application.

### Ubuntu is no longer used as a base image
We now use Debian or Alpine as our base OS (because we're using the official PHP images as a base). This is a huge change, but we're confident this will be the best direction moving forward.

### `ppa:ondrej/php` is no longer used
Since we're using PHP.net as the "official source of truth" for getting our PHP versions, this means we're also dropping support for the `ppa:ondrej/php` repository. If you're using things like `apt-get install php-redis` you will need to change your method of installing PHP extensions.

[Learn how to install your own PHP extension →](/docs/customizing-the-image/installing-additional-php-extensions)

### `webuser` is no longer being used
We used to add a user called `webuser` with the UID of `9999` with shell permissions. To increase security, we're now using the `www-data` user and group that is built into the official PHP images. If you have mounted volumes, you will need to `chown` the files to match the ID of the `www-data` user and groups. For Debian, this is `33:33` and for Alpine, this is `82:82`.

### NGINX and Apache listen on 8080 (HTTP) and 8443 (HTTPS) by default
Our images are now unprivileged by default. This is a major step forward in security and compatibility. Since we are unprivileged by default, we lose the ability to mount on ports less than 1024. If you're using NGINX or Apache, you will need to update your port mappings to use `8080` and `8443` instead of `80` and `443`.

[Learn more about this change →](/docs/getting-started/default-configurations#unprivileged-by-default)

### S6 Overlay is only used in `*-fpm-apache` and `*-fpm-nginx` images
Due to compatibility issues, we only use S6 Overlay in our `*-fpm-apache` and `*-fpm-nginx` images. If you were using S6 Overlay for our other variations (cli, fpm, etc), you will need to migrate your scripts to use the new `/etc/entrypoint.d` folder.

### `SSL_MODE` is now set to `off` by default (HTTP only)
Running end-to-end SSL by default created more problems than good. By default, we're now shipping HTTP-only by default with the option for people to turn this on.

### `AUTORUN_ENABLED` is now set to `false` by default. 
Having this set to "true" by default also created more problems than good. If you want to use any of the Laravel Automation Scripts, be sure to set this to `true`.

### MSMTP is no longer included in the images
For security and image size reasons, we removed MSMTP from the images. If you need to send emails, use an external SMTP service like Postmark/Sendgrid/Mailgun. You can also extend the image yourself to include MSMTP specifically for your use case.

### Variable deprecations
- `WEB_APP_DIRECTORY` has now been renamed to `APP_BASE_DIR`
- `DEBUG_OUTPUT` has been removed for in favor of `LOG_OUTPUT_LEVEL=debug`
- `PUID` & `PGID` are no longer used because it requires root privileges. See the [new way to set the UID and GID →](/docs/guide/understanding-file-permissions)
- `MSMTP_RELAY_SERVER_HOSTNAME` & `MSMTP_RELAY_SERVER_PORT` are no longer used because MSMTP is no longer included in the images.
- `PHP_POOL_NAME` has been renamed to `PHP_FPM_POOL_NAME`

## 📝 Migration Checklist
Here is a good list to perform the migration

#### Repository
- Ensure you're committing to a test environment

#### Docker Compose
- Update the image name (if applicable)
- Check each environment variable exists and is set to a proper value [See the full list of environment variables →](/docs/reference/environment-variable-specification)
- Ensure you updated the ports to `8080` and `8443` for NGINX, Apache, and Unit
- Consider adding `PHP_OPCACHE_ENABLE=1` to your production environment for increased performance

#### Dockerfile
- Update the base image name (if applicable)
- Remove any `ppa:ondrej/php` references
- Remove any Ubuntu specific commands
- Ensure all extensions are installed with the `install-php-extensions` command [Learn how to install your own PHP extension →](/docs/customizing-the-image/installing-additional-php-extensions)
- Ensure your `COPY` commands are copying with the correct permissions (i.e. `--chown=www-data:www-data`)

#### CI/CD
If you're running `fpm-nginx` (or similar) on a runner that's running as your builds as `root`, you may need to add `user = www-data` and `group = www-data` to your `php-fpm.conf` file so you can bring FPM up correctly.

If you have to run things as root in CI, you can do this with a multi stage build and set the targets:

::code-panel
---
label: Dockerfile
---
```dockerfile
############################################
# Base Image
############################################

# Learn more about the Server Side Up PHP Docker Images at:
# https://serversideup.net/open-source/docker-php/
FROM serversideup/php:8.4-fpm-nginx AS base

## Uncomment if you need to install additional PHP extensions
# USER root
# RUN install-php-extensions bcmath gd

############################################
# Development Image
############################################
FROM base AS development

# We can pass USER_ID and GROUP_ID as build arguments
# to ensure the www-data user has the same UID and GID
# as the user running Docker.
ARG USER_ID
ARG GROUP_ID

# Switch to root so we can set the user ID and group ID
USER root
RUN docker-php-serversideup-set-id www-data $USER_ID:$GROUP_ID  && \
    docker-php-serversideup-set-file-permissions --owner $USER_ID:$GROUP_ID --service nginx
USER www-data

############################################
# CI image
############################################
FROM base AS ci

# Sometimes CI images need to run as root
# so we set the ROOT user and configure
# the PHP-FPM pool to run as www-data
USER root
RUN echo "user = www-data" >> /usr/local/etc/php-fpm.d/docker-php-serversideup-pool.conf && \
    echo "group = www-data" >> /usr/local/etc/php-fpm.d/docker-php-serversideup-pool.conf

############################################
# Production Image
############################################
FROM base AS deploy
COPY --chown=www-data:www-data . /var/www/html
USER www-data
```
::

#### Production/Staging Servers
- Update all host volume file permissions to match the `www-data` UID/GID (`33:33` for Debian, `82:82` for Alpine) [Learn how to manage file permissions](/docs/guide/understanding-file-permissions)
- If you're running Docker Swarm with host volume mounts, we created a script that could potentially help ([change-volume-permissions.sh](https://github.com/serversideup/docker-volume-change-permission-script))

#### Deployment
- CI/CD with valid tests is always encouraged
- After completing all steps above, you're now ready to deploy the new images
