#!/usr/bin/python3
# -*- coding: utf-8 -*-

from chirpotle.dissect import base, v102

import unittest

class V1_0_2CryptoTestSuite(unittest.TestCase):
  """Basic test cases."""

  def test_payload_decryption_uplink(self):
    # This is a message produced by the Arduino LMIC library on the Adafruit LoRa Feather 32u4,
    # using ABP, so the keys and counters are well known.
    # The payload is the string "Hello, world!"
    data = (0x40, 0x01, 0x00, 0xFF, 0x03, 0x80, 0x03, 0x00, 0x01, 0x65,
      0x9E, 0xA5, 0x9C, 0xD0, 0x3A, 0xE3, 0xF7, 0xED, 0xC4, 0xDE,
      0x73, 0x0B, 0x8F, 0x99, 0x94, 0xA8
    )
    appSKey = (0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C)
    fCnt = 3
    msg = v102.LoRaWANMessage_V1_0_2(data = data, session = base.DeviceSession(fCntUp=fCnt, appSKey=appSKey))
    self.assertEqual(msg.mhdr.mType, base.MType.UNCONF_DATA_UP, "MHDR.MType is wrong")
    self.assertEqual("".join(chr(x) for x in msg.payload.frmPayload), "Hello, world!", "Encryption failed")

  def test_payload_valid_mic_validation_uplink(self):
    # Same message as in test_payload_decryption_uplink, but this time we verify the MIC
    data = (0x40, 0x01, 0x00, 0xFF, 0x03, 0x80, 0x03, 0x00, 0x01, 0x65,
      0x9E, 0xA5, 0x9C, 0xD0, 0x3A, 0xE3, 0xF7, 0xED, 0xC4, 0xDE,
      0x73, 0x0B, 0x8F, 0x99, 0x94, 0xA8
    )
    nwkSKey = (0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C)
    fCnt = 3
    msg = v102.LoRaWANMessage_V1_0_2(data = data, session = base.DeviceSession(fCntUp=fCnt, nwkSKey=nwkSKey))
    self.assertEqual(msg.mhdr.mType, base.MType.UNCONF_DATA_UP, "MHDR.MType is wrong")
    self.assertTrue(msg.payload.verifyMIC(), "MIC should be verified but it isn't")

  def test_payload_invalid_mic_validation_uplink(self):
    # Same message as in test_payload_valid_mic_validation_uplink, but the MIC has been altered (last byte of data)
    # so that the validation should fail
    data = (0x40, 0x01, 0x00, 0xFF, 0x03, 0x80, 0x03, 0x00, 0x01, 0x65,
      0x9E, 0xA5, 0x9C, 0xD0, 0x3A, 0xE3, 0xF7, 0xED, 0xC4, 0xDE,
      0x73, 0x0B, 0x8F, 0x99, 0x94, 0x00
    )
    nwkSKey = (0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C)
    fCnt = 3
    msg = v102.LoRaWANMessage_V1_0_2(data = data, session = base.DeviceSession(fCntUp=fCnt, nwkSKey=nwkSKey))
    self.assertEqual(msg.mhdr.mType, base.MType.UNCONF_DATA_UP, "MHDR.MType is wrong")
    self.assertFalse(msg.payload.verifyMIC(), "MIC verification should fail but it succeeded")

  def test_payload_decryption_uplink_multiblock(self):
    # Very similar setup to test_payload_decryption_uplink, but with a longer payload
    data = (0x40, 0x42, 0xEE, 0xFF, 0xC0, 0x80, 0x03, 0x00, 0x01, 0xDE,
      0xF3, 0x7C, 0x75, 0x82, 0x4C, 0x9B, 0x4A, 0x89, 0x24, 0x3A,
      0x49, 0x09, 0xBA, 0x2B, 0x30, 0xA8, 0x65, 0x9B, 0x2F, 0x14,
      0xEF, 0xEE, 0xE0, 0xB5, 0x52, 0x70, 0x20, 0x82, 0x3E, 0x20,
      0x7C, 0x75, 0x04, 0x27, 0x9C, 0x3D, 0x97, 0x25, 0xB6, 0x52,
      0x6D, 0xD0, 0xC0, 0xEA
    )
    appSKey = (0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C)
    fCnt = 3
    msg = v102.LoRaWANMessage_V1_0_2(data = data, session = base.DeviceSession(fCntUp=fCnt, appSKey=appSKey))
    self.assertEqual(msg.mhdr.mType, base.MType.UNCONF_DATA_UP, "MHDR.MType is wrong")
    self.assertEqual(msg.payload.fhdr.devAddr, (0xC0, 0xFF, 0xEE, 0x42), "FHDR.devAddr is wrong")
    self.assertEqual(
      "".join(chr(x) for x in msg.payload.frmPayload),
      "Lorem ipsum dolor sit amet, consetetur...",
      "Encryption failed")

def suite():
  suite = unittest.TestSuite()
  suite.addTests(unittest.makeSuite(V1_0_2CryptoTestSuite))
  return suite

if __name__ == '__main__':
  unittest.main()