# 🌬️ Tailwind integration for Jaspr

[Tailwind](https://tailwindcss.com/) is a popular css framework to quickly compose beautiful websites without needing
to write a lot of css.

This package is a first-class tailwind integration for [jaspr](https://github.com/schultek/jaspr).

## Prerequisites

This package expects tailwind to be installed through the `tailwindcss` command. The recommended way is to use
the [standalone tailwind cli](https://tailwindcss.com/blog/standalone-cli).

To install it, download the executable for your platform from the [latest release](https://github.com/tailwindlabs/tailwindcss/releases/latest)
on GitHub and give it executable permissions:

```shell
curl -sLO https://github.com/tailwindlabs/tailwindcss/releases/latest/download/tailwindcss-<your-platform>
chmod +x tailwindcss-<your-platform>
```

Next move it to a common executable location or add its location to your **PATH**:

```shell
# e.g. /usr/local/bin on unix based systems (linux, macos)
mv tailwindcss-<your-platform> /usr/local/bin/tailwindcss
```

Test your installation by typing in your terminal:

```shell
tailwindcss -h
```

## Setup

To start, add `jaspr_tailwind` as a dev dependency to your project:

`dart pub add jaspr_tailwind --dev`

Next add a `styles.tw.css` file inside the `web` directory with the following content:

```css
@tailwind base;
@tailwind components;
@tailwind utilities;
```

Finally, link the generated `styles.css` in your document, or otherwise add it to your website:

In **static** or **server** mode:

```dart
// This file is lib/main.dart

import 'package:jaspr/server.dart';

import './app.dart';

void main() {
  runApp(Document(
    title: 'My Tailwind Site',
    head: [
      // Link the styles.css file, this will be generated by the tailwind integration.
      link(href: 'styles.css', rel: 'stylesheet'),
    ],
    body: App(),
  ));
}
```

or in **client** mode:

```html title="web/index.html"
<html>
  <head>
    <!-- Link the styles.css file, this will be generated by the tailwind integration.-->
    <link href="styles.css" rel="stylesheet" />
  </head>
</html>
```

You can also have more than one input css file. Any file inside the web directory ending in `.tw.css` will be used and compiled
to its `.css` counterpart.

---

## Usage

If you are unfamiliar with tailwind, head over to their [official documentation](https://tailwindcss.com/docs/utility-first)
first (you can skip the installation part).

The `jaspr_tailwind` integration comes preconfigured, so you can use any tailwind classes inside your jaspr components.

A jaspr card component using tailwind would look like this:

```dart
import 'package:jaspr/jaspr.dart';

class SimpleCard extends StatelessComponent {
  const SimpleCard({required this.title, required this.message});

  final String title;
  final String message;

  @override
  Iterable<Component> build(BuildContext context) sync* {
    yield div(classes: 'p-6 max-w-sm mx-auto bg-white rounded-xl shadow-lg flex items-center space-x-4', [
      div(classes: 'shrink-0', [
        img(classes: 'h-12 w-12', src: '/img/logo.svg', alt: '$title Logo'),
      ]),
      div([
        div(classes: 'text-xl font-medium text-black', [text(title)]),
        p(classes: 'text-slate-500', [text(message)]),
      ])
    ]);
  }
}
```

---

## Config

By default, you don't need a tailwind config file for your project. The package automatically scans the project's Dart files and builds the CSS.
However, if you want to customize the default config like the theme or colors, you can add a `tailwind.config.js` file to the root directory of your project.

When using a custom config, you should explicitly set the `content` option to scan Tailwind class names from the Dart file:

```javascript
/** @type {import('tailwindcss').Config} */
module.exports = {
  content: ['./{lib,web}/**/*.dart'],
  theme: {
    extend: {},
  },
  plugins: [],
}
```

***Note**: Setting a custom content configuration is possible, but the tailwind integration won't recompile the css when those
files change. Automatic recompiling is only enabled for `.dart` files.*
