// Copyright 2017 HootSuite Media Inc.
//
// Licensed under the Apache License, Version 2.0 (the License);
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//    http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an AS IS BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// Modified hereafter by contributors to runatlantis/atlantis.

package common_test

import (
	"strings"
	"testing"

	"github.com/runatlantis/atlantis/server/events/vcs/common"
	. "github.com/runatlantis/atlantis/testing"
)

// If under the maximum number of chars, we shouldn't split the comments.
func TestSplitComment_UnderMax(t *testing.T) {
	comment := "comment under max size"
	split := common.SplitComment(comment, len(comment)+1, "sepEnd", "sepStart", 0, "")
	Equals(t, []string{comment}, split)
}

// If the comment needs to be split into 2 we should do the split and add the
// separators properly.
func TestSplitComment_TwoComments(t *testing.T) {
	comment := strings.Repeat("a", 1000)
	sepEnd := "-sepEnd"
	sepStart := "-sepStart"
	split := common.SplitComment(comment, len(comment)-1, sepEnd, sepStart, 0, "")

	expCommentLen := len(comment) - len(sepEnd) - len(sepStart) - 1
	expFirstComment := comment[:len(comment)-expCommentLen]
	expSecondComment := comment[len(comment)-expCommentLen:]
	Equals(t, 2, len(split))
	Equals(t, expFirstComment+sepEnd, split[0])
	Equals(t, sepStart+expSecondComment, split[1])
}

// If the comment needs to be split into 4 we should do the split and add the
// separators properly.
func TestSplitComment_FourComments(t *testing.T) {
	comment := strings.Repeat("a", 1000)
	sepEnd := "-sepEnd"
	sepStart := "-sepStart"
	max := (len(comment) / 4) + len(sepEnd) + len(sepStart)
	split := common.SplitComment(comment, max, sepEnd, sepStart, 0, "")

	expMax := len(comment) / 4
	Equals(t, []string{
		comment[:len(comment)-expMax*3] + sepEnd,
		sepStart + comment[len(comment)-expMax*3:len(comment)-expMax*2] + sepEnd,
		sepStart + comment[len(comment)-expMax*2:len(comment)-expMax] + sepEnd,
		sepStart + comment[len(comment)-expMax:]}, split)
}

func TestSplitComment_Limited(t *testing.T) {
	comment := strings.Repeat("a", 1000)
	sepEnd := "-sepEnd"
	sepStart := "-sepStart"
	truncationHeader := "truncated-"
	max := (len(comment) / 8) + max(len(sepEnd), len(truncationHeader)) + len(sepStart)
	split := common.SplitComment(comment, max, sepEnd, sepStart, 5, truncationHeader)

	expMax := len(comment) / 8
	Equals(t, []string{
		truncationHeader + comment[len(comment)-expMax*5:len(comment)-expMax*4] + sepEnd,
		sepStart + comment[len(comment)-expMax*4:len(comment)-expMax*3] + sepEnd,
		sepStart + comment[len(comment)-expMax*3:len(comment)-expMax*2] + sepEnd,
		sepStart + comment[len(comment)-expMax*2:len(comment)-expMax] + sepEnd,
		sepStart + comment[len(comment)-expMax:]}, split)
}

func TestAutomergeCommitMsg(t *testing.T) {
	tests := []struct {
		name    string
		pullNum int
		want    string
	}{
		{
			name:    "Atlantis PR commit message should include PR number",
			pullNum: 123,
			want:    "[Atlantis] Automatically merging after successful apply: PR #123",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := common.AutomergeCommitMsg(tt.pullNum); got != tt.want {
				t.Errorf("AutomergeCommitMsg() = %v, want %v", got, tt.want)
			}
		})
	}
}
