/*********************************************************************
 *        _       _         _
 *  _ __ | |_  _ | |  __ _ | |__   ___
 * | '__|| __|(_)| | / _` || '_ \ / __|
 * | |   | |_  _ | || (_| || |_) |\__ \
 * |_|    \__|(_)|_| \__,_||_.__/ |___/
 *
 * www.rt-labs.com
 * Copyright 2018 rt-labs AB, Sweden.
 *
 * This software is dual-licensed under GPLv3 and a commercial
 * license. See the file LICENSE.md distributed with this software for
 * full license information.
 ********************************************************************/

/**
 * @file
 * @brief Integration tests of API functions
 *
 * For example
 *   pnet_output_get_data_and_iops()
 *   pnet_input_get_iocs()
 *   pnet_output_set_iocs()
 *   pnet_create_log_book_entry()
 *   pnet_show()
 */

#include "utils_for_testing.h"
#include "mocks.h"

#include "pf_includes.h"

#include <gtest/gtest.h>

class PnetapiTest : public PnetIntegrationTest
{
};

// clang-format off

static uint8_t connect_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x86, 0x01, 0x00, 0x00, 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x72, 0x01,
 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x00, 0x42, 0x01, 0x00, 0x00, 0x01, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0xde, 0xa0,
 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xf0, 0x00, 0x00, 0x01, 0x40, 0x00,
 0x00, 0x11, 0x02, 0x58, 0x88, 0x92, 0x00, 0x0c, 0x72, 0x74, 0x2d, 0x6c, 0x61, 0x62, 0x73, 0x2d,
 0x64, 0x65, 0x6d, 0x6f, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x01, 0x00, 0x20, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
 0x80, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x05, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x02,
 0x00, 0x02, 0x88, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x00, 0x00, 0x20, 0x00, 0x01,
 0x00, 0x01, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x01, 0x04, 0x00, 0x3c,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x04, 0x00, 0x26,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01,
 0x00, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x16, 0x01, 0x00, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x02, 0x00, 0xc8, 0xc0, 0x00, 0xa0, 0x00
};

static uint8_t release_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x14,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00
};

static uint8_t prm_end_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x10,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00
};

static uint8_t appl_rdy_rsp[] =
{
                                                             0x04, 0x02, 0x0a, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x00, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0x79, 0x56, 0x34, 0x12, 0x34, 0x12, 0x78, 0x56, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
 0x07, 0x08, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0xdc, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x81, 0x12,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00
};

static uint8_t write_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x58, 0x00, 0x00, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x44, 0x00,
 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x08,
 0x00, 0x3c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 0x00, 0x7c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xad, 0xa0,
 0xbe, 0xda
};

static uint8_t data_packet1_bad_iops_bad_iocs[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x01, 0x02, 0x03, 0x04, 0x20, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf3, 0x35, 0x00
};

static uint8_t data_packet2_bad_iops_good_iocs[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x80, 0x80, 0x80, 0x80, 0x21, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4, 0x35, 0x00
};

static uint8_t data_packet3_good_iops_bad_iocs[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x01, 0x02, 0x03, 0x04, 0x22, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf5, 0x35, 0x00
};

static uint8_t data_packet4_good_iops_good_iocs[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x80, 0x80, 0x80, 0x80, 0x23, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x35, 0x00
};

// clang-format on

TEST_F (PnetapiTest, PnetapiRunTest)
{
   int ret;
   pnet_pnio_status_t pnio_status = {1, 2, 3, 4};
   bool new_flag = false;
   uint8_t in_data[10];
   uint16_t in_len = sizeof (in_data);
   uint8_t out_data[] = {
      0x33, /* Slot 1, subslot 1 Data */
   };
   uint8_t iops = PNET_IOXS_BAD;
   uint8_t iocs = PNET_IOXS_BAD;
   uint32_t ix;
   const uint16_t slot = 1;
   const uint16_t subslot = 1;

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock write request\n");
   mock_set_pnal_udp_recvfrom_buffer (write_req, sizeof (write_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nTry receiving data before any received\n");
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, -1);
   EXPECT_EQ (new_flag, false);
   EXPECT_EQ (in_len, 0);
   EXPECT_EQ (iops, PNET_IOXS_BAD);

   TEST_TRACE ("\nTest data with bad IOPS and bad IOCS\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet1_bad_iops_bad_iocs,
         sizeof (data_packet1_bad_iops_bad_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x20);
   EXPECT_EQ (iops, 0x05);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (net, TEST_API_IDENT, slot, subslot, &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, 0x04);

   TEST_TRACE ("\nTest data with bad IOPS and good IOCS\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet2_bad_iops_good_iocs,
         sizeof (data_packet2_bad_iops_good_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x21);
   EXPECT_EQ (iops, 0x05);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (net, TEST_API_IDENT, slot, subslot, &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, PNET_IOXS_GOOD);

   TEST_TRACE ("\nTest data with good IOPS and bad IOCS\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet3_good_iops_bad_iocs,
         sizeof (data_packet3_good_iops_bad_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x22);
   EXPECT_EQ (iops, PNET_IOXS_GOOD);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (net, TEST_API_IDENT, slot, subslot, &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, 0x04);

   TEST_TRACE ("\nTest data with good IOPS and good IOCS\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet4_good_iops_good_iocs,
         sizeof (data_packet4_good_iops_good_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x23);
   EXPECT_EQ (iops, PNET_IOXS_GOOD);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (net, TEST_API_IDENT, slot, subslot, &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, PNET_IOXS_GOOD);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   TEST_TRACE ("\nRead more data when no new data received\n");
   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   in_data[0] = 0;
   new_flag = (bool)0x33;
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, false);
   EXPECT_EQ (in_len, 1);            /* Received one byte (same as before) */
   EXPECT_EQ (in_data[0], 0x23);     /* Same as before */
   EXPECT_EQ (iops, PNET_IOXS_GOOD); /* Same as before */

   TEST_TRACE ("\nSend some data to the controller\n");
   TEST_TRACE ("Line %d\n", __LINE__);
   ret = pnet_input_set_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      out_data,
      sizeof (out_data),
      PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);

   TEST_TRACE ("\nAcknowledge the reception of controller data\n");
   ret =
      pnet_output_set_iocs (net, TEST_API_IDENT, slot, subslot, PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   TEST_TRACE ("\nCreate a logbook entry\n");
   pnet_create_log_book_entry (net, appdata.main_arep, &pnio_status, 0x13245768);

   TEST_TRACE ("\nGenerating mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}

TEST_F (PnetapiTest, PnetapiShowTest)
{
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   pnet_show (net, 0x7fffffff);

   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);

   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}
