/*********************************************************************
 *        _       _         _
 *  _ __ | |_  _ | |  __ _ | |__   ___
 * | '__|| __|(_)| | / _` || '_ \ / __|
 * | |   | |_  _ | || (_| || |_) |\__ \
 * |_|    \__|(_)|_| \__,_||_.__/ |___/
 *
 * www.rt-labs.com
 * Copyright 2018 rt-labs AB, Sweden.
 *
 * This software is dual-licensed under GPLv3 and a commercial
 * license. See the file LICENSE.md distributed with this software for
 * full license information.
 ********************************************************************/

/**
 * @file
 * @brief Integration testing of CMRPC.
 *
 * For example
 *  Connect
 *  Read IM0
 *  Release connection
 *  Timeout in CPM
 *  Fragmented connect
 *  ConnectReleaseIOSAR_DA ?
 */

#include "utils_for_testing.h"
#include "mocks.h"

#include "pf_includes.h"

#include <gtest/gtest.h>

class CmrpcUnitTest : public PnetUnitTest
{
};
class CmrpcTest : public PnetIntegrationTest
{
};

// clang-format off

/**
 * Connect request data
 * This is UDP payload, so the first 42 bytes (in dec) have been removed from
 * the Wireshark output.
 */
static uint8_t connect_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x86, 0x01, 0x00, 0x00, 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x72, 0x01,
 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x00, 0x42, 0x01, 0x00, 0x00, 0x01, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0xde, 0xa0,
 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xf0, 0x00, 0x00, 0x01, 0x40, 0x00,
 0x00, 0x11, 0x02, 0x58, 0x88, 0x92, 0x00, 0x0c, 0x72, 0x74, 0x2d, 0x6c, 0x61, 0x62, 0x73, 0x2d,
 0x64, 0x65, 0x6d, 0x6f, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x01, 0x00, 0x20, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
 0x80, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x05, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x02,
 0x00, 0x02, 0x88, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x00, 0x00, 0x20, 0x00, 0x01,
 0x00, 0x01, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x01, 0x04, 0x00, 0x3c,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x04, 0x00, 0x26,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01,
 0x00, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x16, 0x01, 0x00, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x02, 0x00, 0xc8, 0xc0, 0x00, 0xa0, 0x00
};

static uint8_t release_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x14,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00
};

static uint8_t prm_end_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x10,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00
};

static uint8_t appl_rdy_rsp[] =
{
                                                             0x04, 0x02, 0x0a, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x00, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0x79, 0x56, 0x34, 0x12, 0x34, 0x12, 0x78, 0x56, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
 0x07, 0x08, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0xdc, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x81, 0x12,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00
};

static uint8_t write_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x58, 0x00, 0x00, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x44, 0x00,
 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x08,
 0x00, 0x3c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 0x00, 0x7c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xad, 0xa0,
 0xbe, 0xda
};

static uint8_t read_im0_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x00, 0x00,
 0x00, 0x00, 0xde, 0xa0, 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0xde, 0xa0, 0x00, 0x01, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x02,
 0xff, 0xff, 0xff, 0xff, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x40, 0x00, 0x00,
 0x00, 0x40, 0x00, 0x00, 0x10, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x09,
 0x00, 0x3c, 0x01, 0x00, 0x00, 0x02, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
 0xaf, 0xf0, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/*
 * In order to create a fragmented connect using the connect request above:
 * 1) Copy the connect request to both fragments.
 * 2) Strip away the ETH header, the IP header and the UDP header (0x2a bytes) from both packets (already done!!!)
 * 3) In fragment 1 change byte [0x2c] from 0x28 to 0x2c.   (means fragment)
 * 4) In fragment 2 change byte [0x2c] from 0x28 to 0x2e.   (means last fragment)
 * 5) Modify fragment length member of both frames:
 *    a) Fragment 1: bytes [0x74..0x75] from {0x86 0x01} to [0x00 0x01] (means 0x100 bytes).
 *    b) Fragment 2: bytes [0x74..0x75] from {0x86 0x01} to [0x86 0x00] (means 0x86  bytes).
 * 6) Modify fragment number of fragment 2:
 *    Fragment 2: bytes [0x76..0x77] from {0x00 0x00} to {0x01 0x00} (means fragment 1).
 * 7) Modify serial_low member of fragment 2:
 *    Fragment 2: byte [0x79] from 0x00 to 0x01.
 * 7) Cut fragment 1 at 256 bytes of connect request (i.e., at position 0x16a).
 * 8) Remove the first 256 bytes of the connect request from fragment 2 (i.e. bytes 0x7a to 0x17a)
 */
static uint8_t connect_frag_1_req[] =
{
 /*  20 */                                                    /* 2a */0x04, 0x00, 0x2c, 0x00, 0x10, 0x00,
 /*  30 */0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 /*  40 */0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 /*  50 */0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 /*  60 */0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 /*  70 */0xff, 0xff, 0xff, 0xff, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x72, 0x01,
 /*  80 */0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x01, 0x01,
 /*  90 */0x00, 0x42, 0x01, 0x00, 0x00, 0x01, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 /*  a0 */0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0xde, 0xa0,
 /*  b0 */0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xf0, 0x00, 0x00, 0x01, 0x40, 0x00,
 /*  c0 */0x00, 0x11, 0x02, 0x58, 0x88, 0x92, 0x00, 0x0c, 0x72, 0x74, 0x2d, 0x6c, 0x61, 0x62, 0x73, 0x2d,
 /*  d0 */0x64, 0x65, 0x6d, 0x6f, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x88, 0x92,
 /*  e0 */0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x01, 0x00, 0x20, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 /*  f0 */0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 /* 100 */0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
 /* 110 */0x80, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03,
 /* 120 */0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x05, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x02,
 /* 130 */0x00, 0x02, 0x88, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x00, 0x00, 0x20, 0x00, 0x01,
 /* 140 */0x00, 0x01, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00,
 /* 150 */0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
 /* 160 */0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x01,
 /* 170 */0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01
};

static uint8_t connect_frag_2_req[] =
{
 /*  20 */                                                    /* 2a */0x04, 0x00, 0x2e, 0x00, 0x10, 0x00,
 /*  30 */0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 /*  40 */0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 /*  50 */0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 /*  60 */0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 /*  70 */0xff, 0xff, 0xff, 0xff, 0x86, 0x00, 0x01, 0x00, 0x00, 0x01,
 /*  80 */
 /*  90 */
 /*  a0 */
 /*  b0 */
 /*  c0 */
 /*  d0 */
 /*  e0 */
 /*  f0 */
 /* 100 */
 /* 110 */
 /* 120 */
 /* 130 */
 /* 140 */
 /* 150 */
 /* 160 */
 /* 170 */                                                            0x00, 0x03, 0x01, 0x04, 0x00, 0x3c,
 /* 180 */0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
 /* 190 */0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01,
 /* 1a0 */0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x01,
 /* 1b0 */0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x04, 0x00, 0x26,
 /* 1c0 */0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00,
 /* 1d0 */0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01,
 /* 1e0 */0x00, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x16, 0x01, 0x00, 0x00, 0x01, 0x88, 0x92,
 /* 1f0 */0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x02, 0x00, 0xc8, 0xc0, 0x00, 0xa0, 0x00
};

static uint8_t connect_req_iosar_da[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x42, 0x00,
 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x01, 0x01,
 0x00, 0x3e, 0x01, 0x00, 0x00, 0x06, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0xde, 0xa0,
 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xf0, 0x00, 0x00, 0x01, 0x40, 0x00,
 0x01, 0x11, 0x02, 0x58, 0x88, 0x92, 0x00, 0x08, 0x72, 0x74, 0x2d, 0x6c, 0x61, 0x62, 0x73, 0x31
};

static uint8_t release_req_iosar_da[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x14,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00
};

static uint8_t connect_req_siemens[] =
{
                                                            0x04, 0x00, 0x20, 0x00, 0x00, 0x00,
0x00, 0x00, 0xde, 0xa0, 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
0xfe, 0xed, 0xde, 0xa0, 0x00, 0x01, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
0xdf, 0x7d, 0x0a, 0x07, 0x10, 0x32, 0x1b, 0xd0, 0x1f, 0x1f, 0x8a, 0x8c, 0xe0, 0xdc, 0xa0, 0x8a,
0xba, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0xff, 0xff, 0xff, 0xff, 0x01, 0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
0x01, 0x70, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x70, 0x01, 0x01,
0x00, 0x40, 0x01, 0x00, 0x00, 0x01, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8,
0xcc, 0x5f, 0x1c, 0x76, 0x93, 0xf8, 0x00, 0x01, 0xe0, 0xdc, 0xa0, 0x8a, 0xba, 0x33, 0xde, 0xa0,
0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x64, 0x01, 0x0d, 0x00, 0x2a, 0x40, 0x00,
0x00, 0x11, 0x00, 0xc8, 0x88, 0x92, 0x00, 0x0a, 0x70, 0x6c, 0x63, 0x78, 0x62, 0x31, 0x64, 0x30,
0x65, 0x64, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x88, 0x92, 0x00, 0x00,
0x00, 0x02, 0x00, 0x28, 0x80, 0x00, 0x00, 0x20, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0xff, 0xff,
0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01,
0x00, 0x01, 0x00, 0x01, 0x00, 0x05, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x02, 0x00, 0x02,
0x88, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0xff, 0xff, 0x00, 0x20, 0x00, 0x02, 0x00, 0x01,
0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x04,
0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x01, 0x00, 0x00,
0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x01, 0x04, 0x00, 0x3c, 0x01, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03,
0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x00,
0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x01, 0x00, 0x00,
0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x04, 0x00, 0x26, 0x01, 0x00,
0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x01,
0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 0x02,
0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x16, 0x01, 0x00, 0x00, 0x01, 0x88, 0x92, 0x00, 0x00,
0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x00, 0xa0, 0x00
};

static uint8_t write_req_siemens[] =
{
                                                            0x04, 0x00, 0x20, 0x00, 0x00, 0x00,
0x00, 0x00, 0xde, 0xa0, 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
0xfe, 0xed, 0xde, 0xa0, 0x00, 0x01, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
0xdf, 0x7d, 0x0a, 0x07, 0x10, 0x32, 0x1b, 0xd0, 0x1f, 0x1f, 0x8a, 0x8c, 0xe0, 0xdc, 0xa0, 0x8a,
0xba, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03,
0xff, 0xff, 0xff, 0xff, 0x01, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00,
0x01, 0x14, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x08,
0x00, 0x3c, 0x01, 0x00, 0x00, 0x00, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8,
0xcc, 0x5f, 0x1c, 0x76, 0x93, 0xf8, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
0xe0, 0x40, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,
0x00, 0x3c, 0x01, 0x00, 0x00, 0x01, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8,
0xcc, 0x5f, 0x1c, 0x76, 0x93, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
0x80, 0x71, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x50,
0x00, 0x08, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x3c, 0x01, 0x00,
0x00, 0x02, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8, 0xcc, 0x5f, 0x1c, 0x76,
0x93, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x7b, 0x00, 0x00,
0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,
0x00, 0x3c, 0x01, 0x00, 0x00, 0x03, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8,
0xcc, 0x5f, 0x1c, 0x76, 0x93, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0x7c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00
};

static uint8_t prm_end_req_siemens[] =
{
                                                            0x04, 0x00, 0x20, 0x00, 0x00, 0x00,
0x00, 0x00, 0xde, 0xa0, 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
0xfe, 0xed, 0xde, 0xa0, 0x00, 0x01, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
0xdf, 0x7d, 0x0a, 0x07, 0x10, 0x32, 0x1b, 0xd0, 0x1f, 0x1f, 0x8a, 0x8c, 0xe0, 0xdc, 0xa0, 0x8a,
0xba, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04,
0xff, 0xff, 0xff, 0xff, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x01, 0x10,
0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0xf1, 0x07, 0x05, 0x4c, 0x1e, 0x41, 0x43, 0x4f, 0x8b, 0xc8,
0xcc, 0x5f, 0x1c, 0x76, 0x93, 0xf8, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00
};

/* This is a raw Ethernet packet (not UDP payload) */
static uint8_t data_packet[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x80, 0x80, 0x80, 0x20, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf3, 0x35, 0x00
};

// clang-format on

TEST_F (CmrpcTest, CmrpcConnectReleaseTest)
{
   int ret;
   uint32_t ix;
   EXPECT_EQ (mock_os_data.udp_sendto_len, 0);

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_GT (mock_os_data.eth_send_count, 0);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 1);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 178);

   TEST_TRACE ("\nGenerating mock write request\n");
   mock_set_pnal_udp_recvfrom_buffer (write_req, sizeof (write_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.write_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 2);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 164);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.call_counters.write_calls, 1);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 3);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 132);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 4);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 132);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 4);

   TEST_TRACE ("\nGenerating I&M0 request\n");
   mock_set_pnal_udp_recvfrom_buffer (read_im0_req, sizeof (read_im0_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 5);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 224);

   TEST_TRACE ("\nGenerating cyclic data\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (data_packet, sizeof (data_packet));
      run_stack (TEST_DATA_DELAY);
   }
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 5);

   TEST_TRACE ("\nGenerating mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 6);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 132);
}

TEST_F (CmrpcTest, CmrpcConnectionTimeoutTest)
{
   int ret;
   bool new_flag = false;
   uint8_t in_data[10];
   uint16_t in_len = sizeof (in_data);
   uint8_t out_data[] = {
      0x33, /* Slot 1, subslot 1 Data */
   };
   uint8_t iops = PNET_IOXS_BAD;
   uint32_t ix;

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock write request\n");
   mock_set_pnal_udp_recvfrom_buffer (write_req, sizeof (write_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerate a couple of data packets to move pf_cpm to RUN "
               "state\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (data_packet, sizeof (data_packet));
      run_stack (TEST_DATA_DELAY);
   }
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   TEST_TRACE ("\nRead data just to see that it works\n");
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      1,
      1,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iops, PNET_IOXS_GOOD);

   TEST_TRACE ("\nGenerate new data to move PPM to RUN state\n");
   ret = pnet_input_set_data_and_iops (
      net,
      TEST_API_IDENT,
      1,
      1,
      out_data,
      sizeof (out_data),
      PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);

   TEST_TRACE ("\nWait for timeout in CPM\n");
   run_stack (TEST_TIMEOUT_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);

   TEST_TRACE ("\nGenerating mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}

/**
 *  Replicate startup with a Siemens S7-1200 controller
 */
TEST_F (CmrpcTest, CmrpcSiemensConnectTest)
{
   EXPECT_EQ (mock_os_data.udp_sendto_len, 0);

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (
      connect_req_siemens,
      sizeof (connect_req_siemens));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.call_counters.write_calls, 0);
   EXPECT_EQ (appdata.call_counters.read_calls, 0);
   EXPECT_EQ (appdata.call_counters.ccontrol_calls, 0);
   EXPECT_EQ (appdata.call_counters.dcontrol_calls, 0);
   EXPECT_GT (mock_os_data.eth_send_count, 0);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 1);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 178); // Actual value. OK in
                                                 // Wireshark (232 bytes)

   TEST_TRACE ("\nGenerating mock write request\n");
   mock_set_pnal_udp_recvfrom_buffer (
      write_req_siemens,
      sizeof (write_req_siemens));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.call_counters.write_calls, 2);
   EXPECT_EQ (appdata.call_counters.read_calls, 0);
   EXPECT_EQ (appdata.call_counters.ccontrol_calls, 0);
   EXPECT_EQ (appdata.call_counters.dcontrol_calls, 0);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 2);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 356); // Actual value. OK in
                                                 // Wireshark (398 bytes)

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (
      prm_end_req_siemens,
      sizeof (prm_end_req_siemens));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.call_counters.write_calls, 2);
   EXPECT_EQ (appdata.call_counters.read_calls, 0);
   EXPECT_EQ (appdata.call_counters.ccontrol_calls, 0);
   EXPECT_EQ (appdata.call_counters.dcontrol_calls, 1);
   EXPECT_EQ (mock_os_data.udp_sendto_count, 3);
   EXPECT_EQ (mock_os_data.udp_sendto_len, 132); // 20 bytes longer than sent
                                                 // from faulty application
}

TEST_F (CmrpcTest, CmrpcConnectFragmentTest)
{
   int ret;
   uint32_t ix;

   TEST_TRACE ("\nGenerating mock connection request, fragment 1\n");
   mock_set_pnal_udp_recvfrom_buffer (
      connect_frag_1_req,
      sizeof (connect_frag_1_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 0);
   EXPECT_EQ (appdata.call_counters.connect_calls, 0);
   EXPECT_EQ (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock connection request, fragment 2\n");
   mock_set_pnal_udp_recvfrom_buffer (
      connect_frag_2_req,
      sizeof (connect_frag_2_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock connection write request\n");
   mock_set_pnal_udp_recvfrom_buffer (write_req, sizeof (write_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_EQ (appdata.call_counters.write_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nRead I&M0\n");
   mock_set_pnal_udp_recvfrom_buffer (read_im0_req, sizeof (read_im0_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nSending data\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (data_packet, sizeof (data_packet));
      run_stack (TEST_DATA_DELAY);
   }
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   TEST_TRACE ("Sending mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}

TEST_F (CmrpcTest, CmrpcConnectReleaseIOSAR_DA)
{
   // Device-access AR is not yet supported
   TEST_TRACE ("\nGenerating mock connection request IOSAR_DA\n");
   mock_set_pnal_udp_recvfrom_buffer (
      connect_req_iosar_da,
      sizeof (connect_req_iosar_da));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 0);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
   EXPECT_EQ (appdata.call_counters.connect_calls, 0);
   EXPECT_EQ (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock release request IOSAR_DA\n");
   mock_set_pnal_udp_recvfrom_buffer (
      release_req_iosar_da,
      sizeof (release_req_iosar_da));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 0);
   EXPECT_EQ (appdata.call_counters.release_calls, 0);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}

TEST_F (CmrpcUnitTest, CmrpcCheckGenerateUuid)
{
   uint32_t timestamp;
   uint32_t session_number;
   pnet_ethaddr_t mac_address;
   pf_uuid_t uuid;

   timestamp = 0xC1C2C3C4;
   session_number = 0xB1B2B3B4;
   mac_address.addr[0] = 0xA1;
   mac_address.addr[1] = 0xA2;
   mac_address.addr[2] = 0xA3;
   mac_address.addr[3] = 0xA4;
   mac_address.addr[4] = 0xA5;
   mac_address.addr[5] = 0xA6;

   pf_generate_uuid (timestamp, session_number, mac_address, &uuid);

   EXPECT_EQ (uuid.data1, timestamp);
   EXPECT_EQ (uuid.data2, 0xB1B2);
   EXPECT_EQ (uuid.data3, 0x10B3);
   EXPECT_EQ (uuid.data4[0], 0x80);
   EXPECT_EQ (uuid.data4[1], 0xB4);
   EXPECT_EQ (uuid.data4[2], 0xA1);
   EXPECT_EQ (uuid.data4[3], 0xA2);
   EXPECT_EQ (uuid.data4[4], 0xA3);
   EXPECT_EQ (uuid.data4[5], 0xA4);
   EXPECT_EQ (uuid.data4[6], 0xA5);
   EXPECT_EQ (uuid.data4[7], 0xA6);
}
