/*********************************************************************
 *        _       _         _
 *  _ __ | |_  _ | |  __ _ | |__   ___
 * | '__|| __|(_)| | / _` || '_ \ / __|
 * | |   | |_  _ | || (_| || |_) |\__ \
 * |_|    \__|(_)|_| \__,_||_.__/ |___/
 *
 * www.rt-labs.com
 * Copyright 2018 rt-labs AB, Sweden.
 *
 * This software is dual-licensed under GPLv3 and a commercial
 * license. See the file LICENSE.md distributed with this software for
 * full license information.
 ********************************************************************/

/**
 * @file
 * @brief Integration testing of CMRDR.
 *
 * Using for example:
 *  pnet_application_ready()
 *  pnet_output_get_data_and_iops()
 *  pnet_input_get_iocs()
 *  pnet_input_set_data_and_iops()
 *  pnet_output_set_iocs()
 *  pnet_create_log_book_entry()
 *  pnet_diag_add()
 *
 */

#include "utils_for_testing.h"
#include "mocks.h"

#include "pf_includes.h"

#include <gtest/gtest.h>

// clang-format off

static uint8_t connect_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x86, 0x01, 0x00, 0x00, 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x72, 0x01,
 0x00, 0x00, 0x24, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x00, 0x42, 0x01, 0x00, 0x00, 0x01, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0xde, 0xa0,
 0x00, 0x00, 0x6c, 0x97, 0x11, 0xd1, 0x82, 0x71, 0x00, 0x01, 0xf0, 0x00, 0x00, 0x01, 0x40, 0x00,
 0x00, 0x11, 0x02, 0x58, 0x88, 0x92, 0x00, 0x0c, 0x72, 0x74, 0x2d, 0x6c, 0x61, 0x62, 0x73, 0x2d,
 0x64, 0x65, 0x6d, 0x6f, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x01, 0x00, 0x20, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
 0x80, 0x00, 0x00, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x05, 0x01, 0x02, 0x00, 0x50, 0x01, 0x00, 0x00, 0x02,
 0x00, 0x02, 0x88, 0x92, 0x00, 0x00, 0x00, 0x02, 0x00, 0x28, 0x80, 0x00, 0x00, 0x20, 0x00, 0x01,
 0x00, 0x01, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x03, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x01, 0x00, 0x03, 0x01, 0x04, 0x00, 0x3c,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01,
 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x80, 0x01,
 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x04, 0x00, 0x26,
 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01,
 0x00, 0x02, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x16, 0x01, 0x00, 0x00, 0x01, 0x88, 0x92,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x02, 0x00, 0xc8, 0xc0, 0x00, 0xa0, 0x00
};

static uint8_t release_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x14,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00
};

static uint8_t prm_end_req[] =
{
                                                             0x04, 0x00, 0x28, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x01, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0xbb, 0xac, 0x97, 0xe2, 0x76, 0x54, 0x9f, 0x47, 0xa5, 0xbd, 0xa5, 0xe3, 0x7d, 0x98,
 0xe5, 0xda, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x10,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00
};

static uint8_t appl_rdy_rsp[] =
{
                                                             0x04, 0x02, 0x0a, 0x00, 0x10, 0x00,
 0x00, 0x00, 0x00, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0x00, 0xbe, 0xef,
 0xfe, 0xed, 0x01, 0x00, 0xa0, 0xde, 0x97, 0x6c, 0xd1, 0x11, 0x82, 0x71, 0x00, 0xa0, 0x24, 0x42,
 0xdf, 0x7d, 0x79, 0x56, 0x34, 0x12, 0x34, 0x12, 0x78, 0x56, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
 0x07, 0x08, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
 0xff, 0xff, 0xff, 0xff, 0x34, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
 0x00, 0x00, 0xdc, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x81, 0x12,
 0x00, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x30, 0xab, 0xa9, 0xa3, 0xf7, 0x64, 0xb7, 0x44, 0xb3, 0xb6,
 0x7e, 0xe2, 0x8a, 0x1a, 0x02, 0xcb, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00
};

static uint8_t data_packet_good_iops_good_iocs[] =
{
 0x1e, 0x30, 0x6c, 0xa2, 0x45, 0x5e, 0xc8, 0x5b, 0x76, 0xe6, 0x89, 0xdf, 0x88, 0x92, 0x80, 0x00,
 0x80, 0x80, 0x80, 0x80, 0x23, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0x35, 0x00
};

static uint16_t test_mod_diff[] =
{
   0xe002
};

static uint16_t test_reads[] =
{
    0x8000, 0x8001,
    0x800a, 0x800b, 0x800c,
    0x8010, 0x8011, 0x8012, 0x8013,
    0x801e,
    0x8020,
    0x8027, 0x8028, 0x8029,
    0x802a, 0x802b,
    0x802c, 0x802d,
    0x802f,
    0x8030, 0x8031,
    0x8050, 0x8051, 0x8052, 0x8053, 0x8054, 0x8055, 0x8056, 0x8057,
    0x8060, 0x8061, 0x8062,
    0x8070, 0x8071, 0x8072,
    0x8080,
    0x8090,
    0x80af,
    0x80b0,
    0x80cf,
    0xaff0, 0xaff1, 0xaff2, 0xaff3, 0xaff4, 0xaff5, 0xaff6, 0xaff7, 0xaff8, 0xaff9, 0xaffa, 0xaffb, 0xaffc, 0xaffd, 0xaffe, 0xafff,
    0xc000, 0xc001,
    0xc00a, 0xc00b, 0xc00c,
    0xc010, 0xc011, 0xc012, 0xc013,
    0xe000, 0xe001, 0xe002,
    0xe00a, 0xe00b, 0xe00c,
    0xe010, 0xe011, 0xe012, 0xe013,
    0xe030, 0xe031,
    0xe040,
    0xe050,
    0xe060, 0xe061,
    0xf000,
    0xf00a, 0xf00b, 0xf00c,
    0xf010, 0xf011, 0xf012, 0xf013,
    0xf020,
    0xf80c,
    0xf820, 0xf821,
    0xf830, 0xf831,
    0xf840, 0xf841, 0xf842,
    0xf850,
    0xf860,
    0xf870, 0xf871,
    0xf880, 0xf881, 0xf882, 0xf883, 0xf884, 0xf885, 0xf886, 0xf887, 0xf888, 0xf889,
    0xfbff,
};

// clang-format on

static pf_iod_read_request_t read_request;
static pnet_result_t read_status;
static uint16_t seq_nbr = 0;

class CmrdrTest : public PnetIntegrationTest
{
 protected:
   void test_read (uint16_t idx)
   {
      pf_ar_t * p_ar;
      uint8_t buffer[PF_FRAME_BUFFER_SIZE];
      uint16_t pos = 0;

      /* Send data to prevent timeout */
      send_data (
         data_packet_good_iops_good_iocs,
         sizeof (data_packet_good_iops_good_iocs));
      run_stack (TEST_DATA_DELAY);

      memset (&read_status, 0, sizeof (read_status));
      memset (&read_request, 0, sizeof (read_request));
      pf_ar_find_by_arep (net, appdata.main_arep, &p_ar);

      read_request.sequence_number = seq_nbr++;
      /* read_request.ar_uuid = NIL */
      read_request.api = 0;
      read_request.slot_number = 1;
      read_request.subslot_number = 1;
      read_request.index = idx;
      read_request.record_data_length = 0;
      // read_request.target_ar_uuid;   /* Only used if implicit AR */

      pf_cmrdr_rm_read_ind (
         net,
         p_ar,
         &read_request,
         &read_status,
         sizeof (buffer),
         buffer,
         &pos);

      if (read_status.pnio_status.error_code != PNET_ERROR_CODE_NOERROR)
      {
         TEST_TRACE ("Read failed for idx %#x\n", (unsigned)idx);
         appdata.read_fails++;
      }
   }
};

TEST_F (CmrdrTest, CmrdrRunTest)
{
   int ret;
   pnet_pnio_status_t pnio_status = {1, 2, 3, 4};
   bool new_flag = false;
   uint8_t in_data[10];
   uint16_t in_len = sizeof (in_data);
   uint8_t out_data[] = {
      0x33, /* Slot 1, subslot 1 Data */
   };
   uint8_t iops = PNET_IOXS_BAD;
   uint8_t iocs = PNET_IOXS_BAD;
   uint32_t ix;
   pnet_diag_source_t diag_source = {
      .api = TEST_API_IDENT,
      .slot = TEST_SLOT_IDENT,
      .subslot = TEST_SUBSLOT_IDENT,
      .ch = TEST_CHANNEL_IDENT,
      .ch_grouping = PNET_DIAG_CH_INDIVIDUAL_CHANNEL,
      .ch_direction = TEST_CHANNEL_DIRECTION};

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   TEST_TRACE ("Line %d\n", __LINE__);
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nSend a couple of data packets and verify reception\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet_good_iops_good_iocs,
         sizeof (data_packet_good_iops_good_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      TEST_SLOT_IDENT,
      TEST_SUBSLOT_IDENT,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x23);
   EXPECT_EQ (iops, PNET_IOXS_GOOD);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (
      net,
      TEST_API_IDENT,
      TEST_SLOT_IDENT,
      TEST_SUBSLOT_IDENT,
      &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, PNET_IOXS_GOOD);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   TEST_TRACE ("\nSend some data to the controller\n");
   ret = pnet_input_set_data_and_iops (
      net,
      TEST_API_IDENT,
      TEST_SLOT_IDENT,
      TEST_SUBSLOT_IDENT,
      out_data,
      sizeof (out_data),
      PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);

   TEST_TRACE ("\nAcknowledge the reception of controller data\n");
   ret = pnet_output_set_iocs (
      net,
      TEST_API_IDENT,
      TEST_SLOT_IDENT,
      TEST_SUBSLOT_IDENT,
      PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   /* Send data to avoid timeout */
   send_data (
      data_packet_good_iops_good_iocs,
      sizeof (data_packet_good_iops_good_iocs));
   run_stack (TEST_DATA_DELAY);

   TEST_TRACE ("\nCreate a logbook entry\n");
   pnet_create_log_book_entry (net, appdata.main_arep, &pnio_status, 0x13245768);

   TEST_TRACE ("\nCreate a diag and an alarm.\n");
   pnet_diag_std_add (
      net,
      &diag_source,
      TEST_CHANNEL_NUMBER_OF_BITS,
      PNET_DIAG_CH_PROP_MAINT_FAULT,
      TEST_CHANNEL_ERRORTYPE,
      TEST_DIAG_EXT_ERRTYPE,
      TEST_DIAG_EXT_ADDVALUE,
      TEST_DIAG_QUALIFIER_NOTSET);

   TEST_TRACE ("Number of tests: %u\n", (unsigned)NELEMENTS (test_reads));

   TEST_TRACE ("\nNow read all the records\n");
   for (ix = 0; ix < NELEMENTS (test_reads); ix++)
   {
      test_read (test_reads[ix]);
   }
   EXPECT_EQ (appdata.read_fails, 61); // Currently expected number of fails.

   TEST_TRACE ("\nGenerating mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}

TEST_F (CmrdrTest, CmrdrModDiffTest)
{
   int ret;
   pnet_pnio_status_t pnio_status = {1, 2, 3, 4};
   bool new_flag = false;
   uint8_t in_data[10];
   uint16_t in_len = sizeof (in_data);
   uint8_t out_data[] = {
      0x33, /* Slot 1, subslot 1 Data */
   };
   uint8_t iops = PNET_IOXS_BAD;
   uint8_t iocs = PNET_IOXS_BAD;
   uint32_t ix;
   const uint16_t slot = 1;
   const uint16_t subslot = 1;

   // TODO  This test used to have an offset for the submodule identifier value
   // in the on_submodule callback. However it does not make any difference
   // for the test result

   TEST_TRACE ("\nGenerating mock connection request\n");
   mock_set_pnal_udp_recvfrom_buffer (connect_req, sizeof (connect_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 1);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_STARTUP);
   EXPECT_GT (mock_os_data.eth_send_count, 0);

   TEST_TRACE ("\nGenerating mock parameter end request\n");
   mock_set_pnal_udp_recvfrom_buffer (prm_end_req, sizeof (prm_end_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 2);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_PRMEND);
   EXPECT_EQ (appdata.call_counters.connect_calls, 1);

   TEST_TRACE ("\nSimulate application calling APPL_RDY\n");
   ret = pnet_application_ready (net, appdata.main_arep);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nGenerating mock application ready response\n");
   mock_set_pnal_udp_recvfrom_buffer (appl_rdy_rsp, sizeof (appl_rdy_rsp));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.state_calls, 3);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_APPLRDY);

   TEST_TRACE ("\nSend a couple of data packets and verify reception\n");
   for (ix = 0; ix < 100; ix++)
   {
      send_data (
         data_packet_good_iops_good_iocs,
         sizeof (data_packet_good_iops_good_iocs));
      run_stack (TEST_DATA_DELAY);
   }

   iops = 88; /* Something non-valid */
   in_len = sizeof (in_data);
   ret = pnet_output_get_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      &new_flag,
      in_data,
      &in_len,
      &iops);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (in_data[0], 0x23);
   EXPECT_EQ (iops, PNET_IOXS_GOOD);

   iocs = 77; /* Something non-valid */
   ret = pnet_input_get_iocs (net, TEST_API_IDENT, slot, subslot, &iocs);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (new_flag, true);
   EXPECT_EQ (in_len, 1);
   EXPECT_EQ (iocs, PNET_IOXS_GOOD);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   /* Send some data to the controller */
   TEST_TRACE ("Line %d\n", __LINE__);
   ret = pnet_input_set_data_and_iops (
      net,
      TEST_API_IDENT,
      slot,
      subslot,
      out_data,
      sizeof (out_data),
      PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);

   /* Acknowledge the reception of controller data */
   ret =
      pnet_output_set_iocs (net, TEST_API_IDENT, slot, subslot, PNET_IOXS_GOOD);
   EXPECT_EQ (ret, 0);
   EXPECT_EQ (appdata.call_counters.state_calls, 4);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_DATA);

   /* Setup some record for the reader */

   TEST_TRACE ("Line %d\n", __LINE__);
   /* Create a logbook entry */
   pnet_create_log_book_entry (net, appdata.main_arep, &pnio_status, 0x13245768);

   TEST_TRACE ("Number of tests: %u\n", (unsigned)NELEMENTS (test_mod_diff));

   /* Now read all the mod diff record */
   for (ix = 0; ix < NELEMENTS (test_mod_diff); ix++)
   {
      test_read (test_mod_diff[ix]);
   }
   EXPECT_EQ (appdata.read_fails, 0);

   TEST_TRACE ("\nGenerating mock release request\n");
   mock_set_pnal_udp_recvfrom_buffer (release_req, sizeof (release_req));
   run_stack (TEST_UDP_DELAY);
   EXPECT_EQ (appdata.call_counters.release_calls, 1);
   EXPECT_EQ (appdata.call_counters.state_calls, 5);
   EXPECT_EQ (appdata.cmdev_state, PNET_EVENT_ABORT);
}
