# frozen_string_literal: true

require 'parallel'

::RSpec.describe ::Leftovers::CLI, type: :cli do
  describe 'leftovers', :with_temp_dir do
    before do
      allow(::Leftovers).to receive(:try_require_cache).and_call_original
      allow(::Leftovers).to receive(:try_require_cache).with('bundler').and_return(false)
    end

    context 'with no files' do
      it 'runs' do
        expect { run }.to print_output_and_exit_with_success(<<~STDOUT)
          checked 0 files, collected 0 calls, 0 definitions
          \e[32mEverything is used\e[0m
        STDOUT
      end

      it "doesn't generate a TODO file" do
        expect { run('--write-todo') }.to print_output_and_exit_with_success(<<~STDOUT)
          checked 0 files, collected 0 calls, 0 definitions
          No .leftovers_todo.yml file generated, everything is used
        STDOUT
      end

      it 'outputs the version when --version' do
        expect { run '--version' }.to print_output_and_exit_with_success(<<~STDOUT)
          #{::Leftovers::VERSION}
        STDOUT
      end

      it 'outputs no files when --dry-run' do
        expect { run '--dry-run' }.to print_output_and_exit_with_success('')
      end

      it 'outputs the help when --help' do
        expect { run '--help' }.to print_output_and_exit_with_success <<~STDOUT
          Usage: leftovers [options]
                  --[no-]parallel              Run in parallel or not, default --parallel
                  --[no-]progress              Show live counts or not, default --progress
                  --dry-run                    Print a list of files that would be looked at
                  --view-compiled              Print the compiled content of the files
                  --write-todo                 Create a config file with the existing unused items
              -v, --version                    Print the current version
              -h, --help                       Print this message
        STDOUT
      end

      it 'outputs the an error message and help when --nonsense' do
        expect { run '--nonsense' }.to print_error_and_exit <<~STDERR
          \e[31mCLI Error: invalid option: --nonsense\e[0m

          Usage: leftovers [options]
                  --[no-]parallel              Run in parallel or not, default --parallel
                  --[no-]progress              Show live counts or not, default --progress
                  --dry-run                    Print a list of files that would be looked at
                  --view-compiled              Print the compiled content of the files
                  --write-todo                 Create a config file with the existing unused items
              -v, --version                    Print the current version
              -h, --help                       Print this message
        STDERR
      end
    end

    context 'with a test method defined and unused' do
      before do
        temp_file 'test/bar.rb', <<~RUBY
          def test_method; end
        RUBY
      end

      it 'runs with --write-todo' do
        Timecop.freeze('2021-06-14T22:03:35 UTC')

        expect { run('--no-parallel --write-todo') }.to print_output_and_exit_with_success <<~STDOUT
          checked 1 files, collected 0 calls, 1 definitions
          generated .leftovers_todo.yml.
          running leftovers again will read this file and not alert you to any unused items mentioned in it.

          commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
        STDOUT

        expect(temp_dir.join('.leftovers_todo.yml').read).to eq(<<~FILE)
          # This file was generated by `leftovers --write-todo`
          # Generated at: 2021-06-14 22:03:35 UTC
          #
          # for instructions on how to address these
          # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

          keep:
            # Not directly called at all:
            - "test_method" # test/bar.rb:1:5 def test_method; end
        FILE

        ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml')).keep
      end
    end

    context 'with an erb file' do
      before do
        temp_file 'test/foo.erb', <<~HAML
          <%= bar %>
        HAML
      end

      it 'runs with --view-compiled' do
        expect { run('--view-compiled *.erb') }.to print_output_and_exit_with_success <<~STDOUT
          \e[0;2mtest/foo.erb\e[0m
          #coding:UTF-8

           bar
          ;
        STDOUT
      end
    end

    context 'with files with linked config' do
      before do
        temp_file '.leftovers.yml', <<~YML
          dynamic:
            - name: test_method
              defines:
                argument: 0
                transforms:
                  - original
                  - add_suffix: '?'
        YML

        temp_file 'app/foo.rb', <<~RUBY
          test_method :test
        RUBY
      end

      it 'runs' do
        expect { run }.to print_output_and_exit_with_failure(<<~STDOUT)
          checked 2 files, collected 2 calls, 1 definitions
          \e[31mNot directly called at all:\e[0m
          \e[36mapp/foo.rb:1:13\e[0m test, test? \e[2mtest_method \e[33m:test\e[0;2m\e[0m

          how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
        STDOUT
      end

      it 'runs with --write-todo' do
        Timecop.freeze('2021-06-14T22:03:35 UTC')

        expect { run('--no-parallel --write-todo') }
          .to print_output_and_exit_with_success(<<~STDOUT)
            checked 2 files, collected 2 calls, 1 definitions
            generated .leftovers_todo.yml.
            running leftovers again will read this file and not alert you to any unused items mentioned in it.

            commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
          STDOUT

        expect(temp_dir.join('.leftovers_todo.yml').read).to eq(<<~FILE)
          # This file was generated by `leftovers --write-todo`
          # Generated at: 2021-06-14 22:03:35 UTC
          #
          # for instructions on how to address these
          # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

          keep:
            # Not directly called at all:
            - "test" # app/foo.rb:1:13 test_method :test
            - "test?" # app/foo.rb:1:13 test_method :test
        FILE

        ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml')).keep
      end
    end

    context 'with files with unused methods' do
      before do
        temp_file('app/foo.rb', <<~RUBY)
          attr_reader :foo

          def unused_method
            @bar = true
          end
        RUBY
      end

      it 'runs' do
        expect(Parallel).to receive(:each).once.and_call_original # parallel by default

        expect { run }.to print_output_and_exit_with_failure <<~STDOUT
          checked 1 files, collected 2 calls, 3 definitions
          \e[31mNot directly called at all:\e[0m
          \e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
          \e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
          \e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m

          how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
        STDOUT
      end

      it 'runs with an error while collecting' do
        allow(::Leftovers::FileCollector::NodeProcessor).to receive(:new).and_raise(ArgumentError,
                                                                                    "don't")
        expect { run }.to print_error_and_exit(
          %r{
            \e\[31mLeftovers::FileCollector::Error:\ ArgumentError:\ don't\n
            \ \ when\ processing\ app/foo.rb\e\[0m\n\n
          }x
        )
      end

      it 'runs with --write-todo' do
        Timecop.freeze('2021-06-14T22:03:35 UTC')
        expect { run('--write-todo') }.to print_output_and_exit_with_success <<~STDOUT
          checked 1 files, collected 2 calls, 3 definitions
          generated .leftovers_todo.yml.
          running leftovers again will read this file and not alert you to any unused items mentioned in it.

          commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
        STDOUT

        expect(temp_dir.join('.leftovers_todo.yml').read).to eq(<<~FILE)
          # This file was generated by `leftovers --write-todo`
          # Generated at: 2021-06-14 22:03:35 UTC
          #
          # for instructions on how to address these
          # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

          keep:
            # Not directly called at all:
            - "@bar" # app/foo.rb:4:3 @bar = true
            - "foo" # app/foo.rb:1:13 attr_reader :foo
            - "unused_method" # app/foo.rb:3:5 def unused_method
        FILE

        ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml')).keep
      end

      it 'runs with --write-todo and a preexisting TODO file' do
        Timecop.freeze('2021-06-14T22:03:35 UTC')

        todo_file = temp_file('.leftovers_todo.yml')
        expect { run('--write-todo') }.to print_output_and_exit_with_success <<~STDOUT
          Removing previous .leftovers_todo.yml file

          checked 1 files, collected 2 calls, 3 definitions
          generated .leftovers_todo.yml.
          running leftovers again will read this file and not alert you to any unused items mentioned in it.

          commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
        STDOUT

        expect(todo_file.read).to eq(<<~FILE)
          # This file was generated by `leftovers --write-todo`
          # Generated at: 2021-06-14 22:03:35 UTC
          #
          # for instructions on how to address these
          # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

          keep:
            # Not directly called at all:
            - "@bar" # app/foo.rb:4:3 @bar = true
            - "foo" # app/foo.rb:1:13 attr_reader :foo
            - "unused_method" # app/foo.rb:3:5 def unused_method
        FILE

        ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml')).keep
      end

      it 'runs with --no-parallel' do
        expect(Parallel).to receive(:each).exactly(0).times

        expect { run('--no-parallel') }.to print_output_and_exit_with_failure <<~STDOUT
          checked 1 files, collected 2 calls, 3 definitions
          \e[31mNot directly called at all:\e[0m
          \e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
          \e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
          \e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m

          how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
        STDOUT
      end

      it 'runs with --parallel' do
        expect(Parallel).to receive(:each).once.and_call_original

        expect { run('--parallel') }.to print_output_and_exit_with_failure <<~STDOUT
          checked 1 files, collected 2 calls, 3 definitions
          \e[31mNot directly called at all:\e[0m
          \e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
          \e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
          \e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m

          how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
        STDOUT
      end

      it 'outputs the version when --version' do
        expect { run '--version' }.to print_output_and_exit_with_success(<<~STDOUT)
          #{::Leftovers::VERSION}
        STDOUT
      end

      it 'outputs the files when --dry-run' do
        expect { run '--dry-run' }.to print_output_and_exit_with_success <<~STDOUT
          app/foo.rb
        STDOUT
      end

      context 'with tests' do
        before do
          temp_file 'test/foo.rb', <<~RUBY
            expect(unused_method).to eq foo
            self.instance_variable_get(:@bar) == true
          RUBY
        end

        it 'runs' do
          expects_output!

          exitstatus = run

          expect(::Leftovers.stdout.string).to eq(
            <<~STDOUT
              \e[2Kchecked 1 files, collected 2 calls, 3 definitions\r\e[2Kchecked 2 files, collected 10 calls, 3 definitions\r\e[2Kchecked 2 files, collected 10 calls, 3 definitions
              \e[2K\e[31mOnly directly called in tests:\e[0m
              \e[2K\e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
              \e[2K\e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
              \e[2K\e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m
              \e[2K
              how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
            STDOUT
          ).or(
            eq(
              <<~STDOUT
                \e[2Kchecked 1 files, collected 8 calls, 0 definitions\r\e[2Kchecked 2 files, collected 10 calls, 3 definitions\r\e[2Kchecked 2 files, collected 10 calls, 3 definitions
                \e[2K\e[31mOnly directly called in tests:\e[0m
                \e[2K\e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
                \e[2K\e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
                \e[2K\e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m
                \e[2K
                how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
              STDOUT
            )
          )
          expect(::Leftovers.stderr.string).to be_empty
          expect(exitstatus).to be 1
        end

        it 'runs with suppressed progress' do
          expects_output!

          exitstatus = run('--no-progress')

          expect(::Leftovers.stdout.string).to eq <<~STDOUT
            \e[2Kchecked 2 files, collected 10 calls, 3 definitions
            \e[2K\e[31mOnly directly called in tests:\e[0m
            \e[2K\e[36mapp/foo.rb:1:13\e[0m foo \e[2mattr_reader \e[33m:foo\e[0;2m\e[0m
            \e[2K\e[36mapp/foo.rb:3:5\e[0m unused_method \e[2mdef \e[33munused_method\e[0;2m\e[0m
            \e[2K\e[36mapp/foo.rb:4:3\e[0m @bar \e[2m\e[33m@bar\e[0;2m = true\e[0m
            \e[2K
            how to resolve: \e[32m#{::Leftovers.resolution_instructions_link}\e[0m
          STDOUT
          expect(::Leftovers.stderr.string).to be_empty
          expect(exitstatus).to be 1
        end

        it 'runs with --write-todo' do
          Timecop.freeze('2021-06-14T22:03:35 UTC')

          temp_file 'test/bar.rb', <<~RUBY
            def test_method; end
          RUBY

          expect { run('--write-todo') }.to print_output_and_exit_with_success <<~STDOUT
            checked 3 files, collected 10 calls, 4 definitions
            generated .leftovers_todo.yml.
            running leftovers again will read this file and not alert you to any unused items mentioned in it.

            commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
          STDOUT

          expect(temp_dir.join('.leftovers_todo.yml').read).to eq(<<~FILE)
            # This file was generated by `leftovers --write-todo`
            # Generated at: 2021-06-14 22:03:35 UTC
            #
            # for instructions on how to address these
            # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

            test_only:
              # Only directly called in tests:
              - "@bar" # app/foo.rb:4:3 @bar = true
              - "foo" # app/foo.rb:1:13 attr_reader :foo
              - "unused_method" # app/foo.rb:3:5 def unused_method

            keep:
              # Not directly called at all:
              - "test_method" # test/bar.rb:1:5 def test_method; end
          FILE

          config = ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml'))
          config.keep
          config.test_only
        end
      end

      context 'with some test' do
        before do
          temp_file 'test/foo.rb', <<~RUBY
            expect(unused_method).to eq foo
          RUBY
        end

        it 'runs with --write-todo' do
          Timecop.freeze('2021-06-14T22:03:35 UTC')
          temp_file 'test/bar.rb', <<~RUBY
            def test_method; end
          RUBY

          expect { run('--write-todo') }.to print_output_and_exit_with_success <<~STDOUT
            checked 3 files, collected 7 calls, 4 definitions
            generated .leftovers_todo.yml.
            running leftovers again will read this file and not alert you to any unused items mentioned in it.

            commit this file so you/your team can gradually address these items while still having leftovers alert you to any newly unused items.
          STDOUT

          expect(temp_dir.join('.leftovers_todo.yml').read).to eq(<<~FILE)
            # This file was generated by `leftovers --write-todo`
            # Generated at: 2021-06-14 22:03:35 UTC
            #
            # for instructions on how to address these
            # see https://github.com/robotdana/leftovers/tree/v#{::Leftovers::VERSION}/README.md#how-to-resolve

            test_only:
              # Only directly called in tests:
              - "foo" # app/foo.rb:1:13 attr_reader :foo
              - "unused_method" # app/foo.rb:3:5 def unused_method

            keep:
              # Not directly called at all:
              - "@bar" # app/foo.rb:4:3 @bar = true
              - "test_method" # test/bar.rb:1:5 def test_method; end
          FILE

          config = ::Leftovers::Config.new(:todo, path: temp_dir.join('.leftovers_todo.yml'))
          config.keep
          config.test_only
        end
      end
    end
  end
end
