import os
from typing import Callable, Dict, Optional, Tuple

import tensorflow as tf

from .reader import DataReader


class LJSpeech(DataReader):
    """LJ Speech dataset loader.
    Use other opensource vocoder settings, 16bit, sr: 22050.
    """
    SR = 22050
    MAXVAL = 32767.

    def __init__(self,
                 data_dir: Optional[str] = None,
                 download: bool = False,
                 from_tfds: bool = True):
        """Initializer.
        Args:
            data_dir: dataset directory
                , defaults to '~/tensorflow_datasets'.
            download: download dataset or not, only enabled on `from_tfds`.
            from_tfds: load from tfrecord generated by tfds or read raw audio.
        """
        self.rawset, self._preproc = self.load_data(data_dir, download, from_tfds)

    def dataset(self) -> tf.data.Dataset:
        """Return file reader.
        Returns:
            file-format datum reader.
        """
        return self.rawset
    
    def preproc(self) -> Callable:
        """Return data preprocessor.
        Returns:
            preprocessor, expected format 
                text: tf.string, text.
                speech: [tf.float32; T], speech signal in range (-1, 1).
        """
        return self._preproc

    def load_data(self,
                  data_dir: Optional[str] = None,
                  download: bool = False,
                  from_tfds: bool = True) -> Tuple[tf.data.Dataset, Callable]:
        """Load dataset from tfrecord or raw audio files.
        Args:
            data_dir: dataset directory.
                For from_tfds, None is acceptable
                 and set to default value '~/tensorflow_datasets'.
                For from raw audio, None is not acceptable.
            download: download dataset or not, for from_tfds.
            from_tfds: whether use tfds or read raw audio.
        Returns:
            data loader and preprocessor, expected form
                text: tf.string, text.
                speech: [tf.float32; T], speech signal in range (-1, 1).
        """
        if from_tfds:
            import tensorflow_datasets as tfds
            dataset = tfds.load(
                'ljspeech', split='train',
                data_dir=data_dir, download=download)
            return dataset, self._preproc_tfds
        # load audio
        return self.load_audio(data_dir)

    def load_audio(self, data_dir: str) -> Tuple[tf.data.Dataset, Callable]:
        """Load audio with tf apis.
        Args:
            data_dir: dataset directory.
        Returns:
            data loader.
                text: tf.string, text.
                speech: [tf.float32; T], speech signal in range (-1, 1).
        """
        # generate file lists
        files = tf.data.Dataset.list_files(
            os.path.join(data_dir, 'wavs', '*.wav'))
        # read filename-text pair
        with open(os.path.join(data_dir, 'metadata.csv'), encoding='utf-8') as f:
            keys, values = [], []
            for row in f.readlines():
                name, _, normalized = row.replace('\n', '').split('|')
                keys.append(name)
                values.append(normalized)
        # generate lookup table
        table = tf.lookup.StaticHashTable(
            tf.lookup.KeyValueTensorInitializer(keys, values), "")
        # read audio
        return files, self._preproc_audio(table)

    def _preproc_tfds(self, datum: Dict[str, tf.Tensor]) \
            -> Tuple[tf.Tensor, tf.Tensor]:
        """Preprocess datum from tfds.
        Args:
            datum: single datum.
                id: tf.string, string id.
                speech: [tf.int64; T], audio signal in range (-MAXVALUE - 1, MAXVALUE).
                text: tf.string, text.
                text_normalized: tf.string, normalized text.
        Returns:
            tuple,
                text: tf.Tensor, tf.string, text.
                speech: tf.Tensor, [tf.float32; T], mono audio in range (-1, 1).
        """
        # tf.string
        text = datum['text_normalized']
        # [tf.float32; S]
        speech = tf.cast(datum['speech'], tf.float32) / LJSpeech.MAXVAL
        return text, speech

    def _preproc_audio(self, table: tf.lookup.StaticHashTable) -> Callable:
        """Generate audio loader.
        Args:
            table: lookup table for filename to text map.
        Returns:
            path to (text, speech) mapper.
        """
        def load_and_lookup(path: tf.string) -> Tuple[tf.Tensor, tf.Tensor]:
            """Load audio and lookup text.
            Args:
                path: tf.string, path
            Returns:
                tuple,
                    text: tf.string, text.
                    audio: [tf.float32; T], raw speech signal in range(-1, 1).
            """
            raw = tf.io.read_file(path)
            # [T, C(=1, mono)]
            audio, _ = tf.audio.decode_wav(raw, desired_channels=1)
            # [T]
            audio = tf.squeeze(audio, axis=-1)
            # tf.string
            path = tf.strings.split(path, os.sep)
            path = tf.strings.regex_replace(path[-1], '\.wav', '')
            # tf.string, [tf.float32; T]
            return table.lookup(path), audio

        return load_and_lookup
