package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/ast"
	"go/importer"
	"go/parser"
	"go/token"
	"go/types"
	"log"
	"os"
	"strings"
	"text/template"

	"github.com/quasilyte/go-ruleguard/ruleguard/irconv"
	"github.com/quasilyte/go-ruleguard/ruleguard/irprint"
)

// This program generates a loadable IR for ruleguard
// so we don't have to load the rules from AST and typecheck
// them every time.

func main() {
	log.SetFlags(0)
	if err := precompile(); err != nil {
		log.Printf("error: %v", err)
	}
}

func precompile() error {
	flagRules := flag.String("rules", "", "path to a ruleguard rules file")
	flagOutput := flag.String("o", "", "output file name")
	flagVarName := flag.String("varname", "PrecompiledRules", "generated variable name")
	flag.Parse()

	fset := token.NewFileSet()
	filename := strings.TrimSpace(*flagRules)
	fileData, err := os.ReadFile(filename)
	if err != nil {
		return fmt.Errorf("read %s: %v", filename, err)
	}
	r := bytes.NewReader(fileData)
	parserFlags := parser.ParseComments
	f, err := parser.ParseFile(fset, filename, r, parserFlags)
	if err != nil {
		return fmt.Errorf("parse %s: %v", filename, err)
	}
	imp := importer.For("source", nil)
	typechecker := types.Config{Importer: imp}
	types := &types.Info{
		Types: map[ast.Expr]types.TypeAndValue{},
		Uses:  map[*ast.Ident]types.Object{},
		Defs:  map[*ast.Ident]types.Object{},
	}
	pkg, err := typechecker.Check("gorules", fset, []*ast.File{f}, types)
	if err != nil {
		return fmt.Errorf("typecheck %s: %v", filename, err)
	}
	irconvCtx := &irconv.Context{
		Pkg:   pkg,
		Types: types,
		Fset:  fset,
		Src:   fileData,
	}
	irfile, err := irconv.ConvertFile(irconvCtx, f)
	if err != nil {
		return fmt.Errorf("compile %s: %v", filename, err)
	}

	// TODO: build rules info maps, so we don't need to scan tags during the run-time?
	for i := range irfile.RuleGroups {
		g := &irfile.RuleGroups[i]
		tagO := false
		tagScore := false
		for _, tag := range g.DocTags {
			switch tag {
			case "o1", "o2":
				tagO = true
			case "score1", "score2", "score3", "score4", "score5":
				tagScore = true
			case "reformat":
				// OK.
			default:
				return fmt.Errorf("%s: unknown tag: %s", g.Name, tag)
			}
		}
		if !tagO {
			return fmt.Errorf("%s: add o1 or o2 tag")
		}
		if !tagScore {
			return fmt.Errorf("%s: add score[1-5] tag", g.Name)
		}

		for j := range g.Rules {
			rule := &g.Rules[j]
			if strings.HasPrefix(rule.ReportTemplate, "suggestion: ") {
				leftVar := "$$"
				if rule.LocationVar != "" {
					leftVar = "$" + rule.LocationVar
				}
				rule.ReportTemplate = leftVar + " => " + strings.TrimPrefix(rule.ReportTemplate, "suggestion: ")
			}
		}
	}

	var rulesText bytes.Buffer
	irprint.File(&rulesText, irfile)

	fileTemplate := template.Must(template.New("gorules").Parse(`// Code generated by "precompile.go". DO NOT EDIT.

package rulesdata

import "github.com/quasilyte/go-ruleguard/ruleguard/ir"

var {{$.VarName}} = &{{$.RulesText}}
`))

	var generated bytes.Buffer
	err = fileTemplate.Execute(&generated, map[string]interface{}{
		"RulesText": rulesText.String(),
		"VarName":   *flagVarName,
	})
	if err != nil {
		return err
	}

	if err := os.WriteFile(*flagOutput, generated.Bytes(), 0644); err != nil {
		return err
	}

	return nil
}
